# app/services/auth_service.py
import logging
from ..models import User, Organization
from ..config import Config
from datetime import datetime, timedelta
from .. import db
from werkzeug.security import generate_password_hash
from .email_service import generate_confirmation_token, send_confirmation_email, send_password_reset_email

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

class AuthService:
    @staticmethod
    def create_user(user_data):
        """Create a new user"""
        try:
            # Check if email already exists
            if User.query.filter_by(email=user_data['email'].lower()).first():
                logger.warning(f"User creation failed: email {user_data['email']} already exists")
                return None, "Email already exists"

            # Validate password
            from ..utils.validators import validate_password
            is_valid, error = validate_password(user_data['password'])
            if not is_valid:
                logger.warning(f"Password validation failed for {user_data['email']}: {error}")
                return None, error

            user = User(
                first_name=user_data['first_name'],
                last_name=user_data['last_name'],
                email=user_data['email'].lower(),  # Normalize email
                phone=user_data.get('phone'),
                email_confirmed=False  # Initially not confirmed
            )
            user.password = user_data['password']
            
            db.session.add(user)
            db.session.commit()

            logger.info(f"User created successfully: {user.email}")
            
            # Generate a token and send the confirmation email
            token = generate_confirmation_token()
            send_confirmation_email(user, token)
            logger.info(f"Confirmation email sent to: {user.email}")
            
            return user, None
            
        except Exception as e:
            db.session.rollback()
            logger.exception(f"Exception during user creation: {str(e)}")
            return None, str(e)
    
    @staticmethod
    def confirm_email(token):
        """Confirm user's email with token"""
        try:
            print(f"Attempting to confirm email with token: {token}")
            logger.info(f"Confirming email with token: {token}")
            user = User.query.filter_by(email_confirmation_token=token).first()
            if not user:
                print(f"User not found for token: {token}")
                logger.warning("Email confirmation failed: invalid or expired token")
                return False, "Invalid or expired token"
            
            print(f"User found: {user.email}")
            logger.info(f"User found for email confirmation: {user.email}")
            # Confirm the email
            user.email_confirmed = True
            user.email_confirmation_token = None  # Reset token
            
            try:
                db.session.commit()
                print("Email confirmed successfully")
                logger.info(f"Email confirmed for user: {user.email}")
            except Exception as e:
                print(f"Error in committing email confirmation: {str(e)}")
                db.session.rollback()
                logger.exception("Database error during email confirmation")
                return False, f"Database error: {str(e)}"

            org = Organization.query.filter_by(representative_id=user.id).first()
            if not org:
                print(f"Organization not found for user: {user.id}")
                logger.warning(f"Organization not found for user ID: {user.id}")
                return False, "Organization not found"
            
            print(f"Organization found: {org.legal_name}")
            logger.info(f"Organization found for user {user.email}: {org.legal_name}")
            org.status = 'pending'
            org.voting_ends_at = datetime.utcnow() + timedelta(minutes=Config.VOTING_DURATION_MINUTES)
            
            try:
                db.session.commit()
                print("Organization status updated successfully")
                logger.info(f"Organization status set to 'pending' for: {org.legal_name}")
                
                # Now that the organization is pending and email is confirmed,
                # notify all representatives of approved organizations
                from ..services.email_service import send_new_application_notification
                
                # Get all representatives of approved organizations (excluding the current one)
                approved_orgs = Organization.query.filter_by(status='approved').all()
                representatives = [org.representative for org in approved_orgs if org.representative]
                
                if representatives:
                    logger.info(f"Sending notifications to {len(representatives)} representatives about new application")
                    success = send_new_application_notification(org, representatives)
                    if success:
                        logger.info("Notifications sent successfully")
                    else:
                        logger.warning("Some notifications may not have been sent")
                else:
                    logger.info("No representatives to notify")
                    
            except Exception as e:
                print(f"Error updating organization status: {str(e)}")
                db.session.rollback()
                logger.exception("Database error updating organization status")
                return False, f"Database error: {str(e)}"
            
            return True, None
        except Exception as e:
            print(f"General exception in email confirmation: {str(e)}")
            db.session.rollback()
            logger.exception("Unhandled exception during email confirmation")
            return False, str(e)
    
    @staticmethod
    def authenticate_user(email, password):
        """Authenticate user with email and password"""
        email = email.lower()
        user = User.query.filter_by(email=email).first()
        if not user:
            logger.warning(f"Authentication failed: user with email {email} not found")
            return None, "Invalid email or password"
            
        if not user.verify_password(password):
            logger.warning(f"Authentication failed: invalid password for {email}")
            return None, "Invalid email or password"
        
        # Check if the email has been confirmed
        if not user.email_confirmed:
            logger.warning(f"Authentication blocked: email not confirmed for {email}")
            return None, "Please confirm your email before logging in"
        
        logger.info(f"User authenticated successfully: {email}")
        return user, None  # Authentication successful
    
    @staticmethod
    def resend_confirmation_email(email):
        """Resend confirmation email to user"""
        user = User.query.filter_by(email=email).first()
        if not user:
            logger.warning(f"Resend confirmation failed: user not found for {email}")
            return False, "User not found"
            
        if user.email_confirmed:
            logger.info(f"Confirmation email not resent: email already confirmed for {email}")
            return False, "Email already confirmed"
            
        # Generate a new token and send the email
        token = generate_confirmation_token()
        send_confirmation_email(user, token)
        logger.info(f"Confirmation email resent to: {email}")
        
        return True, None

    @staticmethod
    def get_user_by_id(user_id):
        """Get user by their ID"""
        logger.info(f"Fetching user by ID: {user_id}")
        return User.query.get(user_id)
    
    @staticmethod
    def request_password_reset(email):
        """Request password reset for a user"""
        user = User.query.filter_by(email=email.lower()).first()
        if not user:
            logger.warning(f"Password reset request failed: user not found for {email}")
            return False, "User not found"
        
        # Generate a token and send the password reset email
        token = generate_confirmation_token()
        success = send_password_reset_email(user, token)
        
        if success:
            # Save the token and timestamp in the database
            user.password_reset_token = token
            user.password_reset_sent_at = datetime.utcnow()
            db.session.commit()
            logger.info(f"Password reset email sent to: {email}")
            return True, None
        else:
            logger.error(f"Failed to send password reset email to: {email}")
            return False, "Failed to send password reset email"

    @staticmethod
    def reset_password(token, new_password):
        """Reset user's password with token"""
        try:
            user = User.query.filter_by(password_reset_token=token).first()
            if not user:
                logger.warning("Password reset failed: invalid or expired token")
                return False, "Invalid or expired token"
            
            # Check token expiration (24 hours)
            token_age = datetime.utcnow() - user.password_reset_sent_at
            if token_age.total_seconds() > 86400:  # 24 hours
                logger.warning(f"Password reset token expired for user: {user.email}")
                return False, "Token has expired"
            
            # Validate password
            from ..utils.validators import validate_password
            is_valid, error = validate_password(new_password)
            if not is_valid:
                logger.warning(f"Password reset validation failed for {user.email}: {error}")
                return False, error
            
            # Reset the password
            user.password = new_password
            user.password_reset_token = None
            db.session.commit()
            logger.info(f"Password reset successfully for user: {user.email}")
            
            return True, None
        except Exception as e:
            db.session.rollback()
            logger.exception("Exception during password reset")
            return False, str(e)
