# app/services/digital_euro_service.py
"""
Digital Euro Service - Integration with BCE DESP API
Handles holdings (wallets), payments, and reservations
"""

import requests
import logging
from datetime import datetime, timedelta, timezone
import uuid
from schwifty import IBAN
import random

logger = logging.getLogger(__name__)

# BCE DESP Configuration (Production - currently not accessible)
DESP_BASE_URL = 'https://desp.dev/api/rest/v1'
DESP_USER = 'your-desp-username'
DESP_PASSWORD = 'your-desp-password'


class DigitalEuroService:
    """Service for Digital Euro operations via BCE DESP API"""
    
    def __init__(self):
        self.base_url = DESP_BASE_URL
        self.auth = (DESP_USER, DESP_PASSWORD)
    
    def _call_api(self, endpoint, method='GET', data=None):
        """
        Make API call to BCE DESP
        MOCK VERSION - Real API not accessible in local development
        """
        # MOCK API CALL FOR LOCAL TESTING - Comment out for production
        # url = f"{self.base_url}{endpoint}"
        # try:
        #     if method == 'GET':
        #         response = requests.get(url, auth=self.auth, timeout=30)
        #     elif method == 'POST':
        #         response = requests.post(url, json=data, auth=self.auth, timeout=30)
        #     elif method == 'PUT':
        #         response = requests.put(url, json=data, auth=self.auth, timeout=30)
        #     elif method == 'DELETE':
        #         response = requests.delete(url, auth=self.auth, timeout=30)
        #     
        #     response.raise_for_status()
        #     return response.json()
        # except Exception as e:
        #     logger.error(f"BCE API error: {str(e)}")
        #     return {'status': 'error', 'error': str(e)}
        
        # MOCK RESPONSES
        return self._mock_api_call(endpoint, method, data)
    
    def _mock_api_call(self, endpoint, method, data):
        """Mock BCE API responses for local testing"""
        logger.info(f"[MOCK BCE] {method} {endpoint}")
        logger.info(f"[MOCK BCE] Payload: {data}")
        
        # Mock: Create holding
        if method == 'POST' and '/holdings' in endpoint:
            holding_id = str(uuid.uuid4())
            return {
                'entry': holding_id,
                'holdingType': data.get('holdingType', 'endUser'),
                'balance': {
                    'amount': {'amount': '10000.00', 'currency': 'EUR'},
                    'availableAmount': {'amount': '10000.00', 'currency': 'EUR'}
                }
            }
        
        # Mock: Get holding (balance)
        if method == 'GET' and '/holdings/' in endpoint:
            return {
                'entry': endpoint.split('/')[-1],
                'balance': {
                    'amount': {'amount': '10000.00', 'currency': 'EUR'},
                    'availableAmount': {'amount': '10000.00', 'currency': 'EUR'},
                    'maximumAmount': {'amount': '3000.00', 'currency': 'EUR'}
                },
                'holdingType': 'merchant'
            }
        
        # Mock: Create reservation
        if method == 'POST' and '/reservations/' in endpoint:
            reservation_id = endpoint.split('/')[-1]
            return {
                'entry': reservation_id,
                'debtorEntry': data.get('debtorEntry'),
                'creditorEntry': data.get('creditorEntry'),
                'amount': data.get('amount'),
                'expiryDate': data.get('expiryDate'),
                'status': 'active'
            }
        
        # Mock: Cancel reservation
        if method == 'DELETE' and '/reservations/' in endpoint:
            reservation_id = endpoint.split('/')[-1]
            logger.info(f"[MOCK BCE] Cancelled reservation: {reservation_id}")
            return {
                'entry': reservation_id,
                'status': 'cancelled'
            }
        
        # Mock: Payment from reservation
        if method == 'POST' and '/payments/' in endpoint:
            payment_id = endpoint.split('/')[-1]
            return {
                'entry': payment_id,
                'reservationId': data.get('reservationId'),
                'amount': data.get('amount'),
                'status': 'completed',
                'timestamp': datetime.now(timezone.utc).isoformat()
            }
        
        # Mock: Direct P2P payment
        if method == 'POST' and '/payments' in endpoint:
            payment_id = str(uuid.uuid4())
            return {
                'entry': payment_id,
                'debtorEntry': data.get('debtorEntry'),
                'creditorEntry': data.get('creditorEntry'),
                'amount': data.get('amount'),
                'status': 'completed',
                'timestamp': datetime.now(timezone.utc).isoformat()
            }
        
        # Mock: Get payment
        if method == 'GET' and '/payments/' in endpoint:
            payment_id = endpoint.split('/')[-1]
            return {
                'entry': payment_id,
                'status': 'completed',
                'amount': {'amount': '100.00', 'currency': 'EUR'}
            }
        
        # Default
        return {'status': 'error', 'error': 'Unknown endpoint'}
    
    def create_holding(self, holding_type='endUser'):
        """
        Create a new Digital Euro holding (wallet)
        Types: 'endUser' (max 3000 EUR) or 'merchant' (unlimited)
        """
        if holding_type not in ['endUser', 'merchant']:
            return {'error': 'Invalid holding type'}
        
        # Generate unique holding ID
        holding_id = str(uuid.uuid4())
        
        payload = {
            'entry': holding_id,
            'holdingType': holding_type
        }
        
        response = self._call_api(f'/holdings', method='POST', data=payload)
        
        if response.get('status') == 'error':
            logger.error(f"Holding creation failed: {response.get('error')}")
            return {'error': response.get('error')}
        
        logger.info(f"Created holding: {response.get('entry')} (type: {holding_type})")
        return response
    
    def get_balance(self, holding_id):
        """
        Get balance for a holding
        Returns: (amount, available_amount, limit)
        """
        response = self._call_api(f'/holdings/{holding_id}', method='GET')
        
        if response.get('status') == 'error':
            logger.error(f"Balance fetch failed: {response.get('error')}")
            return 0.0, 0.0, 0.0
        
        balance = response.get('balance', {})
        amount = float(balance.get('amount', {}).get('amount', 0))
        available = float(balance.get('availableAmount', {}).get('amount', 0))
        limit = float(balance.get('maximumAmount', {}).get('amount', 0))
        
        return amount, available, limit
    
    def create_reservation(self, debtor_id, creditor_id, amount, expiry_minutes=30):
        """
        Create a reservation (block funds without transferring)
        This is the KEY method for safe transactions!
        """
        reservation_id = str(uuid.uuid4())
        expiry_date = datetime.now(timezone.utc) + timedelta(minutes=expiry_minutes)
        
        payload = {
            'expiryDate': expiry_date.isoformat(),
            'creditorEntry': str(creditor_id),
            'amount': {
                'amount': str(amount),
                'currency': 'EUR'
            },
            'debtorEntry': str(debtor_id)
        }
        
        logger.info(f"Creating reservation {reservation_id} for {amount} EUR")
        response = self._call_api(f'/reservations/{reservation_id}', method='POST', data=payload)
        
        if response.get('status') == 'error':
            logger.error(f"Reservation creation failed: {response.get('error')}")
            return {'error': response.get('error')}
        
        logger.info(f"Reservation created: {reservation_id}")
        return response
    
    def cancel_reservation(self, reservation_id):
        """
        Cancel a reservation (release blocked funds)
        CRITICAL for rollback when marketplace operations fail
        """
        logger.info(f"Cancelling reservation {reservation_id}")
        response = self._call_api(f'/reservations/{reservation_id}', method='DELETE')
        
        if response.get('status') == 'error':
            logger.error(f"Reservation cancellation failed: {response.get('error')}")
            return {'error': response.get('error')}
        
        logger.info(f"Reservation cancelled successfully: {reservation_id}")
        return response
    
    def payment_from_reservation(self, reservation_id, amount):
        """
        Complete payment from an existing reservation
        Called ONLY after marketplace operation succeeds
        """
        payment_id = str(uuid.uuid4())
        
        payload = {
            'amount': {
                'amount': str(amount),
                'currency': 'EUR'
            },
            'reservationId': str(reservation_id),
            'reserveRemaining': False  # Consume entire reservation
        }
        
        logger.info(f"Completing payment {payment_id} from reservation {reservation_id}")
        response = self._call_api(f'/payments/{payment_id}', method='POST', data=payload)
        
        if response.get('status') == 'error':
            logger.error(f"Payment from reservation failed: {response.get('error')}")
            return {'error': response.get('error')}
        
        logger.info(f"Payment completed: {payment_id}")
        return response
    
    def p2p_payment(self, debtor_id, creditor_id, amount):
        """
        Direct P2P payment (without reservation)
        Use this for redemptions where marketplace debit happens first
        """
        payment_id = str(uuid.uuid4())
        
        payload = {
            'debtorEntry': str(debtor_id),
            'creditorEntry': str(creditor_id),
            'amount': {
                'amount': str(amount),
                'currency': 'EUR'
            }
        }
        
        logger.info(f"Creating direct payment {payment_id} for {amount} EUR")
        response = self._call_api(f'/payments/{payment_id}', method='POST', data=payload)
        
        if response.get('status') == 'error':
            logger.error(f"Payment failed: {response.get('error')}")
            return {'error': response.get('error')}
        
        logger.info(f"Payment successful: {payment_id}")
        return response
    
    def generate_euban(self):
        """
        Generate a valid EUBAN (EU IBAN format for Digital Euro)
        Format: EU + 2 check digits + 22 digits
        """
        # Generate random 22-digit account number
        account_number = ''.join([str(random.randint(0, 9)) for _ in range(22)])
        
        # EUBAN format: EU + check digits + account
        # For simplicity, use country code EU82 (mock check digits)
        euban_raw = f"EU82{account_number}"
        
        try:
            # Validate and generate proper check digits
            iban = IBAN.generate('EU', '82', account_number)
            return str(iban)
        except Exception:
            # Fallback: simple format
            return euban_raw


# Singleton instance
digital_euro_service = DigitalEuroService()
