# app/services/document_service.py
import os
import uuid
import logging
from werkzeug.utils import secure_filename
from ..models import Document
from .. import db
from flask import current_app

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

class DocumentService:
    @staticmethod
    def save_document(file, organization_id):
        """Save a document file and create a record in the database"""
        try:
            logger.info(f"Attempting to save document for organization_id={organization_id}")
            
            if not file:
                logger.warning("No file provided for upload")
                return None, "No file provided"

            original_filename = secure_filename(file.filename)
            if not DocumentService._allowed_file(original_filename):
                logger.warning(f"File type not allowed: {original_filename}")
                return None, "File type not allowed. The only ones accepted are: PDF, DOC o DOCX."
                
            # Controlla dimensione del file (max 10MB)
            file_content = file.read()
            file.seek(0)  # Rewind the file after reading
            
            # 10MB = 10 * 1024 * 1024 bytes
            if len(file_content) > 10 * 1024 * 1024:
                logger.warning(f"File too large: {original_filename} ({len(file_content)} bytes)")
                return None, "The file exceeds the maximum allowed size of 10MB"

            # Create uploads directory if it doesn't exist
            if not os.path.exists(current_app.config['UPLOAD_FOLDER']):
                logger.info(f"Creating upload folder at {current_app.config['UPLOAD_FOLDER']}")
                os.makedirs(current_app.config['UPLOAD_FOLDER'])

            # Generate a unique filename with UUID
            file_extension = os.path.splitext(original_filename)[1]
            unique_filename = f"{uuid.uuid4().hex}{file_extension}"
            
            # Save file with unique name
            file_path = os.path.join(current_app.config['UPLOAD_FOLDER'], unique_filename)
            file.save(file_path)
            logger.info(f"File saved to {file_path}")

            # Create document record
            document = Document(
                organization_id=organization_id,
                filename=original_filename,  # Keep the original name in the DB for reference
                file_path=file_path          # But use the path with the unique name
            )
            
            db.session.add(document)
            db.session.commit()
            logger.info(f"Document record created in database for organization_id={organization_id}")
            
            return document, None

        except Exception as e:
            logger.error(f"Error saving document: {e}")
            if 'file_path' in locals() and os.path.exists(file_path):
                os.remove(file_path)
                logger.info(f"Removed partially saved file at {file_path}")
            db.session.rollback()
            return None, str(e)

    @staticmethod
    def _allowed_file(filename):
        """Check if file extension is allowed"""
        return '.' in filename and \
            filename.rsplit('.', 1)[1].lower() in current_app.config['ALLOWED_EXTENSIONS']

    @staticmethod
    def get_organization_documents(org_id):
        """Get all documents for an organization"""
        logger.info(f"Fetching documents for organization_id={org_id}")
        return Document.query.filter_by(organization_id=org_id).all()

    @staticmethod
    def delete_document(document_id):
        """Delete a document and its file"""
        try:
            logger.info(f"Attempting to delete document_id={document_id}")
            document = Document.query.get(document_id)
            if not document:
                logger.warning(f"Document not found: document_id={document_id}")
                return False, "Document not found"

            # Delete file
            if os.path.exists(document.file_path):
                os.remove(document.file_path)
                logger.info(f"File deleted from disk: {document.file_path}")

            # Delete database record
            db.session.delete(document)
            db.session.commit()
            
            logger.info(f"Document record deleted from database: document_id={document_id}")
            return True, None

        except Exception as e:
            logger.error(f"Error deleting document: {e}")
            db.session.rollback()
            return False, str(e)
        
    @staticmethod
    def get_document_by_id(document_id):
        """Get a document by ID"""
        logger.info(f"Fetching document by ID: document_id={document_id}")
        return Document.query.get(document_id)
