# app/services/email_service.py
import os
import secrets
from datetime import datetime
from flask import url_for, current_app

def generate_confirmation_token():
    """Generate a secure token for email confirmation"""
    return secrets.token_urlsafe(32)  # 32 bytes of entropy

def send_confirmation_email(user, token):
    """Send email confirmation link to user"""
    from app import db
    from flask import current_app, url_for
    import os
    
    # Save the token and timestamp in the database
    user.email_confirmation_token = token
    user.email_confirmation_sent_at = datetime.utcnow()
    db.session.commit()
    
    # Calculate the correct path for the logs directory
    # current_app.root_path is the 'app' directory
    # so we need to go one level up
    log_dir = os.path.join(os.path.dirname(current_app.root_path), 'logs')
    os.makedirs(log_dir, exist_ok=True)
    
    # Use configurations from the app's configuration
    smtp_server = current_app.config.get('MAIL_SERVER', 'localhost')
    port = current_app.config.get('MAIL_PORT', 1025)
    sender = current_app.config.get('EMAIL_SENDER', 'info@gftinnovation.eu')
    sent_from = current_app.config.get('EMAIL_NOREPLY', 'no-reply@gftinnovation.eu')
    
    # Build the confirmation URL with url_for
    confirmation_url = url_for('auth.confirm_email', 
                              token=token, 
                              _external=True)
    
    recipient = user.email
    
    message = f"""Subject: {current_app.config.get('EMAIL_SENDER_NAME', 'FAME FEDERATION')} - Confirm Your Email
To: {recipient}
From: {sent_from}

Hello {user.first_name},

Thank you for registering with FAME FEDERATION. To complete your registration, please confirm your email address by clicking on the link below:

{confirmation_url}

If you did not sign up for FAME FEDERATION, please ignore this email.

Best regards,
--FAME FEDERATION--
(Please do NOT REPLY to this email)
"""
    
    # For development environments, we can opt for logging instead of actual sending
    if current_app.config.get('MAIL_SUPPRESS_SEND', False):
        # Log the email instead of actually sending it
        import os
        import logging
        
        # Configure logger
        log_dir = os.path.join(current_app.root_path, '..', 'logs')
        os.makedirs(log_dir, exist_ok=True)
        
        email_logger = logging.getLogger('email_logger')
        if not email_logger.handlers:
            email_logger.setLevel(logging.INFO)
            handler = logging.FileHandler(os.path.join(log_dir, 'email_logs.txt'))
            formatter = logging.Formatter('%(asctime)s - %(message)s')
            handler.setFormatter(formatter)
            email_logger.addHandler(handler)
        
        email_logger.info(
            f"Email would be sent to {user.email}, Token: {token}, URL: {confirmation_url}\n"
            f"Content:\n{message}"
        )
        
        print(f"CONFIRMATION URL: {confirmation_url}")
    else:
        # Actually send the email
        import smtplib, ssl
        
        # Get password from environment variable or configuration
        gmail_password = current_app.config.get('GMAIL_TOKEN') or os.environ.get('GMAILTOKEN')
        
        if not gmail_password and current_app.config.get('MAIL_USERNAME'):
            gmail_password = current_app.config.get('MAIL_PASSWORD')
        
        if not gmail_password:
            raise ValueError("Email password not configured. Set GMAILTOKEN environment variable or MAIL_PASSWORD in config.")
        
        context = ssl.create_default_context()
        with smtplib.SMTP(smtp_server, port) as server:
            server.ehlo()
            
            # Use TLS only if necessary (e.g., Gmail)
            if current_app.config.get('MAIL_USE_TLS', False):
                server.starttls(context=context)
                server.ehlo()
                
            # Login only if required
            if current_app.config.get('MAIL_USERNAME'):
                server.login(sender, gmail_password)
                
            server.sendmail(sent_from, recipient, message)
    
    return True

def send_password_reset_email(user, token):
    """Send password reset link to user"""
    from app import db
    from flask import current_app, url_for
    import os

    # Path for logs
    log_dir = os.path.join(os.path.dirname(current_app.root_path), 'logs')
    os.makedirs(log_dir, exist_ok=True)

    # Configurations from the app
    smtp_server = current_app.config.get('MAIL_SERVER', 'localhost')
    port = current_app.config.get('MAIL_PORT', 1025)
    sender = current_app.config.get('EMAIL_SENDER', 'info@gftinnovation.eu')
    sent_from = current_app.config.get('EMAIL_NOREPLY', 'no-reply@gftinnovation.eu')

    # Build the password reset URL
    reset_url = url_for('auth.reset_password', 
                        token=token, 
                        _external=True)

    recipient = user.email

    message = f"""Subject: {current_app.config.get('EMAIL_SENDER_NAME', 'FAME FEDERATION')} - Password Reset
To: {recipient}
From: {sent_from}

Hello {user.first_name},

You have requested to reset your password. Please click on the link below to set a new password:

{reset_url}

This link will expire in 24 hours.

If you did not request a password reset, please ignore this email.

Best regards,
--FAME FEDERATION--
(Please do NOT REPLY to this email)
"""

    # For development environments, we can opt for logging instead of actual sending
    if current_app.config.get('MAIL_SUPPRESS_SEND', False):
        # Log the email instead of actually sending it
        import logging
        
        # Configure logger
        email_logger = logging.getLogger('email_logger')
        if not email_logger.handlers:
            email_logger.setLevel(logging.INFO)
            handler = logging.FileHandler(os.path.join(log_dir, 'email_logs.txt'))
            formatter = logging.Formatter('%(asctime)s - %(message)s')
            handler.setFormatter(formatter)
            email_logger.addHandler(handler)
        
        email_logger.info(
            f"Password reset email would be sent to {user.email}, Token: {token}, URL: {reset_url}\n"
            f"Content:\n{message}"
        )
        
        print(f"PASSWORD RESET URL: {reset_url}")
        return True
    else:
        # Actually send the email
        import smtplib, ssl
        
        # Get password from environment variable or configuration
        gmail_password = current_app.config.get('GMAIL_TOKEN') or os.environ.get('GMAILTOKEN')
        
        if not gmail_password and current_app.config.get('MAIL_USERNAME'):
            gmail_password = current_app.config.get('MAIL_PASSWORD')
        
        if not gmail_password:
            raise ValueError("Email password not configured. Set GMAILTOKEN environment variable or MAIL_PASSWORD in config.")
        
        try:
            context = ssl.create_default_context()
            with smtplib.SMTP(smtp_server, port) as server:
                server.ehlo()
                
                # Use TLS only if necessary (e.g., Gmail)
                if current_app.config.get('MAIL_USE_TLS', False):
                    server.starttls(context=context)
                    server.ehlo()
                    
                # Login only if required
                if current_app.config.get('MAIL_USERNAME'):
                    server.login(sender, gmail_password)
                    
                server.sendmail(sent_from, recipient, message)
            return True
        except Exception as e:
            print(f"Error sending password reset email: {e}")
            return False
        
def send_voting_result_email(user, organization_name, status):
    """Send voting result notification email to user
    
    Args:
        user: User object to send the email to
        organization_name: Name of the organization
        status: Result status ('approved' or 'rejected')
        
    Returns:
        bool: True if the email was sent successfully, False otherwise
    """
    from flask import current_app
    import os
    import smtplib, ssl
    
    # Calculate the correct path for the logs directory
    # current_app.root_path is the 'app' directory
    # so we need to go one level up
    log_dir = os.path.join(os.path.dirname(current_app.root_path), 'logs')
    os.makedirs(log_dir, exist_ok=True)
    
    # Use configurations from the app's configuration
    smtp_server = current_app.config.get('MAIL_SERVER', 'localhost')
    port = current_app.config.get('MAIL_PORT', 1025)
    sender = current_app.config.get('EMAIL_SENDER', 'info@gftinnovation.eu')
    sent_from = current_app.config.get('EMAIL_NOREPLY', 'no-reply@gftinnovation.eu')
    
    recipient = user.email
    
    # Determine the content based on status
    if status == 'approved':
        subject = f"{current_app.config.get('EMAIL_SENDER_NAME', 'FAME FEDERATION')} - Your Federation Request was Approved!"
        content = f"""
Hello {user.first_name},

Great news! The voting for your organization "{organization_name}" has concluded, and we're pleased to inform you that your federation request has been APPROVED.

Your organization is now an official member of the FAME Federation. You can now access all the features and benefits of the federation.

Please log in to your control_panel to see the details and start using the marketplace.

Best regards,
--FAME FEDERATION--
(Please do NOT REPLY to this email)
"""
    else:
        subject = f"{current_app.config.get('EMAIL_SENDER_NAME', 'FAME FEDERATION')} - Your Federation Request was Not Approved"
        content = f"""
Hello {user.first_name},

The voting for your organization "{organization_name}" has concluded, and we regret to inform you that your federation request was not approved.

Please log in to your control_panel for more details.

Best regards,
--FAME FEDERATION--
(Please do NOT REPLY to this email)
"""

    message = f"""Subject: {subject}
To: {recipient}
From: {sent_from}

{content}
"""
    
    # For development environments, we can opt for logging instead of actual sending
    if current_app.config.get('MAIL_SUPPRESS_SEND', False):
        # Log the email instead of actually sending it
        import logging
        
        # Configure logger
        email_logger = logging.getLogger('email_logger')
        if not email_logger.handlers:
            email_logger.setLevel(logging.INFO)
            handler = logging.FileHandler(os.path.join(log_dir, 'email_logs.txt'))
            formatter = logging.Formatter('%(asctime)s - %(message)s')
            handler.setFormatter(formatter)
            email_logger.addHandler(handler)
        
        email_logger.info(
            f"Email would be sent to {user.email}, Status: {status}\n"
            f"Content:\n{message}"
        )
        
        print(f"VOTING RESULT EMAIL: Status={status}, Recipient={user.email}")
        return True
    else:
        # Actually send the email
        try:
            # Get password from environment variable or configuration
            gmail_password = current_app.config.get('GMAIL_TOKEN') or os.environ.get('GMAILTOKEN')
            
            if not gmail_password and current_app.config.get('MAIL_USERNAME'):
                gmail_password = current_app.config.get('MAIL_PASSWORD')
            
            if not gmail_password:
                raise ValueError("Email password not configured. Set GMAILTOKEN environment variable or MAIL_PASSWORD in config.")
            
            context = ssl.create_default_context()
            with smtplib.SMTP(smtp_server, port) as server:
                server.ehlo()
                
                # Use TLS only if necessary (e.g., Gmail)
                if current_app.config.get('MAIL_USE_TLS', False):
                    server.starttls(context=context)
                    server.ehlo()
                    
                # Login only if required
                if current_app.config.get('MAIL_USERNAME'):
                    server.login(sender, gmail_password)
                    
                server.sendmail(sent_from, recipient, message)
            return True
        except Exception as e:
            print(f"Error sending voting result email: {e}")
            return False
        
def send_new_application_notification(pending_organization, representatives):
    """
    Send notification emails to all representatives of approved organizations 
    about a new federation request pending for voting
    
    Args:
        pending_organization: The Organization object that has just been set to pending status
        representatives: List of User objects who are representatives of approved organizations
        
    Returns:
        bool: True if the emails were sent successfully, False otherwise
    """
    from flask import current_app
    import os
    import logging
    import smtplib, ssl
    
    # Calculate the correct path for the logs directory
    log_dir = os.path.join(os.path.dirname(current_app.root_path), 'logs')
    os.makedirs(log_dir, exist_ok=True)
    
    # Use configurations from the app's configuration
    smtp_server = current_app.config.get('MAIL_SERVER', 'localhost')
    port = current_app.config.get('MAIL_PORT', 1025)
    sender = current_app.config.get('EMAIL_SENDER', 'info@gftinnovation.eu')
    sent_from = current_app.config.get('EMAIL_NOREPLY', 'no-reply@gftinnovation.eu')
    
    subject = f"{current_app.config.get('EMAIL_SENDER_NAME', 'FAME FEDERATION')} - New Federation Request to Review"
    
    # Get the timestamp for when voting ends
    voting_ends = pending_organization.voting_ends_at.strftime('%Y-%m-%d %H:%M UTC') if pending_organization.voting_ends_at else "as per federation rules"
    
    # Get app URL 
    base_url = current_app.config.get('APP_URL', 'https://federation.fame-horizon.eu')
    
    # For development environments, we can opt for logging instead of actually sending
    if current_app.config.get('MAIL_SUPPRESS_SEND', False):
        # Configure logger
        email_logger = logging.getLogger('email_logger')
        if not email_logger.handlers:
            email_logger.setLevel(logging.INFO)
            handler = logging.FileHandler(os.path.join(log_dir, 'email_logs.txt'))
            formatter = logging.Formatter('%(asctime)s - %(message)s')
            handler.setFormatter(formatter)
            email_logger.addHandler(handler)
        
        # Log emails for each recipient
        for rep in representatives:
            content = f"""
Hello {rep.first_name},

A new organization "{pending_organization.legal_name}" has applied for federation membership and requires your review and vote.

Organization details:
- Name: {pending_organization.legal_name}
- Type: {pending_organization.org_type}
- Country: {pending_organization.legal_address.country if pending_organization.legal_address else "N/A"}

Please log in to your control_panel at {base_url}/control_panel to review the application and cast your vote.

The voting period ends on {voting_ends}.

Best regards,
--FAME FEDERATION--
(Please do NOT REPLY to this email)
"""
            message = f"""Subject: {subject}
To: {rep.email}
From: {sent_from}

{content}
"""
            email_logger.info(
                f"Notification email would be sent to {rep.email} about new organization: {pending_organization.legal_name}\n"
                f"Content:\n{message}"
            )
            
            print(f"NEW APPLICATION NOTIFICATION: Recipient={rep.email}, Organization={pending_organization.legal_name}")
        
        return True
    else:
        # Actually send the emails
        try:
            # Get password from environment variable or configuration
            gmail_password = current_app.config.get('GMAIL_TOKEN') or os.environ.get('GMAILTOKEN')
            
            if not gmail_password and current_app.config.get('MAIL_USERNAME'):
                gmail_password = current_app.config.get('MAIL_PASSWORD')
            
            if not gmail_password:
                raise ValueError("Email password not configured. Set GMAILTOKEN environment variable or MAIL_PASSWORD in config.")
            
            success = True
            context = ssl.create_default_context()
            
            for rep in representatives:
                try:
                    content = f"""
Hello {rep.first_name},

A new organization "{pending_organization.legal_name}" has applied for federation membership and requires your review and vote.

Organization details:
- Name: {pending_organization.legal_name}
- Type: {pending_organization.org_type}
- Country: {pending_organization.legal_address.country if pending_organization.legal_address else "N/A"}

Please log in to your control_panel at {base_url}/control_panel to review the application and cast your vote.

The voting period ends on {voting_ends}.

Best regards,
--FAME FEDERATION--
(Please do NOT REPLY to this email)
"""
                    message = f"""Subject: {subject}
To: {rep.email}
From: {sent_from}

{content}
"""
                    
                    with smtplib.SMTP(smtp_server, port) as server:
                        server.ehlo()
                        
                        # Use TLS only if necessary (e.g., Gmail)
                        if current_app.config.get('MAIL_USE_TLS', False):
                            server.starttls(context=context)
                            server.ehlo()
                            
                        # Login only if required
                        if current_app.config.get('MAIL_USERNAME'):
                            server.login(sender, gmail_password)
                            
                        server.sendmail(sent_from, rep.email, message)
                        
                except Exception as e:
                    print(f"Error sending notification to {rep.email}: {e}")
                    success = False
                    
            return success
        except Exception as e:
            print(f"Error in email sending process: {e}")
            return False