# app/services/marketplace_user_service.py

import logging
from ..models import MarketplaceUser
from .. import db
from .member_service import MemberService

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

class MarketplaceUserService:
    @staticmethod
    def create_marketplace_user(user_data):
        """Create a new marketplace user and synchronize with external API"""
        try:
            logger.info("Creating a new marketplace user...")
            
            # Verify that required fields are present
            if not all(key in user_data for key in ['first_name', 'last_name', 'email', 'organization_id']):
                logger.warning("User creation failed: missing required fields")
                return None, "Required fields missing: first_name, last_name, email, and organization_id are required"
            
            # Check if email is already in use by another user in the same organization
            existing_user = MarketplaceUser.query.filter_by(
                email=user_data['email'].lower(),
                organization_id=user_data['organization_id']
            ).first()
            
            if existing_user:
                logger.warning(f"User creation failed: email {user_data['email']} already exists in organization {user_data['organization_id']}")
                return None, "Email already exists for this organization"
            
            # Check nickname length
            if 'nickname' in user_data and user_data['nickname'] and len(user_data['nickname']) > 15:
                logger.warning("User creation failed: nickname exceeds 15 characters")
                return None, "Nickname cannot exceed 15 characters"
            
            # Get organization to retrieve PID
            from ..models import Organization
            organization = Organization.query.get(user_data['organization_id'])
            if not organization:
                logger.warning(f"User creation failed: organization with ID {user_data['organization_id']} not found")
                return None, "Organization not found"
            
            # Prepare data for external API
            api_user_data = {
                "role": user_data.get('role', 'user'),
                "country": organization.legal_address.country,
                "affiliation": organization.pid,
                "contact": {
                    "firstName": user_data['first_name'],
                    "lastName": user_data['last_name'],
                    "email": user_data['email'].lower(),
                }
            }

            # Add optional fields only if present
            if 'nickname' in user_data and user_data['nickname']:
                api_user_data["nickname"] = user_data['nickname']
                
            if 'phone' in user_data and user_data['phone']:
                api_user_data["contact"]["phone"] = user_data['phone']
            
            # Call external API to create user
            logger.info("Sending user data to external API...")
            api_response, api_error = MemberService.create_user(api_user_data)
            
            # Simulate API call - remove in production ######################
            # from ..utils import test_api_proxy
            #api_response, api_error = test_api_proxy.simulate_api_call(api_user_data)
            #########################################################################
            
            
            if api_error:
                logger.error(f"API call failed: {api_error}")
                return None, f"API Error: {api_error}"
            
            logger.info("External API user creation successful")
            # Set status - default is 'invited'
            status = user_data.get('status', 'invited')
            
            # If API was successful, create user in local database
            user = MarketplaceUser(
                first_name=user_data['first_name'],
                last_name=user_data['last_name'],
                email=user_data['email'].lower(),  # Normalizza email
                organization_id=user_data['organization_id'],
                nickname=user_data.get('nickname'),
                phone=user_data.get('phone'),
                role=user_data.get('role', 'user'),
                status=status,
                # User synchronized with external API
            )
            
            # If API returned a UID, save it
            if api_response and 'id' in api_response:
                user.external_uid = api_response['id']
            
            db.session.add(user)
            db.session.commit()
            logger.info(f"User {user.email} created successfully in local DB")
            return user, None
            
        except Exception as e:
            logger.exception("Exception occurred during user creation")
            db.session.rollback()
            return None, str(e)
    
    @staticmethod
    def get_marketplace_user_by_id(user_id):
        """Get marketplace user by ID"""
        logger.info(f"Fetching marketplace user with ID {user_id}")
        return MarketplaceUser.query.get(user_id)
    
    @staticmethod
    def get_marketplace_users_by_organization(organization_id):
        """Get all marketplace users for an organization"""
        logger.info(f"Fetching all marketplace users for organization {organization_id}")
        return MarketplaceUser.query.filter_by(organization_id=organization_id).all()
    
    # Never called, but might be useful in the future
    @staticmethod
    def update_marketplace_user(user_id, update_data):
        """Update marketplace user details and synchronize with external API"""
        try:
            logger.info(f"Updating marketplace user with ID {user_id}")
            user = MarketplaceUser.query.get(user_id)
            if not user:
                logger.warning(f"Update failed: user {user_id} not found")
                return None, "User not found"
            
            # Prepare data for the external API according to the required structure
            api_update_data = {}
            contact_updates = {}
            
            # Update fields if present in the update data
            if 'first_name' in update_data:
                user.first_name = update_data['first_name']
                contact_updates['firstName'] = update_data['first_name']
                
            if 'last_name' in update_data:
                user.last_name = update_data['last_name']
                contact_updates['lastName'] = update_data['last_name']
                
            if 'email' in update_data:
                # Verify that the email is not already in use by another user in the same organization
                if update_data['email'] != user.email:
                    existing_user = MarketplaceUser.query.filter_by(
                        email=update_data['email'],
                        organization_id=user.organization_id
                    ).first()
                    
                    if existing_user:
                        logger.warning(f"Update failed: email {update_data['email']} already exists")
                        return None, "Email already exists for this organization"
                    
                user.email = update_data['email'].lower()
                contact_updates['email'] = update_data['email'].lower()
                
            if 'nickname' in update_data:
                # Verify nickname length
                if update_data['nickname'] and len(update_data['nickname']) > 15:
                    logger.warning("Update failed: nickname exceeds 15 characters")
                    return None, "Nickname cannot exceed 15 characters"
                user.nickname = update_data['nickname']
                api_update_data['nickname'] = update_data['nickname']
                
            if 'phone' in update_data:
                user.phone = update_data['phone']
                contact_updates['phone'] = update_data['phone']
                
            if 'role' in update_data:
                user.role = update_data['role']
                api_update_data['role'] = update_data['role']
                
            if 'country' in update_data:
                api_update_data['country'] = update_data['country']
                
            if 'sync' in update_data:
                user.sync = update_data['sync']
            
            # Add contact only if there are updates for the contact fields
            if contact_updates:
                api_update_data['contact'] = contact_updates
            
            # If the user has an external_uid and there are data updates for the API
            if hasattr(user, 'external_uid') and user.external_uid and (api_update_data or contact_updates):
                # Call API to update external user
                logger.info(f"Synchronizing update with external API for user {user_id}")
                api_success, api_error = MemberService.update_user(user.external_uid, api_update_data)
                
                if api_error:
                    # Rollback in case of API error
                    logger.error(f"API update failed: {api_error}")
                    db.session.rollback()
                    return None, f"API Error: {api_error}"
            
            db.session.commit()
            logger.info(f"Marketplace user {user_id} updated successfully")
            return user, None
            
        except Exception as e:
            logger.exception(f"Exception occurred while updating user {user_id}")
            db.session.rollback()
            return None, str(e)
    
    @staticmethod
    def delete_marketplace_user(user_id):
        """Delete a marketplace user and synchronize with external API"""
        try:
            logger.info(f"Deleting marketplace user with ID {user_id}")
            user = MarketplaceUser.query.get(user_id)
            if not user:
                logger.warning(f"Delete failed: user {user_id} not found")
                return False, "User not found"
            
            # If the user has an external_uid, delete from the external API
            if hasattr(user, 'external_uid') and user.external_uid:
                logger.info(f"Synchronizing delete with external API for user {user_id}")
                api_success, api_error = MemberService.delete_user(user.external_uid)
                
                if api_error:
                    logger.error(f"API delete failed: {api_error}")
                    return False, f"API Error: {api_error}"
            
            # Elimina l'utente dal database locale
            db.session.delete(user)
            db.session.commit()
            logger.info(f"Marketplace user {user_id} deleted successfully")
            return True, None
            
        except Exception as e:
            logger.exception(f"Exception occurred while deleting user {user_id}")
            db.session.rollback()
            return False, str(e)
    
    @staticmethod
    def set_sync_status(user_id, synced=True):
        """
        Imposta lo stato di sincronizzazione di un utente
        
        Args:
            user_id: ID dell'utente
            synced: Valore booleano del flag di sincronizzazione (default: True)
            
        Returns:
            tuple: (success, error_message)
        """
        try:
            logger.info(f"Setting sync status for user {user_id} to {synced}")
            user = MarketplaceUser.query.get(user_id)
            if not user:
                logger.warning(f"Sync status update failed: user {user_id} not found")
                return False, "User not found"
                
            user.sync = synced
            db.session.commit()
            logger.info(f"Sync status updated successfully for user {user_id}")
            return True, None
            
        except Exception as e:
            logger.exception(f"Exception occurred while setting sync status for user {user_id}")
            db.session.rollback()
            return False, str(e)
    
    # TODO Aggiorare Data Model con campo status: onboarded-offboarded 
    
    @staticmethod
    def offboard_marketplace_user(user_id):
        """Offboard a marketplace user and synchronize with external API"""
        try:
            logger.info(f"Offboarding user {user_id}")
            user = MarketplaceUser.query.get(user_id)
            if not user:
                logger.warning(f"Offboard failed: user {user_id} not found")
                return False, "User not found"
            
            # If the user has an external_uid, offboard from the external API
            if hasattr(user, 'external_uid') and user.external_uid:
                logger.info(f"Synchronizing offboard with external API for user {user_id}")
                api_success, api_error = MemberService.offboard_user(user.external_uid)
                
                if api_error:
                    logger.error(f"API offboard failed: {api_error}")
                    return False, f"API Error: {api_error}"
            
            # Update local user status (we could add a 'status' field or similar)
            user.sync = False
            db.session.commit()
            logger.info(f"User {user_id} offboarded successfully")
            return True, None
            
        except Exception as e:
            logger.exception(f"Exception occurred while offboarding user {user_id}")
            db.session.rollback()
            return False, str(e)
        
    #commento
    
    @staticmethod
    def reinvite_marketplace_user(user_id):
        user = MarketplaceUser.query.get(user_id)
        if not user:
            return False, "User not found"
        
        if user.external_uid:
            # Chiama l'API esterna tramite MemberService
            api_success, api_error = MemberService.reinvite_user(user.external_uid)
            if api_error:
                return False, api_error
        
        user.sync = True  # Aggiorna lo stato di sincronizzazione
        db.session.commit()
        return True, None
