"""
FAME Marketplace Wallet Service
Handles FAME token operations and Digital Euro integration
"""
import requests
from app.models.digital_euro_account import DigitalEuroAccount, WalletTransaction
from app.models.organization import Organization
from app.models.user import User
from app.services.digital_euro_service import digital_euro_service
from app.config import Config
from app import db
from datetime import datetime
import logging

logger = logging.getLogger(__name__)

# Mock Marketplace Holding ID (placeholder for real marketplace wallet)
MARKETPLACE_HOLDING_ID = '00000000-0000-0000-0000-000000000000'  # Placeholder


class MarketplaceWalletService:
    """Service for FAME Marketplace wallet operations"""

    def get_organization_trading_accounts(self, organization):
        """
        Get all trading accounts for organization (filtered by PID)
        """
        # Uncomment when GOV API endpoint is available
        try:
            response = requests.get(
                f'{Config.FAME_API_BASE_URL}/v1.0/accounts',
                timeout=30
            )
            response.raise_for_status()
            accounts_result = response.json()
                
        except requests.exceptions.RequestException as e:
            logger.error(f"GOV API Error (get_organization_trading_accounts): {str(e)}")
            accounts_result = {'error': str(e)}
        
        if isinstance(accounts_result, dict) and 'error' in accounts_result:
            logger.error(f"Error fetching accounts: {accounts_result['error']}")
            return []
        
        if not isinstance(accounts_result, list):
            return []
        
        # Filter by organization PID and enrollment status
        org_accounts = []
        for account in accounts_result:
            # Check correct PID, enrolled, not disenrolled
            if (account.get('owningMember') == organization.pid and 
                account.get('enrolled') and 
                not account.get('disenrolled')):
                
                # Fetch user details from GOV API when endpoint is available
                # When API is ready, replace mock values below with real API call:
                
                owning_user_uid = account.get('owningUser')
                if owning_user_uid:
                    try:
                        user_response = requests.get(
                            f'{Config.FAME_API_BASE_URL}/v1.0/users/{owning_user_uid}',
                            timeout=30
                        )
                        user_response.raise_for_status()
                        user_data = user_response.json()
                        
                        # Extract user info from API response
                        account['user_email'] = user_data.get('email', 'N/A')
                        account['user_name'] = f"{user_data.get('firstName', '')} {user_data.get('lastName', '')}".strip()
                        
                        # Optional: Add more fields if needed
                        # account['user_phone'] = user_data.get('phone')
                        # account['user_country'] = user_data.get('country')
                        # account['user_nickname'] = user_data.get('nickname')
                        
                    except requests.exceptions.RequestException as e:
                        logger.error(f"Failed to fetch user details for UID {owning_user_uid}: {str(e)}")
                        # Fallback to default values on API error
                        account['user_email'] = f"user_{owning_user_uid[:8]}@unknown.com"
                        account['user_name'] = f"User {owning_user_uid[:8]}"
                else:
                    account['user_email'] = 'N/A'
                    account['user_name'] = 'Unknown User'                
                
                org_accounts.append(account)
        
        return org_accounts

    
#    def get_organization_trading_accounts(self, organization):
#        """
#        Get all trading accounts for organization (filtered by PID)
#        """
#        # Uncomment when GOV API endpoint is available
#        try:
#            proxy_url = 'https://federation.fame-horizon.eu/api/proxy'
#            
#            payload = {
#                'method': 'GET',
#                'url': f'{Config.FAME_API_BASE_URL}/v1.0/accounts',
#                'headers': {
#                    'Content-Type': 'application/json'
#                }
#            }
#            
#            response = requests.post(
#                proxy_url,
#                json=payload,
#                auth=(Config.INTERNAL_API_USERNAME, Config.INTERNAL_API_PASSWORD),
#                headers={'Content-Type': 'application/json'},
#                timeout=30
#            )
#            response.raise_for_status()
#            result = response.json()
#            accounts_result = result.get('data', result)
#                
#        except requests.exceptions.RequestException as e:
#            logger.error(f"GOV API Error (get_organization_trading_accounts): {str(e)}")
#            accounts_result = {'error': str(e)}        
#        
#        if isinstance(accounts_result, dict) and 'error' in accounts_result:
#            logger.error(f"Error fetching accounts: {accounts_result['error']}")
#            return []
#        
#        if not isinstance(accounts_result, list):
#            return []
#        
#        # Filter by organization PID and enrollment status
#        org_accounts = []
#        for account in accounts_result:
#            # Check correct PID, enrolled, not disenrolled
#            if (account.get('owningMember') == organization.pid and 
#                account.get('enrolled') and 
#                not account.get('disenrolled')):
#                
#                # Fetch user details from GOV API when endpoint is available
#                # When API is ready, replace mock values below with real API call:
#                
#                owning_user_uid = account.get('owningUser')
#                if owning_user_uid:
#                    try:
#                        proxy_url = 'https://federation.fame-horizon.eu/api/proxy'
#                        
#                        user_payload = {
#                            'method': 'GET',
#                            'url': f'{Config.FAME_API_BASE_URL}/v1.0/users/{owning_user_uid}',
#                            'headers': {
#                                'Content-Type': 'application/json'
#                            }
#                        }
#                        
#                        user_response = requests.post(
#                            proxy_url,
#                            json=user_payload,
#                            auth=(Config.INTERNAL_API_USERNAME, Config.INTERNAL_API_PASSWORD),
#                            headers={'Content-Type': 'application/json'},
#                            timeout=30
#                        )
#                        user_response.raise_for_status()
#                        user_result = user_response.json()
#                        user_data = user_result.get('data', user_result)
#                        
#                        # Extract user info from API response
#                        account['user_email'] = user_data.get('email', 'N/A')
#                        account['user_name'] = f"{user_data.get('firstName', '')} {user_data.get('lastName', '')}".strip()
#                        
#                        # Optional: Add more fields if needed
#                        # account['user_phone'] = user_data.get('phone')
#                        # account['user_country'] = user_data.get('country')
#                        # account['user_nickname'] = user_data.get('nickname')
#                        
#                    except requests.exceptions.RequestException as e:
#                        logger.error(f"Failed to fetch user details for UID {owning_user_uid}: {str(e)}")
#                        # Fallback to default values on API error
#                        account['user_email'] = f"user_{owning_user_uid[:8]}@unknown.com"
#                        account['user_name'] = f"User {owning_user_uid[:8]}"
#                else:
#                    account['user_email'] = 'N/A'
#                    account['user_name'] = 'Unknown User'
#                
#                org_accounts.append(account)
#        
#        return org_accounts

    
    def get_balance(self, trading_account_id):
        """Get FAME token balance (direct API call for production)"""
        try:
            response = requests.get(
                f'{Config.FAME_API_BASE_URL}/v1.0/accounts/{trading_account_id}/balance',
                timeout=30
            )
            response.raise_for_status()
            
            # Get API response
            api_data = response.json()
            
            # Converte balance da string a float
            balance_str = api_data.get('balance', '0')
            try:
                balance_float = float(balance_str)
            except (ValueError, TypeError):
                logger.warning(f"Invalid balance format: {balance_str}, defaulting to 0.0")
                balance_float = 0.0
            
            return {
                'status': 'success',
                'balance': balance_float,
                'account_status': api_data.get('status', 'unknown')
            }
            
        except requests.exceptions.RequestException as e:
            logger.error(f"GOV API Error (get_balance): {str(e)}")
            return {'error': str(e), 'status': 'error'}


#    def get_balance(self, trading_account_id):
#        """Get FAME token balance via internal proxy API"""
#        try:
#            proxy_url = 'https://federation.fame-horizon.eu/api/proxy'
#            
#            payload = {
#                'method': 'GET',
#                'url': f'{Config.FAME_API_BASE_URL}/v1.0/accounts/{trading_account_id}/balance',
#                'headers': {
#                    'Content-Type': 'application/json'
#                }
#            }
#            
#            response = requests.post(
#                proxy_url,
#                json=payload,
#                auth=(Config.INTERNAL_API_USERNAME, Config.INTERNAL_API_PASSWORD),
#                headers={'Content-Type': 'application/json'},
#                timeout=30
#            )
#            response.raise_for_status()
#            
#            result = response.json()
#            
#            # Extract data from proxy wrapper
#            api_data = result.get('data', result)
#            
#            # Converte balance da string a float
#            balance_str = api_data.get('balance', '0')
#            try:
#                balance_float = float(balance_str)
#            except (ValueError, TypeError):
#                logger.warning(f"Invalid balance format: {balance_str}, defaulting to 0.0")
#                balance_float = 0.0#            
#            
#            return {
#                'status': 'success',
#                'balance': balance_float,
#                'account_status': api_data.get('status', 'unknown')
#            }
#            
#        except requests.exceptions.RequestException as e:
#            logger.error(f"GOV API Error (get_balance): {str(e)}")
#            return {'error': str(e), 'status': 'error'}
    
    
    def get_transactions(self, trading_account_id, limit=50):
        """Get transaction history"""
        try:
            response = requests.get(
                f'{Config.FAME_API_BASE_URL}/v1.0/accounts/{trading_account_id}/transactions?limit={limit}',
                timeout=30
            )
            response.raise_for_status()
            return response.json()
        except requests.exceptions.RequestException as e:
            logger.error(f"GOV API Error (get_transactions): {str(e)}")
            return {'error': str(e), 'status': 'error'}
        

    def purchase_tokens(self, user, organization, trading_account_id, amount, trading_account_owner=None):
        """
        Purchase FAME tokens using Digital Euro
        
        Flow (SAFE):
        1. Create reservation in Digital Euro (funds blocked, not transferred)
        2. Credit FAME tokens to trading account
        3. If credit succeeds → Confirm reservation (complete payment)
        4. If credit fails → Cancel reservation (rollback, no money lost)
        """
        reservation_id = None
        
        try:    
            # Validate amount is integer
            amount = int(amount)
            if amount <= 0:
                return {'error': 'Amount must be positive', 'status': 'error'}
            
            # Get Digital Euro Account
            de_account = DigitalEuroAccount.query.filter_by(
                organization_id=organization.id,
                is_active=True
            ).first()
            
            if not de_account:
                return {'error': 'No Digital Euro account linked', 'status': 'error'}
            
            # Check balance
            de_amount, de_available, de_limit = digital_euro_service.get_balance(de_account.holding_id)
            if de_available < amount:
                return {
                    'error': f'Insufficient balance. Available: {de_available}',
                    'status': 'error'
                }
            
            # STEP 1: Create RESERVATION (funds blocked, not transferred yet)
            logger.info(f"Creating Digital Euro reservation for {amount} EUR")
            reservation_result = digital_euro_service.create_reservation(
                debtor_id=de_account.holding_id,
                creditor_id=MARKETPLACE_HOLDING_ID,
                amount=amount
            )
            
            if 'error' in reservation_result:
                logger.error(f"DE reservation failed: {reservation_result['error']}")
                return {
                    'error': f"Could not reserve funds: {reservation_result['error']}",
                    'status': 'error'
                }
            
            reservation_id = reservation_result.get('entry')
            logger.info(f"Reservation created: {reservation_id}")
            
            # STEP 2: Try to credit FAME tokens (BEFORE completing payment)
            logger.info(f"Attempting to credit {amount} FAME tokens to {trading_account_id}")
            
            try:
                response = requests.post(
                    f'{Config.FAME_API_BASE_URL}/v1.0/accounts/{trading_account_id}/credit',
                    json={'amount': f"{amount:.2f}"},  # API expects string
                    timeout=30
                )
                response.raise_for_status()
                credit_result = response.json()
            except requests.exceptions.RequestException as e:
                logger.error(f"Marketplace credit failed: {str(e)}")
                credit_result = {'error': str(e)}
            
            if isinstance(credit_result, dict) and 'error' in credit_result:
                logger.error(f"Marketplace credit failed: {credit_result['error']}")
                
                # ROLLBACK: Cancel reservation (funds released back)
                logger.warning(f"Rolling back reservation {reservation_id}")
                cancel_result = digital_euro_service.cancel_reservation(reservation_id)
                
                if 'error' in cancel_result:
                    # Alert admin - manual intervention needed
                    logger.critical(f"CRITICAL: Could not cancel reservation {reservation_id}: {cancel_result['error']}")
                
                return {
                    'error': f"Marketplace credit failed: {credit_result['error']}",
                    'status': 'error'
                }
            
            # STEP 3: Credit succeeded! Now complete the payment
            logger.info(f"FAME credit successful, completing payment from reservation {reservation_id}")
            payment_result = digital_euro_service.payment_from_reservation(
                reservation_id=reservation_id,
                amount=amount
            )
            
            if 'error' in payment_result:
                # Need manual reconciliation
                logger.critical(f"CRITICAL: Payment from reservation failed but FAME already credited!")
                logger.critical(f"Reservation: {reservation_id}, Amount: {amount}")
                return {
                    'error': 'Payment completion failed. Contact support.',
                    'status': 'error',
                    'requires_manual_reconciliation': True,
                    'reservation_id': reservation_id
                }
            
            de_payment_id = payment_result.get('entry')
            
            # STEP 4: Record successful transaction
            tx = WalletTransaction(
                organization_id=organization.id,
                user_id=user.id,
                trading_account_id=trading_account_id,
                trading_account_owner=trading_account_owner,
                transaction_type='purchase',
                amount=amount,
                currency='EUR',
                de_payment_id=de_payment_id,
                marketplace_tx_id=credit_result.get('marketplace_transaction_id'),
                description=f'Purchased {amount} FAME tokens',
                status='completed',
                completed_at=datetime.utcnow()
            )
            db.session.add(tx)
            db.session.commit()
            
            logger.info(f"Purchase successful: {amount} FAME for org {organization.legal_name}")
            return {
                'status': 'success',
                'amount': amount,
                'de_payment_id': de_payment_id,
                'marketplace_tx_id': credit_result.get('marketplace_transaction_id'),
                'message': f'Successfully purchased {amount} FAME tokens'
            }
            
        except Exception as e:
            logger.error(f"Purchase tokens error: {str(e)}")
            
            # Try to cancel reservation if it was created
            if reservation_id:
                logger.warning(f"Exception occurred, attempting to cancel reservation {reservation_id}")
                try:
                    digital_euro_service.cancel_reservation(reservation_id)
                    logger.info(f"Reservation {reservation_id} cancelled successfully")
                except Exception as cancel_error:
                    logger.critical(f"Could not cancel reservation {reservation_id}: {str(cancel_error)}")
            
            db.session.rollback()
            return {'error': f'Internal error: {str(e)}', 'status': 'error'}     

#    def purchase_tokens(self, user, organization, trading_account_id, amount, trading_account_owner=None):
#        """
#        Purchase FAME tokens using Digital Euro
#        
#        Flow (SAFE):
#        1. Create reservation in Digital Euro (funds blocked, not transferred)
#        2. Credit FAME tokens to trading account
#        3. If credit succeeds → Confirm reservation (complete payment)
#        4. If credit fails → Cancel reservation (rollback, no money lost)
#        """
#        reservation_id = None
#        
#        try:
#            # Validate amount is integer
#            amount = int(amount)
#            if amount <= 0:
#                return {'error': 'Amount must be positive', 'status': 'error'}
#            
#            # Get Digital Euro Account
#            de_account = DigitalEuroAccount.query.filter_by(
#                organization_id=organization.id,
#                is_active=True
#            ).first()
#            
#            if not de_account:
#                return {'error': 'No Digital Euro account linked', 'status': 'error'}
#            
#            # Check balance
#            de_amount, de_available, de_limit = digital_euro_service.get_balance(de_account.holding_id)
#            if de_available < amount:
#                return {
#                    'error': f'Insufficient balance. Available: {de_available}',
#                    'status': 'error'
#                }
#            
#            # STEP 1: Create RESERVATION (funds blocked, not transferred yet)
#            logger.info(f"Creating Digital Euro reservation for {amount} EUR")
#            reservation_result = digital_euro_service.create_reservation(
#                debtor_id=de_account.holding_id,
#                creditor_id=MARKETPLACE_HOLDING_ID,
#                amount=amount
#            )
#            
#            if 'error' in reservation_result:
#                logger.error(f"DE reservation failed: {reservation_result['error']}")
#                return {
#                    'error': f"Could not reserve funds: {reservation_result['error']}",
#                    'status': 'error'
#                }
#            
#            reservation_id = reservation_result.get('entry')
#            logger.info(f"Reservation created: {reservation_id}")
#            
#            # STEP 2: Try to credit FAME tokens (BEFORE completing payment)
#            logger.info(f"Attempting to credit {amount} FAME tokens to {trading_account_id}")
#            
#            proxy_url = 'https://federation.fame-horizon.eu/api/proxy'
#            
#            credit_payload = {
#                'method': 'POST',
#                'url': f'{Config.FAME_API_BASE_URL}/v1.0/accounts/{trading_account_id}/credit',
#                'headers': {
#                    'Content-Type': 'application/json'
#                },
#                'data': {'amount': f"{amount:.2f}"}  # ✅ API expects "10.00" format
#            }
#            
#            try:
#                response = requests.post(
#                    proxy_url,
#                    json=credit_payload,
#                    auth=(Config.INTERNAL_API_USERNAME, Config.INTERNAL_API_PASSWORD),
#                    headers={'Content-Type': 'application/json'},
#                    timeout=30
#                )
#                response.raise_for_status()
#                result = response.json()
#                credit_result = result.get('data', result)
#            except requests.exceptions.RequestException as e:
#                logger.error(f"Marketplace credit failed: {str(e)}")
#                credit_result = {'error': str(e)}
#            
#            if isinstance(credit_result, dict) and 'error' in credit_result:
#                logger.error(f"Marketplace credit failed: {credit_result['error']}")
#                
#                # ROLLBACK: Cancel reservation (funds released back)
#                logger.warning(f"Rolling back reservation {reservation_id}")
#                cancel_result = digital_euro_service.cancel_reservation(reservation_id)
#                
#                if 'error' in cancel_result:
#                    # Alert admin - manual intervention needed
#                    logger.critical(f"CRITICAL: Could not cancel reservation {reservation_id}: {cancel_result['error']}")
#                
#                return {
#                    'error': f"Marketplace credit failed: {credit_result['error']}",
#                    'status': 'error'
#                }
#            
#            # STEP 3: Credit succeeded! Now complete the payment
#            logger.info(f"FAME credit successful, completing payment from reservation {reservation_id}")
#            payment_result = digital_euro_service.payment_from_reservation(
#                reservation_id=reservation_id,
#                amount=amount
#            )
#            
#            if 'error' in payment_result:
#                # Need manual reconciliation
#                logger.critical(f"CRITICAL: Payment from reservation failed but FAME already credited!")
#                logger.critical(f"Reservation: {reservation_id}, Amount: {amount}")
#                return {
#                    'error': 'Payment completion failed. Contact support.',
#                    'status': 'error',
#                    'requires_manual_reconciliation': True,
#                    'reservation_id': reservation_id
#                }
#            
#            de_payment_id = payment_result.get('entry')
#            
#            # STEP 4: Record successful transaction
#            tx = WalletTransaction(
#                organization_id=organization.id,
#                user_id=user.id,
#                trading_account_id=trading_account_id,
#                trading_account_owner=trading_account_owner,
#                transaction_type='purchase',
#                amount=amount,
#                currency='EUR',
#                de_payment_id=de_payment_id,
#                marketplace_tx_id=credit_result.get('marketplace_transaction_id'),
#                description=f'Purchased {amount} FAME tokens',
#                status='completed',
#                completed_at=datetime.utcnow()
#            )
#            db.session.add(tx)
#            db.session.commit()
#            
#            logger.info(f"Purchase successful: {amount} FAME for org {organization.legal_name}")
#            return {
#                'status': 'success',
#                'amount': amount,
#                'de_payment_id': de_payment_id,
#                'marketplace_tx_id': credit_result.get('marketplace_transaction_id'),
#                'message': f'Successfully purchased {amount} FAME tokens'
#            }
#            
#        except Exception as e:
#            logger.error(f"Purchase tokens error: {str(e)}")
#            
#            # Try to cancel reservation if it was created
#            if reservation_id:
#                logger.warning(f"Exception occurred, attempting to cancel reservation {reservation_id}")
#                try:
#                    digital_euro_service.cancel_reservation(reservation_id)
#                    logger.info(f"Reservation {reservation_id} cancelled successfully")
#                except Exception as cancel_error:
#                    logger.critical(f"Could not cancel reservation {reservation_id}: {str(cancel_error)}")
#            
#            db.session.rollback()
#            return {'error': f'Internal error: {str(e)}', 'status': 'error'}

        
    def redeem_tokens(self, user, organization, trading_account_id, amount, trading_account_owner=None):
        """
        Redeem FAME tokens back to Digital Euro
        
        Flow (SAFE):
        1. Debit FAME tokens from trading account FIRST
        2. If debit succeeds → Credit Digital Euro
        3. If debit fails → No money movement, no problem
        """
        try:
            # Validate amount is integer
            amount = int(amount)
            if amount <= 0:
                return {'error': 'Amount must be positive', 'status': 'error'}
            
            # Get Digital Euro Account
            de_account = DigitalEuroAccount.query.filter_by(
                organization_id=organization.id,
                is_active=True
            ).first()
            
            if not de_account:
                return {'error': 'No Digital Euro account linked', 'status': 'error'}
            
            # Check FAME balance            
            balance_result = self.get_balance(trading_account_id)
    
            if balance_result.get('status') != 'success':
                return {'error': 'Could not retrieve FAME balance', 'status': 'error'}
    
            fame_balance = balance_result.get('balance', 0.0)
    
            if fame_balance < amount:
                return {
                    'error': f'Insufficient FAME balance. Available: {fame_balance:.2f}',
                    'status': 'error'
                }
            
            # STEP 1: Debit FAME tokens FIRST (safer order)
            logger.info(f"Attempting to debit {amount} FAME tokens from {trading_account_id}")
            
            try:
                response = requests.post(
                    f'{Config.FAME_API_BASE_URL}/v1.0/accounts/{trading_account_id}/debit',
                    json={'amount': f"{amount:.2f}"},  # API expects "10.00" format
                    timeout=30
                )
                response.raise_for_status()
                debit_result = response.json()
            except requests.exceptions.RequestException as e:
                logger.error(f"Marketplace debit failed: {str(e)}")
                debit_result = {'error': str(e)}
            
            if isinstance(debit_result, dict) and 'error' in debit_result:
                logger.error(f"Marketplace debit failed: {debit_result['error']}")
                return {
                    'error': f"Marketplace debit failed: {debit_result['error']}",
                    'status': 'error'
                }
            
            marketplace_tx_id = debit_result.get('marketplace_transaction_id')  # ✅ Correct field name
            logger.info(f"FAME debit successful: {marketplace_tx_id}")
            
            # STEP 2: Credit Digital Euro (FAME already debited)
            logger.info(f"Crediting {amount} EUR to Digital Euro account")
            payment_result = digital_euro_service.p2p_payment(
                debtor_id=MARKETPLACE_HOLDING_ID,
                creditor_id=de_account.holding_id,
                amount=amount
            )
            
            if 'error' in payment_result:
                # Record as failed transaction for audit
                logger.critical(f"CRITICAL: DE payment failed but FAME already debited!")
                logger.critical(f"Trading Account: {trading_account_id}, Amount: {amount}, TX: {marketplace_tx_id}")
                
                tx = WalletTransaction(
                    organization_id=organization.id,
                    user_id=user.id,
                    trading_account_id=trading_account_id,
                    trading_account_owner=trading_account_owner,
                    transaction_type='redeem',
                    amount=amount,
                    currency='EUR',
                    de_payment_id=None,
                    marketplace_tx_id=marketplace_tx_id,
                    description=f'Redemption failed - FAME debited but DE credit failed',
                    status='failed',
                    error_message=f"Digital Euro payment error: {payment_result.get('error')}",
                    completed_at=datetime.utcnow()
                )
                db.session.add(tx)
                db.session.commit()
                
                return {
                    'error': 'Digital Euro credit failed. Contact support for refund.',
                    'status': 'error',
                    'requires_manual_reconciliation': True,
                    'marketplace_tx_id': marketplace_tx_id
                }
            
            de_payment_id = payment_result.get('entry')
            
            # STEP 3: Record successful transaction
            tx = WalletTransaction(
                organization_id=organization.id,
                user_id=user.id,
                trading_account_id=trading_account_id,
                trading_account_owner=trading_account_owner,  
                transaction_type='redeem',
                amount=amount,
                currency='EUR',
                de_payment_id=de_payment_id,
                marketplace_tx_id=marketplace_tx_id,
                description=f'Redeemed {amount} FAME tokens',
                status='completed',
                completed_at=datetime.utcnow()
            )
            db.session.add(tx)
            db.session.commit()
            
            logger.info(f"Redemption successful: {amount} FAME for org {organization.legal_name}")
            return {
                'status': 'success',
                'amount': amount,
                'de_payment_id': de_payment_id,
                'marketplace_tx_id': marketplace_tx_id,
                'message': f'Successfully redeemed {amount} FAME tokens'
            }
            
        except Exception as e:
            logger.error(f"Redeem tokens error: {str(e)}")
            db.session.rollback()
            return {'error': f'Internal error: {str(e)}', 'status': 'error'}
    
    
#    def redeem_tokens(self, user, organization, trading_account_id, amount, trading_account_owner=None):
#        """
#        Redeem FAME tokens back to Digital Euro
#        
#        Flow (SAFE):
#        1. Debit FAME tokens from trading account FIRST
#        2. If debit succeeds → Credit Digital Euro
#        3. If debit fails → No money movement, no problem
#        """
#        try:
#            # Validate amount is integer
#            amount = int(amount)
#            if amount <= 0:
#                return {'error': 'Amount must be positive', 'status': 'error'}
#            
#            # Get Digital Euro Account
#            de_account = DigitalEuroAccount.query.filter_by(
#                organization_id=organization.id,
#                is_active=True
#            ).first()
#            
#            if not de_account:
#                return {'error': 'No Digital Euro account linked', 'status': 'error'}
#            
#            # Check FAME balance            
#            balance_result = self.get_balance(trading_account_id)
#
#            if balance_result.get('status') != 'success':
#                return {'error': 'Could not retrieve FAME balance', 'status': 'error'}
#
#            fame_balance = balance_result.get('balance', 0.0)
#
#            if fame_balance < amount:
#                return {
#                    'error': f'Insufficient FAME balance. Available: {fame_balance:.2f}',
#                    'status': 'error'
#                }
#            
#            # STEP 1: Debit FAME tokens FIRST (safer order)
#            logger.info(f"Attempting to debit {amount} FAME tokens from {trading_account_id}")
#            
#            proxy_url = 'https://federation.fame-horizon.eu/api/proxy'
#            
#            debit_payload = {
#                'method': 'POST',
#                'url': f'{Config.FAME_API_BASE_URL}/v1.0/accounts/{trading_account_id}/debit',
#                'headers': {
#                    'Content-Type': 'application/json'
#                },
#                'data': {'amount': f"{amount:.2f}"}  # API expects "10.00" format
#            }
#            
#            try:
#                response = requests.post(
#                    proxy_url,
#                    json=debit_payload,
#                    auth=(Config.INTERNAL_API_USERNAME, Config.INTERNAL_API_PASSWORD),
#                    headers={'Content-Type': 'application/json'},
#                    timeout=30
#                )
#                response.raise_for_status()
#                result = response.json()
#                debit_result = result.get('data', result)
#            except requests.exceptions.RequestException as e:
#                logger.error(f"Marketplace debit failed: {str(e)}")
#                debit_result = {'error': str(e)}
#            
#            if isinstance(debit_result, dict) and 'error' in debit_result:
#                logger.error(f"Marketplace debit failed: {debit_result['error']}")
#                return {
#                    'error': f"Marketplace debit failed: {debit_result['error']}",
#                    'status': 'error'
#                }
#            
#            marketplace_tx_id = debit_result.get('marketplace_transaction_id')  
#            logger.info(f"FAME debit successful: {marketplace_tx_id}")
#            
#            # STEP 2: Credit Digital Euro (FAME already debited)
#            logger.info(f"Crediting {amount} EUR to Digital Euro account")
#            payment_result = digital_euro_service.p2p_payment(
#                debtor_id=MARKETPLACE_HOLDING_ID,
#                creditor_id=de_account.holding_id,
#                amount=amount
#            )
#            
#            if 'error' in payment_result:
#                # Record as failed transaction for audit
#                logger.critical(f"CRITICAL: DE payment failed but FAME already debited!")
#                logger.critical(f"Trading Account: {trading_account_id}, Amount: {amount}, TX: {marketplace_tx_id}")
#                
#                tx = WalletTransaction(
#                    organization_id=organization.id,
#                    user_id=user.id,
#                    trading_account_id=trading_account_id,
#                    trading_account_owner=trading_account_owner,  
#                    transaction_type='redeem',
#                    amount=amount,
#                    currency='EUR',
#                    de_payment_id=None,
#                    marketplace_tx_id=marketplace_tx_id,
#                    description=f'Redemption failed - FAME debited but DE credit failed',
#                    status='failed',
#                    error_message=f"Digital Euro payment error: {payment_result.get('error')}",
#                    completed_at=datetime.utcnow()
#                )
#                db.session.add(tx)
#                db.session.commit()
#                
#                return {
#                    'error': 'Digital Euro credit failed. Contact support for refund.',
#                    'status': 'error',
#                    'requires_manual_reconciliation': True,
#                    'marketplace_tx_id': marketplace_tx_id
#                }
#            
#            de_payment_id = payment_result.get('entry')
#            
#            # STEP 3: Record successful transaction
#            tx = WalletTransaction(
#                organization_id=organization.id,
#                user_id=user.id,
#                trading_account_id=trading_account_id,
#                trading_account_owner=trading_account_owner,
#                transaction_type='redeem',
#                amount=amount,
#                currency='EUR',
#                de_payment_id=de_payment_id,
#                marketplace_tx_id=marketplace_tx_id,
#                description=f'Redeemed {amount} FAME tokens',
#                status='completed',
#                completed_at=datetime.utcnow()
#            )
#            db.session.add(tx)
#            db.session.commit()
#            
#            logger.info(f"Redemption successful: {amount} FAME for org {organization.legal_name}")
#            return {
#                'status': 'success',
#                'amount': amount,
#                'de_payment_id': de_payment_id,
#                'marketplace_tx_id': marketplace_tx_id,
#                'message': f'Successfully redeemed {amount} FAME tokens'
#            }
#            
#        except Exception as e:
#            logger.error(f"Redeem tokens error: {str(e)}")
#            db.session.rollback()
#            return {'error': f'Internal error: {str(e)}', 'status': 'error'}

# Singleton instance
marketplace_wallet_service = MarketplaceWalletService()
