# app/services/member_service.py
import requests, logging
from flask import current_app
import json
from ..config import Config

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

class MemberService:
    @staticmethod
    def onboard_member(data):
        """
        Placeholder for calling external GOV API to onboard a new member
        
        This will call POST http://gov-internal:7008/gov/v1.0/members
        
        Parameters:
        data (dict): Member data in the format expected by the GOV API
        
        Returns:
        tuple: (member, error) where member is the created member object or None,
               and error is None or an error message
        """
        # TODO: Implement API call to GOV module when available
        try:
            # Placeholder for future implementation
            # response = requests.post('http://gov-internal:7008/gov/v1.0/members', json=data)
            # if response.status_code == 202:
            #     return response.json(), None
            # else:
            #     return None, f"Error {response.status_code}: {response.text}"
            return None, "GOV API not implemented yet"
        except Exception as e:
            return None, str(e)
    
    @staticmethod
    def list_members(filters=None):
        """
        Placeholder for calling external GOV API to list members
        
        This will call GET http://gov-internal:7008/gov/v1.0/members with query params
        
        Parameters:
        filters (dict): Optional filters like status, country, etc.
        
        Returns:
        tuple: (members_list, error) where members_list is a list of member objects or None,
               and error is None or an error message
        """
        # TODO: Implement API call to GOV module when available
        return None, "GOV API not implemented yet"
    
    @staticmethod
    def retrieve_member(pid):
        """
        Placeholder for calling external GOV API to retrieve a member
        
        This will call GET http://gov-internal:7008/gov/v1.0/members/{pid}
        
        Parameters:
        pid (str): The Provenance ID of the member to retrieve
        
        Returns:
        tuple: (member, error) where member is the member object or None,
               and error is None or an error message
        """
        # TODO: Implement API call to GOV module when available
        return None, "GOV API not implemented yet"
    
    @staticmethod
    def migrate_member(pid, new_org_data):
        """
        Placeholder for calling external GOV API to migrate a member
        
        This will call POST http://gov-internal:7008/gov/v1.0/members/{pid}/migrate
        
        Parameters:
        pid (str): The Provenance ID of the member to migrate
        new_org_data (dict): New organization data
        
        Returns:
        tuple: (new_member, error) where new_member is the migrated member object or None,
               and error is None or an error message
        """
        # TODO: Implement API call to GOV module when available
        return None, "GOV API not implemented yet"
    
    @staticmethod
    def update_member(pid, update_data):
        """
        Placeholder for calling external GOV API to update a member
        
        This will call PUT http://gov-internal:7008/gov/v1.0/members/{pid}
        
        Parameters:
        pid (str): The Provenance ID of the member to update
        update_data (dict): Data to update
        
        Returns:
        tuple: (success, error) where success is a boolean indicating success,
               and error is None or an error message
        """
        # TODO: Implement API call to GOV module when available
        return False, "GOV API not implemented yet"
    
    @staticmethod
    def offboard_member(pid):
        """
        Placeholder for calling external GOV API to offboard a member
        
        This will call POST http://gov-internal:7008/gov/v1.0/members/{pid}/offboard
        
        Parameters:
        pid (str): The Provenance ID of the member to offboard
        
        Returns:
        tuple: (success, error) where success is a boolean indicating success,
               and error is None or an error message
        """
        # TODO: Implement API call to GOV module when available
        return False, "GOV API not implemented yet"
    
    @staticmethod
    def check_member_exists(pid):
        """
        Placeholder for calling external GOV API to check if a member exists
        
        This will call GET http://gov-internal:7008/gov/v1.0/members/{pid}/check
        
        Parameters:
        pid (str): The Provenance ID of the member to check
        
        Returns:
        bool: True if member exists and is active, False otherwise
        """
        # TODO: Implement API call to GOV module when available
        return False
    
    @staticmethod
    def lookup_authority(did, affiliation=None):
        """
        Placeholder for calling external GOV API to check if a DID is an active authority
        
        This will call GET http://gov-internal:7008/gov/v1.0/authorities/{did}/check?a={affiliation}
        
        Parameters:
        did (str): The Decentralized Identifier to check
        affiliation (str, optional): The PID of the affiliating member
        
        Returns:
        dict or None: Authority information if valid, None otherwise
        """
        # TODO: Implement API call to GOV module when available
        return None
    
    @staticmethod
    def get_active_authorities():
        """
        Retrieves the list of active authorities from the internal GOV service
        
        Returns:
            list: List of dictionaries containing information about active authorities
        """
        try:
            import requests

            # Call to the internal API
            response = requests.get('http://gov-internal:7008/gov/v1.0/active-authorities', timeout=5) #INTERNAL
            # response = requests.get('https://gov.fame-horizon.eu/dev/v1.0/active-authorities', timeout=5) #EXTERNAL (BACKDOOR)         
            
            if response.status_code == 200:
                return response.json()
            else:
                print(f"Error in requesting active authorities: {response.status_code}")
                return []
                
        except Exception as e:
            print(f"Exception during the request for active authorities: {str(e)}")
            return []
    
    @staticmethod
    def get_active_members():
        """
        Placeholder for calling external GOV API to get all active members
        
        This will call GET http://gov-internal:7008/gov/v1.0/active-members
        
        Returns:
        list: List of active members
        """
        # TODO: Implement API call to GOV module when available
        return []
    
    # Methods for user management
    @staticmethod
    def create_user(user_data):
        """
        Creates a new user via the external GOV API
        
        Parameters:
        user_data (dict): User data in the format expected by the API
        
        Returns:
        tuple: (response_data, error) where response_data is the API response or None,
               and error is None or an error message
        """
        try:
            response = requests.post(f"{Config.FAME_API_BASE_URL}/v1.0/users", json=user_data)
            
            if response.status_code == 202:  # Accepted
                return response.json(), None
            else:
                return None, f"Error {response.status_code}: {response.text}"
        except Exception as e:
            return None, str(e)
    
    @staticmethod
    def list_users(filters=None):
        """
        Retrieves the list of users via the external GOV API
        
        Parameters:
        filters (dict): Optional filters like status, country, etc.
        
        Returns:
        tuple: (users_list, error) where users_list is a list of users or None,
               and error is None or an error message
        """
        try:
            # Build query parameters
            params = {}
            if filters:
                if 'sta' in filters:
                    params['sta'] = filters['sta']
                if 'rol' in filters:
                    params['rol'] = filters['rol']
                if 'cty' in filters:
                    params['cnt'] = filters['cty']  # Mapping to cnt for API
                if 'aff' in filters:
                    params['aff'] = filters['aff']
                if 'email' in filters:
                    params['email'] = filters['email']
                if 'lname' in filters:
                    params['lname'] = filters['lname']
                if 'iid' in filters:
                    params['iid'] = filters['iid']
            
            response = requests.get(f"{Config.FAME_API_BASE_URL}/v1.0/users", params=params)
            
            if response.status_code == 200:
                return response.json(), None
            else:
                return None, f"Error {response.status_code}: {response.text}"
        except Exception as e:
            return None, str(e)
    
    @staticmethod
    def get_user_by_id(uid):
        """
        Retrieves the details of a specific user via the external GOV API
        
        Parameters:
        uid (str): The ID of the user to retrieve
        
        Returns:
        tuple: (user, error) where user is the user object or None,
               and error is None or an error message
        """
        try:
            response = requests.get(f"{Config.FAME_API_BASE_URL}/v1.0/users/{uid}")
            
            if response.status_code == 200:
                return response.json(), None
            else:
                return None, f"Error {response.status_code}: {response.text}"
        except Exception as e:
            return None, str(e)
    
    @staticmethod
    def reinvite_user(uid):
        """
        Restarts the onboarding process for a user via the external GOV API
        
        Parameters:
        uid (str): The ID of the user to reinvite
        
        Returns:
        tuple: (success, error) where success is a boolean indicating success,
            and error is None or an error message
        """
                
        try:
            # Log della richiesta in partenza
            logger.info(f"Initiating reinvite for user UID: {uid}")
            logger.debug(f"API Endpoint: {Config.FAME_API_BASE_URL}/v1.0/users/{uid}/reinvite")
            
            response = requests.post(
                f"{Config.FAME_API_BASE_URL}/v1.0/users/{uid}/reinvite",
                timeout=10
            )
            
            # Log dettagliato della risposta
            logger.debug(f"API Response - Status: {response.status_code}")
            logger.debug(f"API Response - Headers: {response.headers}")
            logger.debug(f"API Response - Body: {response.text[:200]}...")  # Log solo i primi 200 caratteri
            
            if response.status_code in [202, 204]:
                logger.info(f"Successfully reinvited user UID: {uid}")
                return True, None
            else:
                error_msg = f"API Error: {response.status_code} - {response.text}"
                logger.error(f"Failed to reinvite user UID: {uid}. {error_msg}")
                return False, error_msg
                
        except requests.exceptions.Timeout:
            error_msg = "API timeout after 10 seconds"
            logger.error(f"Timeout while reinviting user UID: {uid}. {error_msg}")
            return False, error_msg
            
        except requests.exceptions.RequestException as e:
            error_msg = f"Request failed: {str(e)}"
            logger.error(f"Network error while reinviting user UID: {uid}. {error_msg}", exc_info=True)
            return False, error_msg
            
        except Exception as e:
            error_msg = f"Unexpected error: {str(e)}"
            logger.error(f"Unexpected error while reinviting user UID: {uid}. {error_msg}", exc_info=True)
            return False, error_msg
    
    @staticmethod
    def offboard_user(uid):
        """
        Disables a user via the external GOV API
        
        Parameters:
        uid (str): The ID of the user to disable
        
        Returns:
        tuple: (success, error) where success is a boolean indicating success,
               and error is None or an error message
        """
        try:
            response = requests.post(f"{Config.FAME_API_BASE_URL}/v1.0/users/{uid}/offboard")
            
            if response.status_code in [202, 204]:
                return True, None
            else:
                return False, f"Error {response.status_code}: {response.text}"
        except Exception as e:
            return False, str(e)
    
    @staticmethod
    def delete_user(uid):
        """
        Deletes a user via the external GOV API
        
        Parameters:
        uid (str): The ID of the user to delete
        
        Returns:
        tuple: (success, error) where success is a boolean indicating success,
               and error is None or an error message
        """
        try:
            response = requests.delete(f"{Config.FAME_API_BASE_URL}/v1.0/users/{uid}")
            
            if response.status_code in [202, 204]:
                return True, None
            else:
                return False, f"Error {response.status_code}: {response.text}"
        except Exception as e:
            return False, str(e)