# app/services/organization_service.py
import logging
from datetime import datetime, timedelta
import re
from ..models import Organization, User, PostalAddress, Vote
from .. import db
from ..models.organization import FAME_ROA_PID
import requests
from ..config import Config
from flask import current_app
from ..utils.logo_utils import save_logo

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

class OrganizationService:
        
    @staticmethod
    def create_organization(data):
        """
        Onboard a new member to the federation
        
        Expected data format:
        {
            "org": {
                // LegalEntity object
                "legalName": "...",
                "legalAddress": {
                    // PostalAddress object
                    "addressLine1": "...",
                    "city": "...",
                    "postCode": "...",
                    "country": "..."
                },
                // Legal identifiers (at least one required)
                "lei": "...", // optional
                "duns": "...", // optional
                "bic": "...", // optional
                "tin": "..." // optional
            },
            "type": "FPO|NPO|RES|EDU|LEO|PUB|UKN",
            "rep": {
                // Representative contact info
                "firstName": "...",
                "lastName": "...",
                "email": "...",
                "phone": "..." // optional
            },
            "oa": "..." // PID of delegate OA or DID for autonomous OA
            "logo": file object (optional),
            "website_url": "..." (optional)
        }
        """
        try:
            # Validate organization data
            org_data = data.get('org', {})
            if not org_data:
                logger.warning("Organization data is missing")
                return None, "Organization data is required"
                
            # Check if legal name is provided
            legal_name = org_data.get('legalName')
            if not legal_name:
                logger.warning("Legal name is missing")
                return None, "Legal name is required"
                
            if len(legal_name) < 1 or len(legal_name) > 160:
                logger.warning("Invalid legal name length")
                return None, "Legal name must be between 1 and 160 characters"
                
            # Basic check for invalid characters
            if re.search(r'[\n\r\t]', legal_name):  # No newlines, tabs
                return None, "Legal name cannot contain newlines or tabs"
                
            # Check legal address
            address_data = org_data.get('legalAddress', {})
            if not address_data:
                logger.warning("Legal address is missing")
                return None, "Legal address is required"
                
            # Create postal address object
            postal_address = PostalAddress.from_dict(address_data)          
                
            # Validate organization type
            org_type = data.get('type')
            valid_types = ['FPO', 'NPO', 'RES', 'EDU', 'LEO', 'PUB', 'UKN', 'EUP']
            if org_type not in valid_types:
                logger.warning(f"Invalid organization type: {org_type}")
                return None, f"Organization type must be one of: {', '.join(valid_types)}"
                
            # Validate representative data
            rep_data = data.get('rep', {})
            if not rep_data:
                logger.warning("Representative data is missing")
                return None, "Representative data is required"
                
            # Check for required representative fields
            for field in ['firstName', 'lastName', 'email']:
                if not rep_data.get(field):
                    logger.warning(f"Missing representative field: {field}")
                    return None, f"Representative {field} is required"
                    
            # Find or create representative user
            rep_email = rep_data.get('email')
            representative = User.query.filter_by(email=rep_email).first()
            
            # Process OA data and determine oa_status
            oa_value = data.get('oa')
            oa_status = data.get('oa_status')
            did = None
            delegate_authority_pid = None
            
            if oa_status == 'OA':
                # This is an autonomous OA with its own DID
                did = oa_value
            else:
                delegate_authority_pid = oa_value           
            
            # Ensure at least one identifier is provided
            lei = org_data.get('lei')
            duns = org_data.get('duns')
            bic = org_data.get('bic')
            tin = org_data.get('tin')

            if not any([lei, duns, bic, tin]):
                logger.warning("No legal identifier provided")
                return None, "At least one legal identifier (LEI, DUNS, BIC, TIN/VAT or GA) is required"

            # Identifier validations
            if lei:
                # LEI validation: 20 alphanumeric characters
                if not re.match(r'^[A-Za-z0-9]{20}$', lei):
                    logger.warning("Invalid LEI format")
                    return None, "LEI must be 20 alphanumeric characters"

            if duns:
                # DUNS validation: 9 numeric characters
                if not re.match(r'^[0-9]{9}$', duns):
                    logger.warning("Invalid DUNS format")
                    return None, "DUNS must be 9 numeric characters"

            if bic:
                # BIC validation: 8 or 11 alphanumeric characters
                if not re.match(r'^[A-Za-z0-9]{8}$|^[A-Za-z0-9]{11}$', bic):
                    logger.warning("Invalid BIC format")
                    return None, "BIC must be 8 or 11 alphanumeric characters"

            if tin:
                # TIN/VAT validation: 1-40 alphanumeric characters and spaces, not only spaces
                if not re.match(r'^(?=.*[A-Za-z0-9])[A-Za-z0-9 ]+$', tin) or len(tin) > 40:
                    logger.warning("Invalid TIN/VAT format")
                    return None, "TIN/VAT must contain only alphanumeric characters and spaces, and cannot consist of spaces only"
            
            # NEW: Handle logo upload (optional)
            logo_filename = None
            if 'logo' in data and data['logo']:
                logo_file = data['logo']
                logo_filename, error = save_logo(logo_file)
                if error:
                    logger.warning(f"Logo upload failed: {error}")
                    # Don't fail the registration, just log the warning
                    logo_filename = None
            
            # NEW: Handle website URL (optional)
            website_url = data.get('website_url', '').strip()
            if website_url:
                # Basic validation
                if not website_url.startswith(('http://', 'https://')):
                    logger.warning(f"Invalid website URL format: {website_url}")
                    website_url = None
            else:
                website_url = None
            
            # Create new organization/member
            new_organization = Organization(
                legal_name=legal_name,
                org_type=org_type,
                did=did,
                oa_status=oa_status,
                delegate_authority_pid=delegate_authority_pid,
                representative_id=representative.id,
                status='created',
                logo=logo_filename,  # NEW
                website_url=website_url  # NEW
            )
            
            # Set legal identifiers if provided
            for id_type in ['lei', 'duns', 'bic', 'tin']:
                if id_type in org_data:
                    setattr(new_organization, id_type, org_data.get(id_type))
            
            db.session.add(new_organization)
            db.session.flush()  # Ensure new org has an ID
            logger.info(f"New organization created with temporary ID: {new_organization.id}")
            
            # Create and associate postal address
            postal_address.organization_id = new_organization.id
            db.session.add(postal_address)
            
            # Set the voting expiration
            # new_organization.voting_ends_at = datetime.utcnow() + timedelta(minutes=Config.VOTING_DURATION_MINUTES)
            
            # Commit to database
            db.session.commit()
            
            logger.info(f"Organization {legal_name} created successfully with ID {new_organization.id}")
            if logo_filename:
                logger.info(f"Logo uploaded: {logo_filename}")
            if website_url:
                logger.info(f"Website: {website_url}")
            
            logger.info(f"Organization '{legal_name}' successfully onboarded")
            
            return new_organization, None
            
        except Exception as e:
            db.session.rollback()
            logger.error(f"Failed to create organization: {str(e)}")
            return None, str(e)
    
    @staticmethod
    def get_pending_organizations():
        """Get all pending organizations"""
        return Organization.query.filter_by(status='pending').all()

    @staticmethod
    def get_organization_by_id(org_id):
        """Get organization by ID"""
        return Organization.query.get(org_id)
    
    @staticmethod
    def get_postal_address_by_id(postal_address_id):
        """Get postal address by ID"""
        return PostalAddress.query.get(postal_address_id)
        
    @staticmethod
    def get_organizations(status=None, industry=None):
        """Get organizations with optional filters"""
        query = Organization.query
        
        if status:
            query = query.filter_by(status=status)
        
        if industry:
            # Adaptations may be needed depending on how the industry is stored
            query = query.filter_by(industry=industry)
        
        return query.all()

    @staticmethod
    def update_organization_status(org_id, new_status):
        """Update organization status"""
        try:
            logger.info(f"Updating status of organization ID {org_id} to '{new_status}'")
            org = Organization.query.get(org_id)
            if not org:
                logger.warning(f"Organization with ID {org_id} not found")
                return None, "Organization not found"
                
            org.status = new_status
            db.session.commit()
            logger.info(f"Status of organization ID {org_id} updated to '{new_status}'")
            return org, None
            
        except Exception as e:
            db.session.rollback()
            logger.error(f"Error updating organization status: {str(e)}")
            return None, str(e)
    
    @staticmethod
    def check_member_exists(pid):
        """Check if a member exists and is not offboarded"""
        member = Organization.query.filter_by(pid=pid).first()
        if not member or member.offboarded_at:
            return False
        return True

    @staticmethod
    def get_member_by_pid(pid):
        """Get a member by their PID"""
        return Organization.query.filter_by(pid=pid).first()
    
    @staticmethod
    def get_active_authorities():
        """Get list of all active onboarding authorities"""
        # First get the FAME ROA if it exists
        fame_roa = Organization.query.filter_by(
            pid=FAME_ROA_PID,
            offboarded_at=None
        ).first()
        
        # Get all autonomous OAs
        authorities = Organization.query.filter_by(
            oa_status='OA',
            offboarded_at=None
        ).order_by(Organization.legal_name).all()
        
        # Prepare result starting with FAME ROA
        result = []
        
        # Add FAME ROA first with special formatting
        if fame_roa:
            result.append({
                "did": fame_roa.did,
                "pid": fame_roa.pid,
                "name": "FAME Root Authority (ROA)"
            })
        else:
            # No ROA found, add a placeholder
            result.append({
                "did": None,
                "pid": FAME_ROA_PID,
                "name": "FAME Root Authority (ROA)"
            })
        
        # Add all autonomous OAs
        for auth in authorities:
            result.append({
                "did": auth.did,
                "pid": auth.pid,
                "name": auth.legal_name
            })
            
        return result
    
    @staticmethod
    def get_active_members():
        """Get list of all active members regardless of OA status"""
        members = Organization.query.filter_by(
            offboarded_at=None
        ).order_by(Organization.legal_name).all()
        
        # Prepare result
        result = []
        for member in members:
            result.append({
                "pid": member.pid,
                "name": member.legal_name
            })
            
        return result
    
    @staticmethod
    def offboard_member(pid):
        """Offboard a member"""
        try:
            logger.info(f"Attempting to offboard member with PID {pid}")
            member = Organization.query.filter_by(pid=pid).first()
            if not member:
                logger.warning(f"Member with PID {pid} not found")
                return False, "Member not found"
                
            member.offboarded_at = datetime.utcnow()
            db.session.commit()
            
            logger.info(f"Member with PID {pid} offboarded successfully")
            # Additional offboarding steps like revoking DIDs, etc. could be added here
            
            return True, None
            
        except Exception as e:
            db.session.rollback()
            logger.error(f"Error during offboarding of member with PID {pid}: {str(e)}")
            return False, str(e)
        
    @staticmethod
    def register_approved_member(organization):
        """
        Registers an approved member with the external governance API
        after a successful vote.
        
        Args:
            organization: The Organization object that has been approved
            
        Returns:
            tuple: (success, error_message)
        """
        try:
            # Prepare data according to the format required by the API
            # Create a LegalEntity object
            logger.info(f"Registering approved organization '{organization.legal_name}' with external API")
            legal_entity = {
                "legalName": organization.legal_name
            }
            
            # Add the legal address if present
            if organization.legal_address:
                legal_entity["legalAddress"] = {
                    "addressLine1": organization.legal_address.address_line1,
                    "city": organization.legal_address.city,
                    "postCode": organization.legal_address.post_code,
                    "country": organization.legal_address.country
                }
            
            # Add legal identifiers if present
            for id_type in ['lei', 'duns', 'bic', 'tin']:
                if getattr(organization, id_type):
                    legal_entity[id_type] = getattr(organization, id_type)
            
            # Get representative data
            representative = organization.representative
            rep_data = {
                "firstName": representative.first_name,
                "lastName": representative.last_name,
                "email": representative.email
            }
            
            if representative.phone:
                rep_data["phone"] = representative.phone
            
            # Mask 'EUP' as 'NPO' for external API
            org_type_raw = getattr(organization, 'org_type', None)
            if org_type_raw == 'EUP':
                type_value = 'NPO'
            else:
                type_value = org_type_raw

            # Prepare the complete payload
            payload = {
                "org": legal_entity,
                "type": type_value,
                "rep": rep_data
            }
            logger.info(f"Payload to be sent to external API: {payload}")
            
            # Add OA information based on the status
            if organization.oa_status == 'OA' and organization.did:
                payload["oa"] = organization.did
            elif organization.delegate_authority_pid:
                payload["oa"] = organization.delegate_authority_pid
            
            # Make the API call
            response = requests.post(
                f"{Config.FAME_API_BASE_URL}/v1.0/members",
                json=payload,
                headers={
                    "Content-Type": "application/json",
                    "Authorization": f"Bearer {Config.FAME_API_KEY}"  # If required
                },
                timeout=10  # timeout in seconds
            )
            
            # Handle the response
            if response.status_code == 202:
                data = response.json()
                external_pid = data.get("id")

                if external_pid:
                    logger.info(f"Received external PID from API: {external_pid}")  # Debug
                    organization.pid = external_pid  # Update the PID in the database
                    db.session.commit()
                else:
                    logger.warning("No PID received from external API")
                
                return True, None
            else:
                # Error: log the response and return the error
                error_msg = f"API error: {response.status_code} - {response.text}"
                logger.error(error_msg)
                return False, error_msg
                
        except requests.RequestException as e:
            logger.error(f"Request error while registering member: {str(e)}")
            return False, f"Request error: {str(e)}"
        except Exception as e:
            logger.error(f"Unexpected error while registering member: {str(e)}")
            return False, f"Error: {str(e)}"
    
    @staticmethod
    def check_expired_votes():
        """Check and close expired votings"""
        pending_orgs = Organization.query.filter_by(status='pending')\
            .filter(Organization.voting_ends_at != None).all()
        now = datetime.utcnow()
        logger.info(f"Checking for expired votes at {now.isoformat()}")
        
        for org in pending_orgs:
            if now > org.voting_ends_at:
                # Save the previous status
                logger.info(f"Closing voting for organization '{org.legal_name}'")
                previous_status = org.status
                
                # Close the voting
                org.close_voting()
                
                # If status changed, send notification email to the representative
                if previous_status != org.status:
                    # Get the representative
                    representative = org.representative
                    if representative:
                        # Send notification email
                        OrganizationService.send_voting_result_notification(org, representative)
                
                # If the organization has been approved, register it with the external API
                if previous_status == 'pending' and org.status == 'approved':
                    success, error = OrganizationService.register_approved_member(org)
                    if not success:
                        # Log the error but do not block the process
                        logger.error(f"Failed to register approved member '{org.legal_name}': {error}")
        
        try:
            db.session.commit()
            logger.info("Expired vote check completed and changes committed")
            return True, None
        except Exception as e:
            db.session.rollback()
            logger.error(f"Error committing expired vote changes: {str(e)}")
            return False, str(e)
    
    @staticmethod
    def send_voting_result_notification(organization, representative):
        """
        Send a notification email to the organization representative
        about the voting result
        
        Args:
            organization: The Organization object
            representative: The User object (representative)
        """
        from .email_service import send_voting_result_email
        
        try:
            logger.info(f"Sending voting result notification to {representative.email} for {organization.legal_name}")
            
            # Call the email service function
            success = send_voting_result_email(
                representative, 
                organization.legal_name, 
                organization.status
            )
            
            if success:
                logger.info(f"Successfully sent voting result notification to {representative.email}")
            else:
                logger.error(f"Failed to send voting result notification to {representative.email}")
                
        except Exception as e:
            logger.error(f"Exception sending voting result notification: {str(e)}")
    