# app/services/vote_service.py
import logging
from datetime import datetime
from ..models import Vote, Organization
from .. import db

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

class VoteService:
    @staticmethod
    def submit_vote(organization_id, voter_id, vote_value):
        """Submit a vote for an organization"""
        try:
            logger.info(f"Submitting vote for organization_id={organization_id} by voter_id={voter_id}")

            # Validate organization status
            org = Organization.query.get(organization_id)
            if not org or org.status != 'pending':
                logger.warning(f"Vote rejected: Invalid organization or voting closed for organization_id={organization_id}")
                return None, "Invalid organization or voting closed"

            # Check if voting period has expired
            if datetime.utcnow() > org.voting_ends_at:
                logger.info(f"Voting period expired for organization_id={organization_id}")
                org.close_voting()
                db.session.commit()
                return None, "Voting period has expired"

            # Check for existing vote
            existing_vote = Vote.query.filter_by(
                organization_id=organization_id,
                voter_id=voter_id
            ).first()

            if existing_vote:
                logger.info(f"Updating existing vote by voter_id={voter_id} for organization_id={organization_id}")
                existing_vote.vote = vote_value
            else:
                logger.info(f"Creating new vote by voter_id={voter_id} for organization_id={organization_id}")
                vote = Vote(
                    organization_id=organization_id,
                    voter_id=voter_id,
                    vote=vote_value
                )
                db.session.add(vote)

            db.session.commit()
            logger.info(f"Vote submitted successfully for voter_id={voter_id} on organization_id={organization_id}")
            return existing_vote or vote, None

        except Exception as e:
            logger.error(f"Error while submitting vote: {e}")
            db.session.rollback()
            return None, str(e)

    @staticmethod
    def get_user_votes(user_id):
        """Get all votes by a specific user"""
        logger.info(f"Fetching all votes by user_id={user_id}")
        return Vote.query.filter_by(voter_id=user_id).all()

    @staticmethod
    def get_organization_votes(org_id):
        """Get all votes for a specific organization"""
        logger.info(f"Fetching all votes for organization_id={org_id}")
        return Vote.query.filter_by(organization_id=org_id).all()

    @staticmethod
    def get_vote_counts(org_id):
        """Get vote counts for an organization"""
        logger.info(f"Counting votes for organization_id={org_id}")
        yes_votes = Vote.query.filter_by(
            organization_id=org_id, 
            vote='yes'
        ).count()
        no_votes = Vote.query.filter_by(
            organization_id=org_id, 
            vote='no'
        ).count()
        logger.info(f"Vote count for organization_id={org_id}: YES={yes_votes}, NO={no_votes}")
        return yes_votes, no_votes
