# app/utils/logo_utils.py
"""
Utilities for handling organization logo uploads and validation
"""

import os
import uuid
from werkzeug.utils import secure_filename
from flask import current_app

def allowed_logo_file(filename):
    """Check if file extension is allowed for logos"""
    if '.' not in filename:
        return False
    
    ext = filename.rsplit('.', 1)[1].lower()
    return ext in current_app.config.get('ALLOWED_LOGO_EXTENSIONS', {'png', 'jpg', 'jpeg', 'svg'})

def save_logo(file, old_logo=None):
    """
    Save a logo file and return the filename
    
    Args:
        file: FileStorage object from request.files
        old_logo: Optional filename of old logo to delete
        
    Returns:
        tuple: (filename, error_message)
        filename is None if there was an error
    """
    
    if not file:
        return None, "No file provided"
    
    if file.filename == '':
        return None, "Empty filename"
    
    # Check file type
    if not allowed_logo_file(file.filename):
        allowed = ', '.join(current_app.config.get('ALLOWED_LOGO_EXTENSIONS', {'png', 'jpg', 'jpeg', 'svg'}))
        return None, f"Invalid file type. Allowed types: {allowed}"
    
    # Check file size
    file.seek(0, os.SEEK_END)
    file_size = file.tell()
    file.seek(0)
    
    max_size = current_app.config.get('MAX_LOGO_SIZE', 2 * 1024 * 1024)
    if file_size > max_size:
        max_mb = max_size / (1024 * 1024)
        return None, f"File too large. Maximum size: {max_mb}MB"
    
    try:
        # Generate secure filename with UUID
        ext = file.filename.rsplit('.', 1)[1].lower()
        filename = f"{uuid.uuid4().hex}.{ext}"
        
        # Get logo folder path
        logo_folder = current_app.config.get('LOGO_FOLDER')
        
        # Ensure directory exists
        os.makedirs(logo_folder, exist_ok=True)
        
        # Save file
        filepath = os.path.join(logo_folder, filename)
        file.save(filepath)
        
        # Delete old logo if exists
        if old_logo:
            old_filepath = os.path.join(logo_folder, old_logo)
            if os.path.exists(old_filepath):
                try:
                    os.remove(old_filepath)
                except Exception as e:
                    # Log but don't fail if we can't delete old file
                    print(f"Warning: Could not delete old logo {old_logo}: {e}")
        
        return filename, None
        
    except Exception as e:
        return None, f"Error saving file: {str(e)}"

def delete_logo(filename):
    """
    Delete a logo file
    
    Args:
        filename: Name of the logo file to delete
        
    Returns:
        bool: True if successful, False otherwise
    """
    if not filename:
        return False
    
    try:
        logo_folder = current_app.config.get('LOGO_FOLDER')
        filepath = os.path.join(logo_folder, filename)
        
        if os.path.exists(filepath):
            os.remove(filepath)
            return True
        
        return False
        
    except Exception as e:
        print(f"Error deleting logo {filename}: {e}")
        return False

def get_logo_url(filename):
    """
    Get the URL for a logo file
    
    Args:
        filename: Name of the logo file
        
    Returns:
        str: URL to access the logo, or None if no filename
    """
    if not filename:
        return None
    
    from flask import url_for
    return url_for('main.serve_logo', filename=filename)