# app/utils/validators.py
import re, phonenumbers
from werkzeug.utils import secure_filename
from flask import current_app

def validate_password(password):
    """
    Validate password strength
    Returns (bool, str) tuple - (is_valid, error_message)
    """
    if len(password) < 8:
        return False, "Password must be at least 8 characters long"
        
    if not re.search(r"[A-Z]", password):
        return False, "Password must contain at least one uppercase letter"
        
    if not re.search(r"[a-z]", password):
        return False, "Password must contain at least one lowercase letter"
        
    if not re.search(r"\d", password):
        return False, "Password must contain at least one number"
        
    if not re.search(r"[!@#$%^&*(),.?\":{}|<>]", password):
        return False, "Password must contain at least one special character"
        
    return True, None

def validate_email(email):
    """
    Validate email format
    Returns (bool, str) tuple - (is_valid, error_message)
    """
    pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
    if not re.match(pattern, email):
        return False, "Invalid email format"
    return True, None

def allowed_file(filename):
    """Check if file extension is allowed"""
    return '.' in filename and \
           filename.rsplit('.', 1)[1].lower() in current_app.config['ALLOWED_EXTENSIONS']
           
def validate_phone_number(phone_number):
    """
    Validate an international phone number
    
    Returns:
        tuple: (is_valid, error_message)
    """
    if not phone_number:
        return True, None  # The number is optional, so if it's empty it's fine
    
    # Check maximum length (40 characters as per HALF_LINER_MAXL)
    if len(phone_number) > 40:
        return False, "The phone number cannot exceed 40 characters"
    
    # Check format
    try:
        # Must start with + followed by the country code
        if not phone_number.startswith('+'):
            return False, "The phone number must start with '+' followed by the country code"
        
        # Parsing with phonenumbers
        parsed_number = phonenumbers.parse(phone_number, None)
        
        print(parsed_number)
        print(phone_number)
        
        # Check validity
        if not phonenumbers.is_valid_number(parsed_number):
            return False, "Invalid phone number format"
            
        return True, None
    except phonenumbers.phonenumberutil.NumberParseException:
        return False, "Unable to interpret the phone number, check the format"