# migrations/add_logo_website.py
"""
Migration script to add logo and website_url columns to organizations table
This script is SAFE and will NOT delete any existing data
"""

import sqlite3
import os
from datetime import datetime

def backup_database(db_path):
    """Create a backup before migration"""
    backup_path = f"{db_path}.backup_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
    
    # Connect to source database
    source = sqlite3.connect(db_path)
    
    # Connect to backup database
    backup = sqlite3.connect(backup_path)
    
    # Copy database
    source.backup(backup)
    
    source.close()
    backup.close()
    
    print(f"✓ Backup created: {backup_path}")
    return backup_path

def check_columns_exist(cursor, table_name, columns):
    """Check if columns already exist in table"""
    cursor.execute(f"PRAGMA table_info({table_name})")
    existing_columns = [row[1] for row in cursor.fetchall()]
    
    results = {}
    for col in columns:
        results[col] = col in existing_columns
    
    return results

def migrate_database(db_path):
    """Add logo and website_url columns to organizations table"""
    
    print("\n" + "="*60)
    print("FAME FEDERATION - Database Migration")
    print("Adding 'logo' and 'website_url' columns")
    print("="*60 + "\n")
    
    # Step 1: Backup
    print("Step 1: Creating backup...")
    backup_path = backup_database(db_path)
    
    # Step 2: Connect to database
    print("\nStep 2: Connecting to database...")
    conn = sqlite3.connect(db_path)
    cursor = conn.cursor()
    
    # Step 3: Check current state
    print("\nStep 3: Checking current database state...")
    
    # Count existing organizations
    cursor.execute("SELECT COUNT(*) FROM organizations")
    org_count = cursor.fetchone()[0]
    print(f"   Found {org_count} existing organizations")
    
    # Check if columns already exist
    columns_to_add = ['logo', 'website_url']
    column_status = check_columns_exist(cursor, 'organizations', columns_to_add)
    
    for col, exists in column_status.items():
        if exists:
            print(f"   ⚠ Column '{col}' already exists - skipping")
        else:
            print(f"   ✓ Column '{col}' needs to be added")
    
    # Step 4: Add new columns
    print("\nStep 4: Adding new columns...")
    
    try:
        # Add logo column if it doesn't exist
        if not column_status['logo']:
            cursor.execute("""
                ALTER TABLE organizations 
                ADD COLUMN logo VARCHAR(255) NULL
            """)
            print("   ✓ Added 'logo' column")
        
        # Add website_url column if it doesn't exist
        if not column_status['website_url']:
            cursor.execute("""
                ALTER TABLE organizations 
                ADD COLUMN website_url VARCHAR(500) NULL
            """)
            print("   ✓ Added 'website_url' column")
        
        conn.commit()
        
        # Step 5: Verify migration
        print("\nStep 5: Verifying migration...")
        
        # Check that all data is still there
        cursor.execute("SELECT COUNT(*) FROM organizations")
        new_org_count = cursor.fetchone()[0]
        
        if new_org_count != org_count:
            raise Exception(f"Data loss detected! Original: {org_count}, Current: {new_org_count}")
        
        print(f"   ✓ All {new_org_count} organizations preserved")
        
        # Check new columns
        cursor.execute("PRAGMA table_info(organizations)")
        all_columns = [row[1] for row in cursor.fetchall()]
        
        if 'logo' in all_columns and 'website_url' in all_columns:
            print("   ✓ New columns successfully added")
        else:
            raise Exception("New columns not found after migration")
        
        # Sample data check
        cursor.execute("""
            SELECT id, legal_name, logo, website_url 
            FROM organizations 
            LIMIT 3
        """)
        
        print("\n   Sample data (first 3 organizations):")
        for row in cursor.fetchall():
            print(f"      ID: {row[0]}, Name: {row[1]}, Logo: {row[2]}, Website: {row[3]}")
        
        print("\n" + "="*60)
        print("✓ MIGRATION COMPLETED SUCCESSFULLY!")
        print("="*60)
        print(f"\nBackup saved at: {backup_path}")
        print("\nYou can safely proceed with updating your code.")
        print("\nIf anything goes wrong, restore with:")
        print(f"   cp {backup_path} {db_path}")
        print("="*60 + "\n")
        
    except Exception as e:
        print(f"\n✗ ERROR during migration: {e}")
        print(f"\nRestoring from backup: {backup_path}")
        
        conn.close()
        
        # Restore backup
        os.replace(backup_path, db_path)
        
        print("✓ Database restored to original state")
        print("No data was lost.")
        
        return False
    
    finally:
        conn.close()
    
    return True

def rollback_migration(db_path, backup_path):
    """Rollback migration by restoring from backup"""
    
    print("\n" + "="*60)
    print("ROLLBACK - Restoring database from backup")
    print("="*60 + "\n")
    
    if not os.path.exists(backup_path):
        print(f"✗ Backup file not found: {backup_path}")
        return False
    
    try:
        # Replace current database with backup
        os.replace(backup_path, db_path)
        print("✓ Database successfully restored from backup")
        print("✓ All original data preserved")
        return True
    except Exception as e:
        print(f"✗ Error during rollback: {e}")
        return False

if __name__ == "__main__":
    # Path to database
    db_path = "data/db/database.db"
    
    if not os.path.exists(db_path):
        print(f"✗ Database not found at: {db_path}")
        print("Please run this script from the project root directory")
        exit(1)
    
    # Run migration
    success = migrate_database(db_path)
    
    if not success:
        print("\n⚠ Migration failed but database was restored")
        print("Please check the error messages above")
        exit(1)