# FAME Operational Governance API

The **Operational Governance API** (GOV) is a core backend module of the **FAME Marketplace platform**, responsible for managing organizational governance, user onboarding, credential issuance, and administrative operations. It orchestrates the lifecycle of participants in the FAME ecosystem and manages the issuance of Verifiable Credentials (VCs).

## Overview

This service provides critical governance functions for the FAME platform, enabling:

- **Organization Management**: Registration and lifecycle management of organizations participating in FAME
- **User Onboarding**: Streamlined onboarding process for new users and organizations
- **Credential Issuance**: Integration with identity providers for issuing Verifiable Credentials
- **Request Management**: Tracking and managing onboarding and access requests
- **Email Notifications**: Automated communication with users and administrators
- **Integration Orchestration**: Coordination between PT, TM, and identity services

The API is built with **NestJS** and TypeScript, offering both public and internal endpoints for integration with other FAME platform components.

## Quick Start

There are three primary ways to deploy this service:

### 1. Local Development (Node.js)
For development and testing purposes:
```bash
npm install
npm run build
# Set required environment variables (see Environment Configuration)
export MYSQL_PASSWORD="..."
export MONGO_URI="..."
export GMAIL_PASS="..."
node dist/src/main.js
```
→ Access at `http://localhost:7006/swagger` (public) or `http://localhost:7008/swagger` (internal)

### 2. Docker Container
For containerized deployment:
```bash
npm run build
npm run docker:build
npm run docker:push
docker run -d -p 7006:7006 --env-file .env -e MYSQL_PASSWORD='...' harbor.gftinnovation.eu/fame/framework/gov:3.0
```
→ Suitable for standalone deployment with manual configuration

### 3. Kubernetes (Production)
For production deployment on Kubernetes clusters:
```bash
# Create ConfigMap and Secrets
kubectl apply -f gov-service.yaml
```
→ Recommended for production - separates configuration from secrets, supports dual deployment (public + internal)

**Important Security Note**: The `.env` file in the repository does **not** contain sensitive values. You must set the following environment variables separately:
- `MYSQL_PASSWORD` - MySQL/MariaDB database password
- `MONGO_URI` - MongoDB connection string with credentials
- `GMAIL_PASS` - Gmail application password for sending emails

## Table of Contents

- [Overview](#overview)
- [Architecture](#architecture)
- [Prerequisites](#prerequisites)
- [Environment Configuration](#environment-configuration)
- [Building the Application](#building-the-application)
- [Running the Application Locally](#running-the-application-locally)
- [Docker Deployment](#docker-deployment)
- [Kubernetes Deployment](#kubernetes-deployment)
- [Deployment Checklist](#deployment-checklist)
- [API Documentation](#api-documentation)
- [Troubleshooting](#troubleshooting)
- [License](#license)
- [Project Status](#project-status)

## Architecture

The service is organized into several functional modules:

- **Organizations Module**: Organization registration and management
- **Users Module**: User profile and credential management
- **Requests Module**: Onboarding request tracking and workflow
- **Invitations Module**: Credential issuance invitation management
- **Notifications Module**: Email communication with users and admins
- **Integration Module**: Communication with other FAME services (PT, TM, JWT, Identity)

## Prerequisites

Before building and deploying the service, ensure you have the following installed:

- **Node.js** 20.x (LTS Alpine recommended for Docker builds)
- **npm** (comes with Node.js)
- **Docker** (for containerized deployment)
- For deployment to the GFT hosting infrastructure: access to **Harbor registry** at `harbor.gftinnovation.eu`

## Environment Configuration

The service requires various environment variables to be configured. The configuration approach differs depending on the deployment scenario:

### Local Development/Testing (using `.env` file)

For running the service locally during development and testing, create a `.env` file at the project root. **Important**: The `.env` file in the repository does not contain sensitive values (secrets) for security reasons. You must manually set the following **sensitive variables** as environment variables or add them to your local `.env` file (do not commit them):

- `MYSQL_PASSWORD` - MySQL/MariaDB database password
- `MONGO_URI` - Complete MongoDB connection string including credentials
- `GMAIL_PASS` - Gmail application-specific password for email notifications

### Production Deployment (Kubernetes)

For production deployment on Kubernetes (as used in the FAME production environment):

- **Non-sensitive configuration** is defined in a **ConfigMap** (see example in `fame-integration/gov/gov-service.yaml`)
- **Sensitive values** are stored in **Kubernetes Secrets** and injected as environment variables using `secretKeyRef`

The Kubernetes deployment uses:
- ConfigMap named `gov-config` for general configuration
- Secret named `fame-secrets` containing:
  - `UNIDB_PASS` → `MYSQL_PASSWORD` and `MONGO_PASS`
  - `GMAIL_PASS` → `GMAIL_PASS`

### Configuration Parameters

You will need to know the network addresses of all the services that this service depends on. These addresses depend on your specific hosting environment. Key configuration parameters include:

### Service Integration URLs
```bash
INTERNAL_URL=http://<address of GOV API internal endpoint>/gov/v1.0
PT_URL=http://<address of PT API internal endpoint>/pt/v1.0
TM_URL=http://<address of TM API internal endpoint>/tm/v1.0
JWT_URL=http://<address of JWT service>/api/create-machine-identities
VOC_ISSUANCE_URL=http://<address of AAI Identity service>/webapp
VOC_CLAIM_URL=https://<address of ROA UI>/claim/?iid=
AUTH_URL=https://<address of AAI API public endpoint>/api
```

### MongoDB Configuration
```bash
MONGO_URI=mongodb://<username:password>@<host:port>/FAME
MONGO_NAME=FAME
MONGO_COLLECTION=requests
MONGO_AUTH_SOURCE=admin
```

### MySQL Configuration
```bash
MYSQL_HOST=<MySQL host address>
MYSQL_PORT=3306
MYSQL_USER=root
MYSQL_PASSWORD=<database-password>
MYSQL_DATABASE=gov
```

### Email Configuration
```bash
GMAIL_USER=service@fame-horizon.eu
GMAIL_PASS=<gmail-app-password>
ADMIN_EMAIL=<administrator-email-address>
```

### Authentication & Authorization
```bash
DISABLE_AUTH=false
DISABLE_ROLES=false
```

## Building the Application

### 1. Install Dependencies

```bash
npm install
```

This will install all required dependencies from [package.json](package.json), including:
- NestJS framework and modules
- MongoDB driver (Mongoose)
- MySQL driver (TypeORM)
- Nodemailer for email notifications
- JWT authentication libraries
- Validation and transformation utilities

### 2. Compile TypeScript Code

```bash
npm run build
```

This command executes the NestJS build process, which:
- Compiles TypeScript files from `src/` to JavaScript in `dist/`
- Generates source maps for debugging
- Validates TypeScript configuration

The compiled output will be placed in the `dist/` directory.

### 3. Code Quality (Optional)

Run the linter to check and auto-fix code style issues:

```bash
npm run lint
```

## Running the Application Locally

**Prerequisites**: Before running locally, ensure you have:
1. Completed the build process (`npm install` && `npm run build`)
2. Created a `.env` file with all required configuration (see [Environment Configuration](#environment-configuration))
3. Manually set all sensitive environment variables (these are **not** in the repository `.env` file)

### Standard Mode (Public API)

The public API runs on port **7006** and exposes endpoints for external consumption.

```bash
# Ensure all sensitive environment variables are set
export MYSQL_PASSWORD="your-database-password"
export MONGO_URI="mongodb://username:password@host:port/FAME"
export GMAIL_PASS="your-gmail-app-password"

# Set INTERNAL_API to false or omit it
export INTERNAL_API=false

# Start the application
node dist/src/main.js
```

Access points:
- **API**: `http://localhost:7006`
- **Swagger Documentation**: `http://localhost:7006/swagger`

### Internal Mode (Private API)

The internal API runs on port **7008** and provides privileged endpoints for inter-service communication.

```bash
# Ensure all sensitive environment variables are set (same as above)
export MYSQL_PASSWORD="your-database-password"
export MONGO_URI="mongodb://username:password@host:port/FAME"
export GMAIL_PASS="your-gmail-app-password"

# Set INTERNAL_API to true
export INTERNAL_API=true

# Start the application
node dist/src/main.js
```

Access points:
- **API**: `http://localhost:7008`
- **Swagger Documentation**: `http://localhost:7008/swagger`

The application will automatically generate the appropriate Swagger specification file (`swagger-specification.public.json` or `swagger-specification.private.json`) on startup based on the `INTERNAL_API` setting.

## Docker Deployment

### Building the Docker Image

The service uses a multi-stage Docker build optimized for production deployment.

```bash
npm run docker:build
```

This executes:
```bash
docker build -t harbor.gftinnovation.eu/fame/framework/gov:3.0 .
```

The [Dockerfile](Dockerfile) performs the following:
1. Uses **Node.js 20 Alpine** base image for minimal footprint
2. Sets working directory to `/app`
3. Installs production dependencies using npm cache mounts for faster builds
4. Copies the entire source code (including pre-built `dist/` directory)
5. Exposes port **7006** (public API) or **7008** (internal API)
6. Sets entrypoint to `node dist/src/main.js`

**Important**: You must run `npm run build` **before** building the Docker image, as the Dockerfile expects the compiled code to be present in the `dist/` directory.

### Pushing to Harbor Registry

After building the image, push it to the Harbor container registry:

```bash
npm run docker:push
```

This executes:
```bash
docker login harbor.gftinnovation.eu
docker push harbor.gftinnovation.eu/fame/framework/gov:3.0
```

You will be prompted for Harbor credentials. Ensure you have the appropriate permissions to push to the `fame/framework` project.

### Running the Docker Container

**Important**: When running Docker containers, you must provide sensitive environment variables separately (do not include them in the `.env` file). You can pass them as individual `-e` flags or through a secure secrets management system.

#### Public API Container

```bash
docker run -d \
  --name gov-public \
  -p 7006:7006 \
  --env-file .env \
  -e INTERNAL_API=false \
  -e MYSQL_PASSWORD='<database-password>' \
  -e MONGO_URI='mongodb://username:password@host:port/FAME' \
  -e GMAIL_PASS='<gmail-app-password>' \
  harbor.gftinnovation.eu/fame/framework/gov:3.0
```

#### Internal API Container

```bash
docker run -d \
  --name gov-internal \
  -p 7008:7008 \
  --env-file .env \
  -e INTERNAL_API=true \
  -e MYSQL_PASSWORD='<database-password>' \
  -e MONGO_URI='mongodb://username:password@host:port/FAME' \
  -e GMAIL_PASS='<gmail-app-password>' \
  harbor.gftinnovation.eu/fame/framework/gov:3.0
```

## Kubernetes Deployment

For production deployment on Kubernetes clusters (as used in the FAME production environment), the service is deployed using Kubernetes manifests that separate configuration from secrets.

### Deployment Architecture

The GOV API is deployed as **two separate deployments** in Kubernetes:

1. **Public API** (`gov`) - Exposed externally on port 7006
2. **Internal API** (`gov-internal`) - Used for inter-service communication on port 7008

Both deployments use the same Docker image but differ in the `INTERNAL_API` environment variable.

### Kubernetes Resources

The deployment consists of:

- **ConfigMap** (`gov-config`) - Non-sensitive configuration variables
- **Secret** (`fame-secrets`) - Sensitive credentials and keys
- **Deployments** (`gov` and `gov-internal`) - Application pods
- **Services** (`gov` and `gov-internal`) - Network services
- **Ingress** - External routing rules (defined in `gov-ingress.yaml`)
- **Databases** - MySQL (`gov-db`) and MongoDB (`request-db`) deployments

### Creating the ConfigMap

Create a ConfigMap with all non-sensitive configuration:

```yaml
apiVersion: v1
kind: ConfigMap
metadata:
  name: gov-config
  namespace: fame
data:
  INTERNAL_URL: http://gov-internal:7008/gov/v1.0
  PT_URL: http://pt-internal:7010/pt/v1.0
  TM_URL: http://tm-internal:3000/tm/v1.0
  JWT_URL: http://fame-machine-auth-aai-jwt-service:8027/api/create-machine-identities
  VOC_ISSUANCE_URL: http://fame-aai-identity-service:8022/webapp
  VOC_CLAIM_URL: https://roa.fame-horizon.eu/claim/?iid=
  AUTH_URL: https://auth.fame-horizon.eu/api
  MONGO_NAME: FAME
  MONGO_COLLECTION: requests
  MONGO_AUTH_SOURCE: admin
  MYSQL_HOST: gov-db
  MYSQL_PORT: "3306"
  MYSQL_USER: root
  MYSQL_DATABASE: gov
  GMAIL_USER: service@fame-horizon.eu
  ADMIN_EMAIL: mauro.isaja@eng.it
  DISABLE_AUTH: "false"
  DISABLE_ROLES: "false"
```

### Creating the Secrets

Create Kubernetes secrets for sensitive values:

```bash
kubectl create secret generic fame-secrets \
  --from-literal=UNIDB_PASS='<database-password>' \
  --from-literal=GMAIL_PASS='<gmail-app-password>' \
  --namespace=fame
```

### Deploying to Kubernetes

Once the ConfigMap and Secrets are created, apply the deployment manifests:

```bash
# Apply the deployment and service definitions
kubectl apply -f gov-service.yaml

# Apply database deployments
kubectl apply -f gov-db.yaml
kubectl apply -f request-db.yaml

# Verify deployments
kubectl get deployments -n fame | grep gov

# Check pod status
kubectl get pods -n fame | grep gov

# View logs
kubectl logs -n fame deployment/gov
kubectl logs -n fame deployment/gov-internal
```

### Example Deployment Manifest Structure

The deployment manifest includes both public and internal API deployments:

```yaml
apiVersion: apps/v1
kind: Deployment
metadata:
  name: gov
  namespace: fame
spec:
  replicas: 1
  selector:
    matchLabels:
      app: gov
  template:
    metadata:
      labels:
        app: gov
    spec:
      containers:
      - name: gov
        image: harbor.gftinnovation.eu/fame/framework/gov:3.0
        imagePullPolicy: Always
        command: ["node", "dist/src/main.js"]
        # Load all non-sensitive config from ConfigMap
        envFrom:
        - configMapRef:
            name: gov-config
        # Inject sensitive values from Secrets
        env:
        - name: MYSQL_PASSWORD
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: UNIDB_PASS
        - name: GMAIL_PASS
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: GMAIL_PASS
        - name: MONGO_PASS
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: UNIDB_PASS
        - name: MONGO_URI
          value: mongodb://root:$(MONGO_PASS)@request-db:27017/FAME
        - name: INTERNAL_API
          value: "false"
        ports:
        - containerPort: 7006
      imagePullSecrets:
        - name: registrysecret
```

### Updating the Deployment

To update the service with a new version:

```bash
# Build and push new Docker image (update version tag)
npm run build
docker build -t harbor.gftinnovation.eu/fame/framework/gov:3.1 .
docker push harbor.gftinnovation.eu/fame/framework/gov:3.1

# Update the deployment manifest with new image tag
# Then apply the changes
kubectl apply -f gov-service.yaml

# Or use kubectl set image for quick updates
kubectl set image deployment/gov gov=harbor.gftinnovation.eu/fame/framework/gov:3.1 -n fame
kubectl set image deployment/gov-internal gov-internal=harbor.gftinnovation.eu/fame/framework/gov:3.1 -n fame

# Monitor rollout status
kubectl rollout status deployment/gov -n fame
kubectl rollout status deployment/gov-internal -n fame
```

### Accessing Kubernetes Services

Once deployed, the services are accessible:

- **Public API** (external): Via Ingress rules defined in `gov-ingress.yaml`
- **Internal API** (cluster-internal): `http://gov-internal:7008` from other pods
- **Swagger Documentation**:
  - Public: `https://<ingress-host>/gov/swagger`
  - Internal: `http://gov-internal:7008/swagger`

### Complete Kubernetes Manifests

For complete reference implementations, see the manifest files in the `fame-integration` project:
- [gov-service.yaml](../fame-integration/gov/gov-service.yaml) - Deployments, Services, and ConfigMap
- [gov-ingress.yaml](../fame-integration/gov/gov-ingress.yaml) - Ingress routing rules
- [gov-db.yaml](../fame-integration/gov/gov-db.yaml) - MySQL database deployment
- [request-db.yaml](../fame-integration/gov/request-db.yaml) - MongoDB deployment

## Deployment Checklist

### For Local Development/Testing

When running the service locally for development and testing:

1. **Build Process**
   - [ ] Run `npm install` to install dependencies
   - [ ] Run `npm run build` to compile TypeScript
   - [ ] Verify `dist/` directory contains compiled code

2. **Environment Configuration**
   - [ ] Copy `.env` file from repository as starting point
   - [ ] Manually set sensitive environment variables:
     - [ ] `MYSQL_PASSWORD` - Database password
     - [ ] `MONGO_URI` - MongoDB connection string with credentials
     - [ ] `GMAIL_PASS` - Gmail application password
   - [ ] Configure service URLs to point to your local or test environment
   - [ ] Set `INTERNAL_API=false` for public API or `INTERNAL_API=true` for internal API

3. **Infrastructure Access**
   - [ ] Ensure MySQL/MariaDB is accessible and credentials are correct
   - [ ] Ensure MongoDB is accessible and credentials are correct
   - [ ] Verify Gmail SMTP access with application password
   - [ ] Confirm network access to dependent services (PT, TM, JWT, Identity)

4. **Verification**
   - [ ] Application starts without errors: `node dist/src/main.js`
   - [ ] Swagger UI accessible at `http://localhost:7006/swagger` or `http://localhost:7008/swagger`
   - [ ] Test basic API endpoints
   - [ ] Check logs for database connections
   - [ ] Verify email sending functionality (check logs)

### For Docker Deployment

When deploying using Docker containers:

1. **Build Process**
   - [ ] Run `npm install` to install dependencies
   - [ ] Run `npm run build` to compile TypeScript
   - [ ] Verify `dist/` directory contains compiled code

2. **Docker Image**
   - [ ] Build Docker image: `npm run docker:build`
   - [ ] Tag image appropriately (update version in [package.json](package.json))
   - [ ] Push to Harbor registry: `npm run docker:push`
   - [ ] Verify image is accessible in Harbor

3. **Environment Configuration**
   - [ ] Create `.env` file with non-sensitive values
   - [ ] Set sensitive environment variables separately (do not include in `.env` file)
   - [ ] Configure `INTERNAL_API` flag based on deployment mode

4. **Deployment**
   - [ ] Run container with appropriate environment variables
   - [ ] Map correct port (7006 for public, 7008 for internal)
   - [ ] Configure network connectivity to MySQL, MongoDB, and other services
   - [ ] Set up networking for inter-service communication

5. **Verification**
   - [ ] Container starts without errors: `docker logs <container-name>`
   - [ ] API responds at `http://<host>:<port>/swagger`
   - [ ] Test authentication endpoints
   - [ ] Validate database connections (check container logs)
   - [ ] Verify email functionality

### For Kubernetes Production Deployment

When deploying to production Kubernetes clusters:

1. **Build Process**
   - [ ] Run `npm install` to install dependencies
   - [ ] Run `npm run build` to compile TypeScript
   - [ ] Verify `dist/` directory contains compiled code

2. **Docker Image**
   - [ ] Build Docker image with appropriate version tag
   - [ ] Push to Harbor registry: `npm run docker:push`
   - [ ] Verify image is accessible in Harbor
   - [ ] Update image tag in Kubernetes deployment manifests

3. **Kubernetes Configuration**
   - [ ] Create or update ConfigMap (`gov-config`) with non-sensitive configuration:
     - [ ] Service integration URLs (PT, TM, JWT, Identity, Auth)
     - [ ] MongoDB configuration (name, collection, auth source)
     - [ ] MySQL configuration (host, port, user, database)
     - [ ] Email configuration (sender address, admin email)
     - [ ] Feature flags (DISABLE_* variables)
   - [ ] Create or update Kubernetes Secrets (`fame-secrets`) with sensitive values:
     - [ ] `UNIDB_PASS` → `MYSQL_PASSWORD` and `MONGO_PASS`
     - [ ] `GMAIL_PASS` → Gmail application password
   - [ ] Review deployment manifests for both `gov` and `gov-internal`
   - [ ] Ensure `INTERNAL_API` is set correctly (false for `gov`, true for `gov-internal`)

4. **Infrastructure**
   - [ ] Ensure MySQL service is accessible within the cluster
   - [ ] Ensure MongoDB service is accessible within the cluster
   - [ ] Verify SMTP connectivity for email notifications
   - [ ] Confirm network policies allow inter-service communication
   - [ ] Review Ingress rules for external access to public API
   - [ ] Configure resource limits (CPU, memory) appropriately

5. **Deployment Execution**
   - [ ] Apply ConfigMap: `kubectl apply -f <configmap-file>`
   - [ ] Apply or verify Secrets exist in namespace
   - [ ] Apply database deployments if needed
   - [ ] Apply deployments: `kubectl apply -f gov-service.yaml`
   - [ ] Apply ingress rules: `kubectl apply -f gov-ingress.yaml`
   - [ ] Verify deployments are running: `kubectl get deployments -n fame`
   - [ ] Check pod status: `kubectl get pods -n fame | grep gov`

6. **Verification**
   - [ ] Pods start without errors: `kubectl logs -n fame deployment/gov`
   - [ ] Public API accessible via Ingress: `https://<ingress-host>/gov/swagger`
   - [ ] Internal API accessible from within cluster: `http://gov-internal:7008`
   - [ ] Test authentication endpoints
   - [ ] Validate MySQL connection (check pod logs)
   - [ ] Validate MongoDB connection (check pod logs)
   - [ ] Test email sending functionality
   - [ ] Monitor resource usage: `kubectl top pods -n fame | grep gov`
   - [ ] Verify no crashloop or restart issues: `kubectl get pods -n fame -w`

## API Documentation

Once the service is running, comprehensive API documentation is available via Swagger UI:

- **Public API**: `http://<hostname>:7006/swagger`
- **Internal API**: `http://<hostname>:7008/swagger`

The Swagger interface provides:
- Complete endpoint documentation with request/response schemas
- Interactive API testing capabilities
- Authentication flow examples
- DTO validation rules

Static Swagger specifications are also generated as JSON files:
- [swagger-specification.public.json](swagger-specification.public.json)
- [swagger-specification.private.json](swagger-specification.private.json)

## Troubleshooting

### Common Issues

**MySQL Connection Failed**
- Verify `MYSQL_HOST` and `MYSQL_PORT` are correctly configured
- Ensure MySQL/MariaDB is running and accessible
- Check `MYSQL_PASSWORD` is correct
- Verify database `gov` exists

**MongoDB Connection Failed**
- Verify `MONGO_URI` is correctly formatted
- Ensure MongoDB is running and accessible
- Check authentication credentials in connection string
- Verify `MONGO_AUTH_SOURCE` is set correctly

**Email Sending Failed**
- Verify `GMAIL_USER` and `GMAIL_PASS` are correct
- Ensure Gmail application-specific password is being used (not regular password)
- Check SMTP connectivity and firewall rules
- Review logs for detailed error messages

**Authentication Failures**
- Verify `AUTH_URL` is correct and the auth service is accessible
- If testing locally, consider setting `DISABLE_AUTH=true`

**Service Integration Errors**
- Verify all service URLs (PT, TM, JWT, Identity) are correct
- Check network connectivity to dependent services
- Review service logs for detailed error messages
- Ensure services are running and healthy

**Port Already in Use**
- Check if another service is running on port 7006 or 7008
- Stop conflicting services or change the port in [src/main.ts](src/main.ts:48)

## License

This project is licensed under the terms specified in [LICENSE](LICENSE).

## Project Status

**Active Development** - This service is actively maintained as part of the FAME Marketplace platform.

For issues, feature requests, or contributions, please contact the FAME development team.
