import {
    CanActivate,
    ExecutionContext,
    Injectable,
    ForbiddenException,
} from '@nestjs/common';
import { Reflector } from '@nestjs/core';
import { ROLES_KEY } from './roles-decorator';
import { UserInfo } from './usser-info';
import { ConfigService } from '@nestjs/config';
import { isAuthDisabled, isRoleCheckDisabled } from 'src/utils';

@Injectable()
export class RolesGuard implements CanActivate {

    private readonly skip: boolean = false;

    constructor(private configService: ConfigService, private reflector: Reflector) {
        this.skip = (isAuthDisabled(this.configService) || isRoleCheckDisabled(this.configService));
    }

    canActivate(context: ExecutionContext): boolean {
        if (this.skip) {
            return true;
        }

        // 1. Get the required roles from the metadata
        const requiredRoles = this.reflector.get<string[]>(
            ROLES_KEY,
            context.getHandler(),
        );
        
        // If no roles are specified, allow by default
        if (!requiredRoles || requiredRoles.length === 0) {
            return true;
        }

        // 2. Get the current user from request.user
        // note that this relies on AuthGuard to puts the user in the request object
        const request = context.switchToHttp().getRequest();
        const user = request.user as UserInfo;

        // 3. Check if user.role is in the requiredRoles
        if (requiredRoles.includes(user?.role)) {
            return true;
        }

        // Otherwise, throw forbidden
        throw new ForbiddenException(
            `User role '${user?.role}' is not in [${requiredRoles.join(', ')}]`,
        );
    }
}
