import {ApiProperty, ApiPropertyOptional} from "@nestjs/swagger";
import { IsAlphanumeric, IsEthereumAddress, IsNotEmpty, IsOptional, IsString, Length } from 'class-validator';
import { SHORTID_MAXL, TEXTCOL_MAXL, UserRole } from '../constants';

export class AccountEnrolmentRequest {

    @ApiProperty({
        description: 'Address of the Ethereum account to be enrolled (REQUIRED)',
        example: '0x42eA4063fC0D1E9276c54e1C2cC7EE6d5616b67f',
    })
    @IsNotEmpty()
    @IsEthereumAddress()
    tid : string;
    
    @ApiProperty({
        description: 'DID of the Onboarding Authority of the account owner (REQUIRED, any DID-compliant string)',
        example: 'did:key:z6MkuASbvtFxKxdoKnzSWQWCRL6rJFikcRNq2Eu67mT3HDUX',
    })
    @IsNotEmpty()
    @IsString()
    @Length(1, TEXTCOL_MAXL)
    did: string;

    @ApiProperty({
        description: 'UID of the account owner (REQUIRED, 22-characters short UUID)',
        example: 'w5xQXA2f22UEzLGf9z9Zio',
    })
    @IsNotEmpty()
    @IsAlphanumeric()
    @Length(SHORTID_MAXL, SHORTID_MAXL)
    uid: string;
    
    @ApiPropertyOptional({
        description: 'Affiliation PID of the account owner (OPTIONAL, 22-characters short UUID)',
        example: 'w5xQXA2f22UEzLGf9z9Zio',
    })
    @IsOptional()
    @IsAlphanumeric()
    @Length(SHORTID_MAXL, SHORTID_MAXL)
    affiliation?: string;
}

