import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import { IsEmail, IsNotEmpty, IsOptional, IsPhoneNumber, IsString, Matches, MaxLength, MinLength } from 'class-validator';
import { HALF_LINER_MAXL, ONE_LINER_MAXL } from '../constants';


export class ContactInfo {

    @ApiProperty({
        description: `First name of contact (REQUIRED in input, 1-${HALF_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(HALF_LINER_MAXL)
    @Matches(/^[ \p{Script=Latin}\.]+$/u, {
        message: 'First name must only include Latin letters, dots, and spaces.'
    })
    @Matches(/^(?!.*\.\.).+$/u, {
        message: 'First name must not contain consecutive dots.'
    })
    @Matches(/^(?!.* {2,}).+$/u, {
        message: 'First name must not contain consecutive spaces.'
    })
    @Matches(/.*\S.*/u, {
        message: 'First name cannot be entirely composed of spaces.'
    })
    firstName: string;

    @ApiProperty({
        description: `Last name of contact (REQUIRED in input, 1-${HALF_LINER_MAXL} characters)`,
    })
    @MinLength(1)
    @MaxLength(HALF_LINER_MAXL)
    @Matches(/^[ \p{Script=Latin}\.]+$/u, {
        message: 'Last name must only include Latin letters, dots, and spaces.'
    })
    @Matches(/^(?!.*\.\.).+$/u, {
        message: 'Last name must not contain consecutive dots.'
    })
    @Matches(/^(?!.* {2,}).+$/u, {
        message: 'Last name must not contain consecutive spaces.'
    })
    @Matches(/.*\S.*/u, {
        message: 'Last name cannot be entirely composed of spaces.'
    })
    lastName: string;

    @ApiProperty({
        description: `Email of contact (REQUIRED in input)`,
    })
    @IsNotEmpty()
    @MaxLength(ONE_LINER_MAXL)
    @IsEmail()
    email: string;

    @ApiPropertyOptional({
        description: 'Phone number of contact (OPTIONAL in input) - NOTE: the number must start with a + followed by the country code (e.g. +1 for the US)',
    })
    @IsOptional()
    @MaxLength(HALF_LINER_MAXL)
    @IsPhoneNumber()
    phone?: string;
}
