import { IsAlphanumeric, IsEmail, IsEnum, IsISO31661Alpha2, IsOptional, IsString, Length } from 'class-validator';
import { HALF_LINER_MAXL, SHORTID_MAXL, UserRole, UserStatus } from '../constants';
import { ApiPropertyOptional } from '@nestjs/swagger';
import { PaginationParams } from './pagination-params';

export class UserFilter extends PaginationParams {
    
    @ApiPropertyOptional({
        description: `Filter by invitation ID (OPTIONAL, 22-characters short UUID)`,
        example: 'w5xQXA2f22UEzLGf9z9Zio',
    })
    @IsOptional()
    @IsAlphanumeric()
    @Length(SHORTID_MAXL, SHORTID_MAXL)
    iid?: string;

    @ApiPropertyOptional({
        description: `Filter by user role (OPTIONAL, accepted values: [${Object.values(UserRole).join(', ')}])`,
    })
    @IsOptional()
    @IsEnum(UserRole)
    rol?: UserRole;

    @ApiPropertyOptional({
        description: `Filter by affiliation PID (OPTIONAL, 22-characters short UUID)`,
        example: 'w5xQXA2f22UEzLGf9z9Zio',
    })
    @IsOptional()
    @IsAlphanumeric()
    @Length(SHORTID_MAXL, SHORTID_MAXL)
    aff?: string;

    @ApiPropertyOptional({
        description: 'Filter by country of residence (OPTIONAL, ISO 3166-1 alpha-2 code)',
    })
    @IsOptional()
    @IsISO31661Alpha2()
    cnt?: string;

    @ApiPropertyOptional({
        description: 'Filter by last name (OPTIONAL, partial match, case insensitive)',
    })
    @IsOptional()
    @IsString()
    @Length(1, HALF_LINER_MAXL)
    lname?: string;

    @ApiPropertyOptional({
        description: 'Filter by email address (OPTIONAL, partial match, case insensitive)',
    })
    @IsOptional()
    @IsEmail()
    email?: string;

    @ApiPropertyOptional({
        description: `Filter by user status (OPTIONAL, accepted values: [${Object.values(UserStatus).join(', ')}])`,
    })
    @IsOptional()
    @IsEnum(UserStatus)
    sta?: UserStatus;
}
