import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import { SHORTID_MAXL, UserRole } from '../constants';
import { ContactInfo } from './contact-info';
import { Type } from 'class-transformer';
import { IsAlphanumeric, IsEnum, IsISO31661Alpha2, IsNotEmpty, IsOptional, IsString, Length, ValidateNested } from 'class-validator';

export class UserOnboardingRequest {

    @ApiProperty({
        description: `Role assigned (REQUIRED, accepted values: [${Object.values(UserRole).join(', ')}])`,
        enum: UserRole,
    })
    @IsNotEmpty()
    @IsEnum(UserRole)
    role: UserRole;

    @ApiPropertyOptional({
        description: `Country of residence (REQUIRED, ISO3166 2-letters identifier)`,
    })
    @IsOptional()
    @IsISO31661Alpha2()
    country?: string;

    @ApiPropertyOptional({
        description: `Affiliation PID (OPTIONAL, 22-characters short UUID)`,
        example: 'w5xQXA2f22UEzLGf9z9Zio',
    })
    @IsOptional()
    @IsAlphanumeric()
    @Length(SHORTID_MAXL, SHORTID_MAXL)
    affiliation?: string;

    @ApiPropertyOptional({
        description: `Nickame (OPTIONAL, min 3 / max 22 characters)`,
    })
    @IsOptional()
    @IsString()
    @Length(3, SHORTID_MAXL)
    nickname?: string;
    
    @ApiProperty({
        description: 'Contact information of the user (REQUIRED)',
    })
    @IsNotEmpty()
    @ValidateNested()
    @Type(() => ContactInfo)
    contact: ContactInfo;
}
