import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { EmailService } from './email-service';
import { BackgroundJob } from './background-job-state';

@Injectable()
export class BackgroundJobNotifier {

    constructor(
        private readonly config: ConfigService,
        private readonly emailService: EmailService
    ) {}

    async notifyJobCompletedWithFailures(job: BackgroundJob): Promise<void> {
        try {
            const report = this.generateFailureReport(job);

            // Always log the report locally
            Logger.error('Background job completed with failures', {
                jobId: job.jobId,
                subjectId: job.subjectId,
                subjectType: job.subjectType,
                completedCount: job.completedAccounts.length,
                failedCount: job.failedAccounts.length,
                report
            });

            // Send email if admin email is configured
            const adminEmail = this.config.get<string>('ADMIN_EMAIL');
            if (adminEmail && adminEmail.trim().length > 0) {
                await this.sendEmailReport(adminEmail, job, report);
                Logger.log(`Failure report for job ${job.jobId} sent to ${adminEmail}`);
            } else {
                Logger.warn(`No ADMIN_EMAIL configured - failure report for job ${job.jobId} not sent via email`);
            }
        } catch (err) {
            Logger.error(`Failed to send failure notification for job ${job.jobId}`, err);
        }
    }

    private generateFailureReport(job: BackgroundJob): string {
        const startTime = new Date(job.createdAt).toISOString();
        const endTime = new Date().toISOString();
        const totalAccounts = job.completedAccounts.length + job.failedAccounts.length;

        let report = '';
        report += '='.repeat(80) + '\n';
        report += 'CRITICAL SECURITY ALERT: Account Disenrollment Failures\n';
        report += '='.repeat(80) + '\n\n';

        report += 'EXECUTIVE SUMMARY:\n';
        if (job.subjectType === 'member') {
            report += `During member offboarding, ${job.failedAccounts.length} out of ${totalAccounts} accounts failed to be removed from the blockchain allowlist.\n`;
        } else {
            report += `During user offboarding, ${job.failedAccounts.length} out of ${totalAccounts} accounts failed to be removed from the blockchain allowlist.\n`;
        }
        report += 'This represents a critical security issue that requires immediate manual intervention.\n\n';

        report += 'JOB DETAILS:\n';
        report += `- Job ID: ${job.jobId}\n`;
        report += `- Subject Type: ${job.subjectType}\n`;
        report += `- Subject ID: ${job.subjectId}\n`;

        if (job.context.memberDetails) {
            report += `- Member Name: ${job.context.memberDetails.name}\n`;
            report += `- Member Country: ${job.context.memberDetails.country}\n`;
            report += `- Member Type: ${job.context.memberDetails.type}\n`;
        }

        if (job.context.userDetails) {
            report += `- User OA: ${job.context.userDetails.oa}\n`;
            report += `- User ID: ${job.context.userDetails.uid}\n`;
        }

        report += `- Initiated By: ${job.context.initiatedBy}\n`;
        if (job.context.sourceEndpoint) {
            report += `- Source Endpoint: ${job.context.sourceEndpoint}\n`;
        }
        report += `- Started: ${startTime}\n`;
        report += `- Completed: ${endTime}\n\n`;

        // Include caller context if available
        if (job.context.callerContext) {
            const callerContext = job.context.callerContext;
            report += 'CALLER INFORMATION:\n';
            if (callerContext.userInfo) {
                report += `- User ID: ${callerContext.userInfo.uid}\n`;
                report += `- User DID: ${callerContext.userInfo.issuerDID}\n`;
                report += `- User Role: ${callerContext.userInfo.role}\n`;
                if (callerContext.userInfo.affiliation) {
                    report += `- User Affiliation: ${callerContext.userInfo.affiliation}\n`;
                }
                if (callerContext.userInfo.region) {
                    report += `- User Region: ${callerContext.userInfo.region}\n`;
                }
                if (callerContext.userInfo.nickname) {
                    report += `- User Nickname: ${callerContext.userInfo.nickname}\n`;
                }
            } else {
                report += `- System-initiated operation\n`;
            }
            report += '\n';
        }

        report += 'PROCESSING SUMMARY:\n';
        report += `- Total Accounts Processed: ${totalAccounts}\n`;
        report += `- Successfully Disenrolled: ${job.completedAccounts.length}\n`;
        report += `- Failed Disenrollments: ${job.failedAccounts.length}\n\n`;

        if (job.completedAccounts.length > 0) {
            report += 'SUCCESSFULLY DISENROLLED ACCOUNTS:\n';
            job.completedAccounts.forEach((tid, index) => {
                report += `${index + 1}. ${tid}\n`;
            });
            report += '\n';
        }

        if (job.failedAccounts.length > 0) {
            report += 'FAILED ACCOUNT DISENROLLMENTS (REQUIRES MANUAL INTERVENTION):\n';
            job.failedAccounts.forEach((failure, index) => {
                const failureTime = new Date(failure.timestamp).toISOString();
                report += `${index + 1}. Account TID: ${failure.tid}\n`;
                report += `   Error: ${failure.error}\n`;
                report += `   Failed at: ${failureTime}\n\n`;
            });
        }

        report += 'REQUIRED ACTIONS:\n';
        report += '1. URGENT: Manually verify and remove the failed accounts from the blockchain allowlist\n';
        report += '2. Investigate the root cause of the failures (network issues, service downtime, etc.)\n';
        report += '3. Ensure all listed accounts are properly disenrolled before considering the offboarding complete\n';
        report += '4. Update monitoring to prevent similar issues in the future\n\n';

        report += 'SECURITY IMPLICATIONS:\n';
        report += 'Accounts listed as failed are currently marked as offboarded in the local database\n';
        report += 'but may still have active blockchain permissions. This creates a security gap where\n';
        report += 'supposedly inactive accounts could potentially still perform blockchain operations.\n\n';

        report += 'END OF REPORT\n';
        report += '='.repeat(80) + '\n';

        return report;
    }

    private async sendEmailReport(adminEmail: string, job: BackgroundJob, report: string): Promise<void> {
        const subject = `CRITICAL: Account Disenrollment Failures - Job ${job.jobId}`;

        let subjectInfo = job.subjectId;
        if (job.subjectType === 'member' && job.context.memberDetails) {
            subjectInfo = `${job.context.memberDetails.name} (${job.subjectId})`;
        } else if (job.subjectType === 'user' && job.context.userDetails) {
            subjectInfo = `${job.context.userDetails.oa}:${job.context.userDetails.uid}`;
        }

        let callerInfo = 'System';
        if (job.context.callerContext?.userInfo) {
            callerInfo = `${job.context.callerContext.userInfo.uid} (${job.context.callerContext.userInfo.role})`;
        }

        const contextDescription = job.subjectType === 'member' ? 'offboarding of member' : 'user account disenrollment for';

        const emailBody = `CRITICAL SECURITY ALERT\n\n` +
            `Account disenrollment failures detected during ${contextDescription}: ${subjectInfo}\n` +
            `Initiated by: ${callerInfo}\n\n` +
            `${job.failedAccounts.length} out of ${job.completedAccounts.length + job.failedAccounts.length} accounts failed to be removed from blockchain allowlist.\n\n` +
            `Immediate manual intervention required.\n\n` +
            `Detailed Report:\n\n${report}`;

        await this.emailService.sendMail(adminEmail, subject, emailBody);
    }
}