import { Injectable, OnModuleInit, OnModuleDestroy, Logger } from '@nestjs/common';

export interface BackgroundJobItem {
    tid: string;
    startedAt: number;
    retryCount: number;
}

export interface BackgroundJobContext {
    initiatedBy: 'internal_api' | 'external_api' | 'system';
    sourceEndpoint?: string;
    timestamp: number;
    memberDetails?: {
        pid: string;
        name: string;
        country: string;
        type: string;
    };
    userDetails?: {
        oa: string;
        uid: string;
    };
    callerContext?: any; // Will contain CallerContext from member-offboarding-request
}

export interface BackgroundJob {
    jobId: string;
    subjectId: string; // Either memberPid or user identifier (oa:uid)
    subjectType: 'member' | 'user';
    pendingAccounts: string[];
    currentItem: BackgroundJobItem | null;
    completedAccounts: string[];
    failedAccounts: { tid: string; error: string; timestamp: number }[];
    createdAt: number;
    status: 'running' | 'completed' | 'failed';
    context: BackgroundJobContext;
}

@Injectable()
export class BackgroundJobState implements OnModuleInit, OnModuleDestroy {

    private readonly MAX_AGE: number = 300000; // 5 minutes max age for completed jobs
    private readonly TIMEOUT_MS: number = 10000; // 10 seconds timeout per account
    private readonly MAX_RETRIES: number = 0; // No retries as per requirements

    private cleanupIntervalId: NodeJS.Timeout;
    private timeoutCheckIntervalId: NodeJS.Timeout;

    private jobs = new Map<string, BackgroundJob>();

    // Callback for when a job completes with failures
    private onJobCompletedWithFailures?: (job: BackgroundJob) => void;

    onModuleInit(): void {
        this.cleanupIntervalId = setInterval(() => this.cleanup(), this.MAX_AGE);
        this.timeoutCheckIntervalId = setInterval(() => this.checkTimeouts(), 1000); // Check every second
        Logger.debug('BackgroundJobState intervals started');
    }

    onModuleDestroy(): void {
        if (this.cleanupIntervalId) {
            clearInterval(this.cleanupIntervalId);
        }
        if (this.timeoutCheckIntervalId) {
            clearInterval(this.timeoutCheckIntervalId);
        }
        Logger.debug('BackgroundJobState intervals cleared');
    }

    createJob(jobId: string, subjectId: string, subjectType: 'member' | 'user', accountIds: string[], context: BackgroundJobContext): void {
        if (this.jobs.has(jobId)) {
            throw new Error(`Background job with ID "${jobId}" already exists.`);
        }

        const job: BackgroundJob = {
            jobId,
            subjectId,
            subjectType,
            pendingAccounts: [...accountIds],
            currentItem: null,
            completedAccounts: [],
            failedAccounts: [],
            createdAt: Date.now(),
            status: 'running',
            context
        };

        this.jobs.set(jobId, job);
        Logger.debug(`Created background job ${jobId} for ${subjectType} ${subjectId} with ${accountIds.length} accounts`);
    }

    getNextAccount(jobId: string): string | null {
        const job = this.jobs.get(jobId);
        if (!job || job.status !== 'running' || job.pendingAccounts.length === 0) {
            return null;
        }

        const nextTid = job.pendingAccounts.shift();
        if (nextTid) {
            job.currentItem = {
                tid: nextTid,
                startedAt: Date.now(),
                retryCount: 0
            };
            Logger.debug(`Job ${jobId}: Processing account ${nextTid}`);
        }

        return nextTid;
    }

    markAccountCompleted(jobId: string, tid: string): void {
        const job = this.jobs.get(jobId);
        if (!job) {
            Logger.warn(`Job ${jobId} not found when marking account ${tid} as completed`);
            return;
        }

        if (job.currentItem?.tid === tid) {
            job.completedAccounts.push(tid);
            job.currentItem = null;
            Logger.debug(`Job ${jobId}: Account ${tid} completed successfully`);

            // Check if job is finished
            if (job.pendingAccounts.length === 0) {
                job.status = 'completed';
                Logger.log(`Background job ${jobId} completed. Processed ${job.completedAccounts.length} accounts, ${job.failedAccounts.length} failed`);

                // Trigger notification if there were any failures
                if (job.failedAccounts.length > 0 && this.onJobCompletedWithFailures) {
                    this.onJobCompletedWithFailures(job);
                }
            }
        }
    }

    markAccountFailed(jobId: string, tid: string, error: string): void {
        const job = this.jobs.get(jobId);
        if (!job) {
            Logger.warn(`Job ${jobId} not found when marking account ${tid} as failed`);
            return;
        }

        if (job.currentItem?.tid === tid) {
            job.failedAccounts.push({
                tid,
                error,
                timestamp: Date.now()
            });
            job.currentItem = null;
            Logger.error(`Job ${jobId}: Account ${tid} failed - ${error}`);

            // Check if job is finished
            if (job.pendingAccounts.length === 0) {
                job.status = 'completed';
                Logger.log(`Background job ${jobId} completed. Processed ${job.completedAccounts.length} accounts, ${job.failedAccounts.length} failed`);

                // Trigger notification if there were any failures
                if (job.failedAccounts.length > 0 && this.onJobCompletedWithFailures) {
                    this.onJobCompletedWithFailures(job);
                }
            }
        }
    }

    getCurrentItem(jobId: string): BackgroundJobItem | null {
        const job = this.jobs.get(jobId);
        return job?.currentItem || null;
    }

    getJob(jobId: string): BackgroundJob | null {
        return this.jobs.get(jobId) || null;
    }

    getAllJobs(): Map<string, BackgroundJob> {
        return this.jobs;
    }

    setJobCompletionCallback(callback: (job: BackgroundJob) => void): void {
        this.onJobCompletedWithFailures = callback;
    }

    private checkTimeouts(): void {
        const now = Date.now();

        for (const [jobId, job] of this.jobs) {
            if (job.currentItem && (now - job.currentItem.startedAt) > this.TIMEOUT_MS) {
                Logger.warn(`Job ${jobId}: Account ${job.currentItem.tid} timed out after ${this.TIMEOUT_MS}ms`);
                this.markAccountFailed(jobId, job.currentItem.tid, 'Timeout waiting for external service response');
            }
        }
    }

    private cleanup(): void {
        const now = Date.now();

        for (const [jobId, job] of this.jobs) {
            if (job.status === 'completed' && (now - job.createdAt) > this.MAX_AGE) {
                this.jobs.delete(jobId);
                Logger.debug(`Expired background job ${jobId} removed`);
            }
        }
    }
}