import { Injectable, OnModuleInit, OnModuleDestroy, Logger } from '@nestjs/common';

@Injectable()
export class ProcessState implements OnModuleInit, OnModuleDestroy {

    // Max age of a request in milliseconds. Default is 60 seconds.
    private readonly MAX_AGE: number = 60000;

    // Holds the reference to the cleanup interval.
    private cleanupIntervalId: NodeJS.Timeout;

    // Each entry stores the request and its creation timestamp.
    private requests = new Map<string, { request: any; createdAt: number }>();

    // Start the cleanup interval when the module initializes.
    onModuleInit(): void {
        this.cleanupIntervalId = setInterval(() => this.cleanup(), this.MAX_AGE);
        Logger.debug('ProcessState cleanup interval started');
    }

    // Clear the cleanup interval when the module is destroyed.
    onModuleDestroy(): void {
        if (this.cleanupIntervalId) {
            clearInterval(this.cleanupIntervalId);
            Logger.debug('ProcessState cleanup interval cleared');
        }
    }

    // Add a new request. Throws if the key already exists.
    push(key: string, request: any): void {
        if (this.requests.has(key)) {
            throw new Error(`Process with key "${key}" already exists.`);
        }
        this.requests.set(key, { request, createdAt: Date.now() });
    }

    // Retrieve a request by key. Returns undefined if not found.
    peek(key: string): any | undefined {
        const entry = this.requests.get(key);
        return entry ? entry.request : undefined;
    }

    // Remove a request by key. Returns the removed object, or undefined if not found.
    pop(key: string): any | undefined {
        const entry = this.requests.get(key);
        if (entry) {
            this.requests.delete(key);
            return entry.request;
        }
        return undefined;
    }

    // Private method: cleans up requests older than MAX_AGE.
    private cleanup(): void {
        const now = Date.now();
        for (const [key, { createdAt }] of this.requests) {
            if (now - createdAt > this.MAX_AGE) {
                this.requests.delete(key);
                Logger.debug(`Expired process with key ${key} removed`);
            }
        }
    }
}
