import { Injectable } from '@nestjs/common';
import { Identifier } from 'src/dtos/identifier';
import { UserBlacklistFilter } from 'src/dtos/user-blacklist-filter';
import { MemberBlacklistFilter } from 'src/dtos/member-blacklist-filter';
import { AuthorityBlacklistFilter } from 'src/dtos/authority-blacklist-filter';
import { UserQualifiedId } from 'src/dtos/user-qualified-id';
import { BlacklistStorage } from 'src/membership/blacklist-storage';
import { ErrorCondition, throwExceptionWithErrorCondition } from 'src/utils';
import { BlacklistAuthorityEntity, BlacklistMemberEntity, BlacklistUserEntity } from './blacklist-entities';

@Injectable()
export class BlacklistService {

    constructor(private db: BlacklistStorage) { }

    async addUserBan(oa: string, uid: string): Promise<void> {
        try {
            await this.db.insertUserEntry(oa, uid);
        } catch (err) {
            if (/duplicate|unique/i.test(err.message)) {
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, "Entry already exists");
            } else {
                throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to add entry");
            }
        }
    }

    async removeUserBan(oa: string, uid: string): Promise<boolean> {
        try {
            return await this.db.deleteUserEntry(oa, uid);
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to remove entry");
        }
    }

    async isUserBanned(oa: string, uid: string): Promise<boolean> {
        try {
            return await this.db.checkUserEntry(oa, uid);
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to check entry");
        }
    }

    async listBannedUsers(query?: UserBlacklistFilter): Promise<UserQualifiedId[]> {
        try {
            const results: BlacklistUserEntity[] = await this.db.getUserEntries(query);
            return results.map(item => {
                return {
                    oa: item.oa,
                    uid: item.uid
                };
            });
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to retrieve entries");
        }
    }

    async addMemberBan(id: string): Promise<void> {
        try {
            await this.db.insertMemberEntry(id);
        } catch (err) {
            if (/duplicate|unique/i.test(err.message)) {
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, "Entry already exists");
            } else {
                throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to add entry");
            }
        }
    }

    async removeMemberBan(id: string): Promise<boolean> {
        try {
            return await this.db.deleteMemberEntry(id);
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to remove entry");
        }
    }

    async isMemberBanned(id: string): Promise<boolean> {
        try {
            return await this.db.checkMemberEntry(id);
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to check entry");
        }
    }

    async listBannedMembers(query?: MemberBlacklistFilter): Promise<Identifier[]> {
        try {
            const results: BlacklistMemberEntity[] = await this.db.getMemberEntries(query);
            return results.map(item => {
                return {
                    id: item.pid
                };
            });
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to retrieve entries");
        }
    }

    async addAuthorityBan(id: string): Promise<void> {
        try {
            await this.db.insertAuthorityEntry(id);
        } catch (err) {
            if (/duplicate|unique/i.test(err.message)) {
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, "Entry already exists");
            } else {
                throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to add entry");
            }
        }
    }

    async removeAuthorityBan(id: string): Promise<boolean> {
        try {
            return await this.db.deleteAuthorityEntry(id);
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to remove entry");
        }
    }

    async isAuthorityBanned(id: string): Promise<boolean> {
        try {
            return await this.db.checkAuthorityEntry(id);
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to check entry");
        }
    }

    async listBannedAuthorities(query?: AuthorityBlacklistFilter): Promise<Identifier[]> {
        try {
            const results: BlacklistAuthorityEntity[] = await this.db.getAuthorityEntries(query);
            return results.map(item => {
                return {
                    id: item.did
                };
            });
        } catch (err) {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, "System failure, failed to retrieve entries");
        }
    }
}
