import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { BlacklistAuthorityEntity, BlacklistMemberEntity, BlacklistUserEntity } from './blacklist-entities';
import { UserBlacklistFilter } from 'src/dtos/user-blacklist-filter';
import { MemberBlacklistFilter } from 'src/dtos/member-blacklist-filter';
import { AuthorityBlacklistFilter } from 'src/dtos/authority-blacklist-filter';

@Injectable()
export class BlacklistStorage {

    constructor(
        @InjectRepository(BlacklistUserEntity)
        private readonly users: Repository<BlacklistUserEntity>,
        @InjectRepository(BlacklistMemberEntity)
        private readonly members: Repository<BlacklistMemberEntity>,
        @InjectRepository(BlacklistAuthorityEntity)
        private readonly authorities: Repository<BlacklistAuthorityEntity>,
    ) { }

    async insertUserEntry(oa: string, uid: string): Promise<void> {
        await this.users.insert({ oa: oa, uid: uid });
    }

    async deleteUserEntry(oa: string, uid: string): Promise<boolean> {
        const result = await this.users.delete({ oa, uid });
        return result.affected > 0;
    }

    async checkUserEntry(oa: string, uid: string): Promise<boolean> {
        const cnt = await this.users.count({ where: { oa, uid } });
        return cnt > 0;
    }

    async getUserEntries(query?: UserBlacklistFilter): Promise<BlacklistUserEntity[]> {
        const queryBuilder = this.users.createQueryBuilder('user');

        if (query) {
            if (query.oa) {
                queryBuilder.andWhere('user.oa = :oa', { oa: query.oa.trim() });
            }
            if (query.uid) {
                queryBuilder.andWhere('user.uid = :uid', { uid: query.uid.trim() });
            }
        }

        // Apply sorting
        queryBuilder.orderBy('user.reg', 'DESC');

        // Apply pagination if provided
        if (query?.l !== undefined) {
            queryBuilder.limit(query.l);
        }
        if (query?.o !== undefined) {
            queryBuilder.offset(query.o);
        }

        return queryBuilder.getMany();
    }

    async insertMemberEntry(pid: string): Promise<void> {
        await this.members.insert({ pid: pid });
    }

    async deleteMemberEntry(pid: string): Promise<boolean> {
        const result = await this.members.delete({ pid: pid });
        return result.affected > 0;
    }

    async checkMemberEntry(pid: string): Promise<boolean> {
        const cnt = await this.members.count({ where: { pid: pid } });
        return cnt > 0;
    }

    async getMemberEntries(query?: MemberBlacklistFilter): Promise<BlacklistMemberEntity[]> {
        const queryBuilder = this.members.createQueryBuilder('member');

        // Apply sorting
        queryBuilder.orderBy('member.reg', 'DESC');

        // Apply pagination if provided
        if (query?.l !== undefined) {
            queryBuilder.limit(query.l);
        }
        if (query?.o !== undefined) {
            queryBuilder.offset(query.o);
        }

        return queryBuilder.getMany();
    }

    async insertAuthorityEntry(did: string): Promise<void> {
        await this.authorities.insert({ did: did });
    }

    async deleteAuthorityEntry(did: string): Promise<boolean> {
        const result = await this.authorities.delete({ did: did });
        return result.affected > 0;
    }

    async checkAuthorityEntry(did: string): Promise<boolean> {
        const cnt = await this.authorities.count({ where: { did: did } });
        return cnt > 0;
    }

    async getAuthorityEntries(query?: AuthorityBlacklistFilter): Promise<BlacklistAuthorityEntity[]> {
        const queryBuilder = this.authorities.createQueryBuilder('authority');

        // Apply sorting
        queryBuilder.orderBy('authority.reg', 'DESC');

        // Apply pagination if provided
        if (query?.l !== undefined) {
            queryBuilder.limit(query.l);
        }
        if (query?.o !== undefined) {
            queryBuilder.offset(query.o);
        }

        return queryBuilder.getMany();
    }
}
