import {
    BadRequestException,
    Body,
    Controller,
    Delete,
    Get,
    Headers,
    HttpCode,
    Logger,
    NotFoundException,
    Param,
    Post,
    Put,
    Query,
} from '@nestjs/common';
import { ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiTags } from '@nestjs/swagger';
import { MembershipService } from './membership-service';
import { Identifier } from 'src/dtos/identifier';
import { MemberOnboardingRequest } from 'src/dtos/member-onboarding-request';
import { translateToHttpException } from 'src/utils';
import { AccountEnrolmentRequest } from 'src/dtos/account-enrolment-request';
import { AccountStatus, EntityStatus, HTTP_ERR_400, HTTP_ERR_404, HTTP_ERR_500, HTTP_ERR_502 } from 'src/constants';
import { AuthorityFilter } from 'src/dtos/authority-filter';
import { Authority } from 'src/dtos/authority';
import { MemberFilter } from 'src/dtos/member-filter';
import { MemberRef } from 'src/dtos/member-ref';
import { AccountFilter } from 'src/dtos/account-filter';
import { Account } from 'src/dtos/account';
import { SecurityContext } from 'src/dtos/security-context';
import { Amount } from 'src/dtos/amount';
import { TransactionResult } from 'src/dtos/transaction-result';
import { TransactionList } from 'src/dtos/transaction-list';
import { AccountBalance } from 'src/dtos/account-balance';
import { UserQualifiedId } from 'src/dtos/user-qualified-id';
import { LegalEntity } from 'src/dtos/legal-entity';
import { MemberUpdateRequest } from 'src/dtos/member-update-request';

@ApiTags('Internal API: Membership Management')
@Controller('/gov/v1.0')
export class MembershipControllerInternal {

    constructor(private readonly membership: MembershipService) { }

    @Post('/members')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Starts the onboarding of a new member [INTERNAL VERSION]',
        description: `This operation starts the onboarding process for a new member. The operation immediately returns the
  assigned PID, but the actual onboarding process is executed offline: the caller will have to check later for its outcome.`,
    })
    @ApiBody({
        type: MemberOnboardingRequest,
        description: `The request body contains the information required to onboard a new member.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The onboarding process has been started and will be executed offline. The response body contains the PID
  that will be assigned to the new member is the process is successful: this reference can be used to check the actual status of
  the onboarding. If the process is unsuccessful, the PID can be discarded as no persistent information linked to it will be kept.`,
        type: Identifier
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async onboardMember(@Body() req: MemberOnboardingRequest): Promise<Identifier> {
        Logger.log('Member onboarding requested (internal call): ' + JSON.stringify(req));
        try {
            return await this.membership.startOnboardingProcess(req);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Delete('/members/:pid')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Starts the offboarding of an existing member [INTERNAL VERSION]',
        description: `This operation starts the offboarding process for an existing member. This operation is irreversible, although the member data
  will be kept for auditing purposes. The process is executed offline and the caller will have to check later for its outcome.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member to offboard'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The offboarding process has been started and will be executed offline`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async offboardMember(@Param('pid') pid: string, @Headers('x-caller-context') callerContextHeader?: string): Promise<void> {
        Logger.log('Member offboarding requested for PID ' + pid + ' (internal call)');

        let callerContext = null;
        if (callerContextHeader) {
            try {
                callerContext = JSON.parse(callerContextHeader);
                Logger.log('Caller context received: ' + JSON.stringify(callerContext));
            } catch (err) {
                Logger.warn('Failed to parse caller context header: ' + err.message);
            }
        }

        try {
            await this.membership.startOffboardingProcess(pid, callerContext);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/members/:pid/migrate')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Starts the migration of an existing member to a new legal entity [INTERNAL VERSION]',
        description: `This operation starts the migration process for an existing member to a new legal entity. The operation immediately returns the
  new PID, but the actual migration process is executed offline: the caller will have to check later for its outcome.`,
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member to migrate'
    })
    @ApiBody({
        type: LegalEntity,
        description: `The request body contains the new legal entity information for the member.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The migration process has been started and will be executed offline. The response body contains the new PID
  that will be assigned to the member if the process is successful. If the process is unsuccessful, the new PID can be discarded
  as no changes will be applied to the member, and the original PID will remain valid.`,
        type: Identifier
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async migrateMember(@Param('pid') pid: string, @Body() req: LegalEntity, @Headers('x-caller-context') callerContextHeader?: string): Promise<Identifier> {
        Logger.log('Member migration requested for PID ' + pid + ' (internal call)');

        let callerContext = null;
        if (callerContextHeader) {
            try {
                callerContext = JSON.parse(callerContextHeader);
                Logger.log('Caller context received: ' + JSON.stringify(callerContext));
            } catch (err) {
                Logger.warn('Failed to parse caller context header: ' + err.message);
            }
        }

        try {
            return await this.membership.startMigrationProcess(pid, req, callerContext);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put('/members/:pid')
    @HttpCode(204)
    @ApiOperation({
        summary: 'Updates the non-trusted information for a member [INTERNAL VERSION]',
        description: `This operation updates the non-trusted information for member, such as the organization type,
 contact info and link to the Onboarding Authority - i.e., everything that is NOT stored as a Blockchain record (see the
 "migrate member" operation for applying changes that affect Blockchain data). This internal version does not perform authorization checks.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member to update'
    })
    @ApiBody({
        type: MemberUpdateRequest,
        description: `The request body contains the new non-trusted information for the member. At least one field must be provided.
 Anything that is not provided will be left unchanged. Note: Only the representative contact info (rep) can be cleared by setting it to null explicitly.
 Organization type and OA are required fields and cannot be set to null.`,
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: 'The operation was completed successfully'
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async updateMember(@Param('pid') pid: string, @Body() req: MemberUpdateRequest): Promise<void> {
        Logger.log('Member update requested for PID ' + pid + ' (internal call)');
        try {
            await this.membership.updateMember(pid, req);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put('/members/:pid/confirm-registration')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Confirms source registration (callback operation)',
        description: `Callback operation to confirm that the source registration process has been completed successfully.
  This operation is called by the P&T module after the registration process has been completed, and it is used to finalize
  the onboarding of a new member.`,
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member being onboarded'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The confirmation has been acknowledged and the onboarding process will be finalized offline.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    confirmSourceRegistration(@Param('pid') pid: string): void {
        Logger.log('Registration confirmed (internal call) for source with PID ' + pid);
        if (!this.membership.sourceRegistrationConfirmed(pid)) {
            throw new BadRequestException('No pending process exists with this ID');
        }
    }

    @Put('/members/:pid/confirm-deregistration')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Confirms source deregistration (callback operation)',
        description: `Callback operation to confirm that the source deregistration process has been completed successfully.
  This operation is called by the P&T module after the deregistration process has been completed, and it is used to finalize
  the offboarding of a member.`,
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member being offboarded'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The confirmation has been acknowledged and the offboarding process will be finalized offline.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    confirmSourceDeregistration(@Param('pid') pid: string): void {
        Logger.log('Deregistration confirmed (internal call) for source with PID ' + pid);
        if (!this.membership.sourceDeregistrationConfirmed(pid)) {
            throw new BadRequestException('No pending process exists with this ID');
        }
    }

    @Put('/members/:pid/confirm-migration')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Confirms source migration (callback operation)',
        description: `Callback operation to confirm that the source migration process has been completed successfully.
  This operation is called by the P&T module after the migration process has been completed, and it is used to finalize
  the migration of a member to a new legal entity. Note that the PID parameter is the OLD PID of the member being migrated.`,
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'Old PID of the member being migrated (the PID before migration)'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The confirmation has been acknowledged and the migration process will be finalized offline.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    confirmSourceMigration(@Param('pid') pid: string): void {
        Logger.log('Migration confirmed (internal call) for source with old PID ' + pid);
        if (!this.membership.sourceMigrationConfirmed(pid)) {
            throw new BadRequestException('No pending process exists with this ID');
        }
    }

    @Get('/authorities/:did/check')
    @HttpCode(204)
    @ApiOperation({
        summary: 'Check authority / affiliation [DEPRECATED]',
        description: `This operation checks that the given authority DID and optional affiliation PID are valid and consistent.
  NOTE: this operation is deprecated and will be removed in future releases. Use the /clearance-check operation instead.`
    })
    @ApiParam({
        name: 'did',
        required: true,
        type: String,
        description: 'DID of the authority being checked'
    })
    @ApiQuery({
        name: 'a',
        required: false,
        type: String,
        description: 'PID of the affiliation being checked for consistency'
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The combination of parameters is valid `
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async checkLoginContextDeprecated(@Param('did') did: string, @Query('a') a?: string): Promise<void> {
        if (!await this.membership.checkLoginContext(did, a)) {
            throw new NotFoundException('The specified authority/affiliation does not exist or is not valid');
        }
    }

    @Get('/active-members')
    @HttpCode(200)
    @ApiOperation({
        summary: 'List active members [INTERNAL VERSION]',
        description: `This operation provides the full list of active members.`
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains a list of references to members`,
        type: MemberRef,
        isArray: true
    })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveMembers(@Query() pagination?: MemberFilter): Promise<MemberRef[]> {
        const query = new MemberFilter();
        query.sta = EntityStatus.ACTIVE;
        if (pagination?.l !== undefined) query.l = pagination.l;
        if (pagination?.o !== undefined) query.o = pagination.o;
        return await this.membership.listMembers(query);
    }

    @Get('/active-members/:pid')
    @HttpCode(200)
    @ApiOperation({
        summary: 'Resolve the PID of an active member [INTERNAL VERSION]',
        description: `This operation resolves a PID into the name and basic details of an active member.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID to be resolved'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains the basic details of the target member`,
        type: MemberRef
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveActiveMemberRef(@Param('pid') pid: string): Promise<MemberRef> {
        const match = await this.membership.retrieveActiveMemberRef(pid);
        if (!match) {
            throw new NotFoundException('Member record not found');
        }
        return match;
    }

    @Get('/active-authorities')
    @HttpCode(200)
    @ApiOperation({
        summary: 'List active authorities',
        description: `This operation provides the full list of active authorities.`
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains a list of references to active authorities`,
        type: Authority,
        isArray: true
    })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveAuthorities(@Query() pagination?: AuthorityFilter): Promise<Authority[]> {
        const query = new AuthorityFilter();
        query.sta = EntityStatus.ACTIVE;
        if (pagination?.l !== undefined) query.l = pagination.l;
        if (pagination?.o !== undefined) query.o = pagination.o;
        return await this.membership.listAuthorities(query);
    }

    @Get('/active-authorities/:did')
    @HttpCode(200)
    @ApiOperation({
        summary: 'Resolve the DID of an active authority',
        description: `This operation resolves a DID into the name and basic details of an active authority.`
    })
    @ApiParam({
        name: 'did',
        required: true,
        type: String,
        description: 'DID to be resolved'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains the name and basic details of the matching authority`,
        type: Authority
    })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveActiveAuthority(@Param('pid') did: string): Promise<Authority> {
        const match = await this.membership.retrieveAuthority(did, true);
        if (!match) {
            throw new NotFoundException('Authority record not found');
        }
        return match;
    }

    @Post('/accounts')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Starts the process of enrolling a new trading account [INTERNAL VERSION]',
        description: `This operation starts the enrolment process for a new trading account. The operation
  immediately returns a response indicating that the request has been accepted, but the actual processing is
  executed offline: the caller will have to check later for its outcome. Note that this internal operatio
  allows to set any arbitrary DID-UID pair as the account owner.`
    })
    @ApiBody({
        type: AccountEnrolmentRequest,
        description: 'The request body contains the context of the enrolment request',
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: 'The enrolment process has been started and will be executed offline',
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async enrolAccount(@Body() req: AccountEnrolmentRequest): Promise<void> {
        Logger.log('Account enrolment requested (internal call) for TID ' + req.tid);
        try {
            await this.membership.startAccountEnrolmentProcess(req);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Delete('/accounts/:tid')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Starts the process of disenrolling an existing trading account [INTERNAL VERSION]',
        description: `This operation starts the disenrolment process for an existing trading account. The operation
  immediately returns a response indicating that the request has been accepted, but the actual processing is
  executed offline: the caller will have to check later for its outcome. This operation cannot be reverted.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The offboarding process has been started and will be executed offline`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async disenrolAccount(@Param('tid') tid: string): Promise<void> {
        Logger.log('Account disenrolment requested (internal call) for TID ' + tid);
        if (!await this.membership.checkAccountStatus(tid, true)) {
            Logger.warn('Bad target: disenrolment request rejected');
            throw new NotFoundException('Trading account does not exist');
        }
        try {
            await this.membership.startAccountDisenrolmentProcess(tid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put('/accounts/:tid/allowed')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Confirms account permissioning (callback operation)',
        description: `This operation confirms that the account permissioning process has been completed successfully.
  This operation is called by the P&T module after the permissioning process has been completed, and it is used to finalize
  the enrolment of a new account.`,
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the account being enrolled'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The confirmation has been acknowledged and the enrolment process will be finalized offline.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    confirmAccountAllowed(@Param('tid') tid: string): void {
        Logger.log('Permission grant confirmed (internal call) for account with TID ' + tid);
        if (!this.membership.accountPermissionGrantConfirmed(tid)) {
            throw new BadRequestException('No pending process exists with this ID');
        }
    }

    @Put('/accounts/:tid/disallowed')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Confirms account de-permissioning (callback operation)',
        description: `This operation confirms that the account de-permissioning process has been completed successfully.
  This operation is called by the P&T module after the de-permissioning process has been completed, and it is used to finalize
  the disenrolment of an account.`,
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the account being disenrolled'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The confirmation has been acknowledged and the disenrolment process will be finalized offline.`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    confirmAccountDisallowed(@Param('tid') tid: string): void {
        Logger.log('Permission revocation confirmed (internal call) for account with TID ' + tid);
        if (!this.membership.accountPermissionRevocationConfirmed(tid)) {
            throw new BadRequestException('No pending process exists with TID ' + tid);
        }
    }

    @Get('/accounts')
    @ApiOperation({
        summary: 'List all trading accounts [INTERNAL VERSION]',
        description: `This operation provides the list of registered (both active and not active) trading accounts,
  which can be filtered by various parameters.`
    })
    @ApiQuery({
        name: 'usr',
        required: false,
        type: String,
        description: 'Owner UID (user) to filter accounts by'
    })
    @ApiQuery({
        name: 'own',
        required: false,
        type: String,
        description: 'Owner PID (user affiliation) to filter accounts by'
    })
    @ApiQuery({
        name: 'aut',
        required: false,
        type: String,
        description: 'Onboarding Authority DID (user onboarder) to filter accounts by'
    })
    @ApiQuery({
        name: 'sta',
        required: false,
        enum: EntityStatus,
        description: `Status to filter accounts by, accepted values: [${Object.values(EntityStatus).join(', ')}]`,
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The response body contains a list of trading accounts',
        type: Account,
        isArray: true
    })
    async listAllAccounts(@Query() query?: AccountFilter): Promise<Account[]> {
        return await this.membership.listAccounts(query); // unprofiled operation
    }

    @Get('/accounts/:tid')
    @ApiOperation({
        summary: 'Resolves the TID of a trading account [INTERNAL VERSION]',
        description: `This operation resolves the TID of a trading account.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID to be resolved'
    })
    @ApiResponse({
        status: 200,
        description: 'The response body contains the details of the target trading account',
        type: Account
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveAccount(@Param('tid') tid: string): Promise<Account> {
        let account: Account = await this.membership.retrieveAccount(tid);
        if (account) {
            return account;
        } else {
            throw new NotFoundException('Trading account not found');
        }
    }

    @Get('/accounts/:tid/balance')
    @ApiOperation({
        summary: 'Gets the current balance of a trading account [INTERNAL VERSION]',
        description: `This operation returns the current balance and status of a trading account.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account'
    })
    @ApiResponse({
        status: 200,
        description: 'The response body contains the balance and status of the target trading account',
        type: AccountBalance
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async retrieveAccountBalance(@Param('tid') tid: string): Promise<AccountBalance> {
        const account = await this.retrieveAccount(tid); // not found exception will be raised if the account does not exist
        try {
            return {
                balance: await this.membership.retrieveAccountBalance(tid),
                status: account.disenrolled ? AccountStatus.INACTIVE : AccountStatus.ACTIVE
            };
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get('/accounts/:tid/transactions')
    @HttpCode(200)
    @ApiOperation({
        summary: 'Retrieves the token management transaction history for a given trading account. [INTERNAL VERSION]',
        description: `This operation retrieves all token management transactions (credit and debit operations) for the specified trading account,
  listed in descending chronological order.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the trading account'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The request has been processed successfully',
        type: TransactionList
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listAccountTransactions(@Param('tid') tid: string): Promise<TransactionList> {
        try {
            return await this.membership.listAccountTransactions(tid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/accounts/:tid/credit')
    @HttpCode(200)
    @ApiOperation({
        summary: 'Mints a given amount of FDE tokens and transfers them to a given trading account. [INTERNAL VERSION]',
        description: `This operation mints a given amount of FDE tokens and transfers them to the specified trading account.
  The operation is only available to administrators and operators. Upon successful completion, the operation returns
  a transaction record including a unique transaction ID and timestamp.`
    })
    @ApiBody({
        type: Amount,
        description: `Amount of FDE tokens to be minted and transferred (REQUIRED)`,
        required: true,
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The request has been processed successfully',
        type: TransactionResult
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async creditAccount(@Param('tid') tid: string, @Body() req: Amount): Promise<TransactionResult> {
        Logger.log(`Token minting requested for account ${tid} (internal call): ${JSON.stringify(req)}`);
        try {
            return await this.membership.creditAccount(tid, req.amount);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/accounts/:tid/debit')
    @HttpCode(200)
    @ApiOperation({
        summary: 'Burns a given amount of FDE tokens in a given trading account. [INTERNAL VERSION]',
        description: `This operation burns a given amount of FDE tokens in the specified trading account.
  The operation is only available to administrators and operators. Upon successful completion, the operation returns
  a transaction record including a unique transaction ID and timestamp.`
    })
    @ApiBody({
        type: Amount,
        description: `Amount of FDE tokens to be burned (REQUIRED)`,
        required: true,
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The request has been processed successfully',
        type: TransactionResult
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async debitAccount(@Param('tid') tid: string, @Body() req: Amount): Promise<TransactionResult> {
        Logger.log(`Token burning requested for account ${tid} (internal call): ${JSON.stringify(req)}`);
        try {
            return await this.membership.debitAccount(tid, req.amount);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/clearance-check')
    @HttpCode(204)
    @ApiOperation({
        summary: 'Check clearance for the given security context',
        description: `Verifies that all the entities in the given context are currently valid. In particular, the DID must match
  an active Onboarding Authority, the PID must match an onboarded Member, and no entities must be blacklisted. Note that PID and UID
  are optional, but DID is always required because it sets the context for both PID and UID.`
    })
    @ApiBody({
        type: SecurityContext,
        description: 'Contains the security context to be checked',
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The security context is valid`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: 'The security context is not valid' })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async checkSecurityContext(@Body() sc: SecurityContext): Promise<void> {
        try {
            await this.membership.checkSecurityContext(sc);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/users/offboarding-notification')
    @HttpCode(204)
    @ApiOperation({
        summary: 'Process user offboarding notification [INTERNAL VERSION]',
        description: `This operation is called to notify that a user has been offboarded (either internally or by an external trusted system).
  It blacklists the user and disenrolls all accounts owned by the user. This internal version does not perform authorization checks.`
    })
    @ApiBody({
        type: UserQualifiedId,
        description: 'The qualified identifier (OA DID + UID) of the user being offboarded',
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The user has been blacklisted and account disenrollment has been initiated`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async processUserOffboardingNotification(@Body() target: UserQualifiedId, @Headers('x-caller-context') callerContextHeader?: string): Promise<void> {
        Logger.log(`User offboarding notification received (internal call) for ${target.oa}:${target.uid}`);

        let callerContext = null;
        if (callerContextHeader) {
            try {
                callerContext = JSON.parse(callerContextHeader);
                Logger.log('Caller context received: ' + JSON.stringify(callerContext));
            } catch (err) {
                Logger.warn('Failed to parse caller context header: ' + err.message);
            }
        }

        // If no caller context provided, create default internal context
        if (!callerContext) {
            callerContext = {
                source: 'internal_api',
                endpoint: '/gov/v1.0/users/offboarding-notification',
                timestamp: Date.now()
            };
        }

        try {
            await this.membership.processUserAccountDisenrollment(target, callerContext);
        } catch (err) {
            translateToHttpException(err);
        }
    }
}
