import {
    BadRequestException,
    Body,
    Controller,
    Delete,
    Get,
    HttpCode,
    HttpException,
    HttpStatus,
    Logger,
    NotFoundException,
    Param,
    Post,
    Put,
    Query,
    UnauthorizedException,
    UseGuards,
} from '@nestjs/common';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiTags } from '@nestjs/swagger';
import { MembershipService } from './membership-service';
import { AuthGuard } from '../auth/auth-guard';
import { RolesGuard } from 'src/auth/roles-guard';
import { Roles } from 'src/auth/roles-decorator';
import { Identifier } from 'src/dtos/identifier';
import { ROLE_ADMIN, ROLE_OPERATOR, ROLE_SUPERUSER, ROLE_USER } from 'src/auth/auth-constants';
import { User } from 'src/auth/user-decorator';
import { UserInfo } from 'src/auth/usser-info';
import { MemberOnboardingRequest } from 'src/dtos/member-onboarding-request';
import { Member } from 'src/dtos/member';
import { MemberRef } from 'src/dtos/member-ref';
import { MemberFilter } from 'src/dtos/member-filter';
import { AccountStatus, EntityStatus, HTTP_ERR_400, HTTP_ERR_401, HTTP_ERR_403, HTTP_ERR_404, HTTP_ERR_500, HTTP_ERR_502, ListOperation, OrganizationType } from 'src/constants';
import { AccountIdentifier } from 'src/dtos/account-identifier';
import axios, { AxiosInstance } from 'axios';
import { ConfigService } from '@nestjs/config';
import { AccountFilter } from 'src/dtos/account-filter';
import { Account } from 'src/dtos/account';
import { ErrorCondition, throwExceptionWithErrorCondition, translateToHttpException } from 'src/utils';
import { AccountEnrolmentRequest } from 'src/dtos/account-enrolment-request';
import { MachineIdentityRequest } from 'src/dtos/machine-identity-request';
import { Amount } from 'src/dtos/amount';
import { LegalEntity } from 'src/dtos/legal-entity';
import { MemberUpdateRequest } from 'src/dtos/member-update-request';
import { UserBlacklistFilter } from 'src/dtos/user-blacklist-filter';
import { MemberBlacklistFilter } from 'src/dtos/member-blacklist-filter';
import { AuthorityBlacklistFilter } from 'src/dtos/authority-blacklist-filter';
import { UserBlacklistOp } from 'src/dtos/user-blacklist-op';
import { UserQualifiedId } from 'src/dtos/user-qualified-id';
import { CallerContext } from 'src/dtos/member-offboarding-request';
import { BlacklistService } from './blacklist-service';
import { Authority } from 'src/dtos/authority';
import { AuthorityFilter } from 'src/dtos/authority-filter';
import { TransactionResult } from 'src/dtos/transaction-result';
import { TransactionList } from 'src/dtos/transaction-list';
import { AccountBalance } from 'src/dtos/account-balance';

@ApiTags('Open API: Membership Management')
@Controller('/api/v1.0')
@ApiBearerAuth()
@UseGuards(AuthGuard, RolesGuard)
export class MembershipController {

    private readonly internalProxy: AxiosInstance = null;

    constructor(
        private readonly config: ConfigService,
        private readonly membership: MembershipService,
        private readonly blacklist: BlacklistService
    ) {
        this.internalProxy = axios.create({
            baseURL: this.config.get<string>('INTERNAL_URL'), // Base URL from environment variables
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
    }

    @Post('/members')
    @HttpCode(202)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Starts the process of onboarding of a new member',
        description: `This operation starts the onboarding process for a new member. The operation immediately returns the
  assigned PID, but the actual onboarding process is executed offline: the caller will have to check later for its outcome.
  This operation is only available to administrators.`,
    })
    @ApiBody({
        type: MemberOnboardingRequest,
        description: `The request body contains the information required to onboard a new member.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The onboarding process has been started and will be executed offline. The response body contains the PID
  that will be assigned to the new member is the process is successful: this reference can be used to check the actual status of
  the onboarding. If the process is unsuccessful, the PID can be discarded as no persistent information linked to it will be kept.`,
        type: Identifier
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async onboardMember(@User() user: UserInfo, @Body() req: MemberOnboardingRequest): Promise<Identifier> {
        Logger.log('Member onboarding requested: ' + JSON.stringify(req));
        Logger.log('Calling user: ' + JSON.stringify(user));
        Logger.log('Call forwarded to INTERNAL service endpoint');
        // We need to delegate to the GOV INTERNAL process because:
        // 1. This is a steteful workflow, and the state is managed in-memory.
        // 2. The second and final step of the workflow will run within the internal process,
        //    because it is triggered by P&T module calling a GOV internal service endpoint.
        // If we don't delegate, we will end up having the workflow state residing this process
        // instead of the internal one, and the callback will fail.
        try {
            const { data } = await this.internalProxy.post('/members', req);
            return data;
        } catch (error) {
            if (error.response) {
                // Forward the exact status and error message from the external service.
                throw new HttpException(error.response.data, error.response.status);
            }
            // Fallback for errors that don't have a response.
            throw new HttpException('Internal server error', HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @Put('/members/:pid')
    @HttpCode(204)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Updates the non-trusted information for a member',
        description: `This operation updates the non-trusted information for member, such as the organization type,
  contact info and link to the Onboarding Authority - i.e., everything that is NOT stored as a Blockchain record (see the
  "migrate member" operation for applying changes that affect Blockchain data). This operation is only available to administrators.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member to update'
    })
    @ApiBody({
        type: MemberUpdateRequest,
        description: `The request body contains the new non-trusted information for the member. At least one field must be provided.
  Anything that is not provided will be left unchanged. Note: Only the representative contact info (rep) can be cleared by setting it to null explicitly.
  Organization type and OA are required fields and cannot be set to null.`,
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: 'The operation was completed successfully'
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async updateMember(@User() user: UserInfo, @Param('pid') pid: string, @Body() req: MemberUpdateRequest): Promise<void> {
        Logger.log('Member update requested for PID ' + pid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        try {
            await this.membership.updateMember(pid, req);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/members/:pid/migrate')
    @HttpCode(202)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Starts the process of changing the legal entity information of an existing member',
        description: `This operation starts the migration process for an existing member to a new legal entity. The process is executed
  offline and the caller will have to check later for its outcome. This operation is only available to administrators.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member to migrate'
    })
    @ApiBody({
        type: LegalEntity,
        description: `The request body contains the new legal entity information for the member. This information will be be stored as a new
  Blockchain record linked to the member.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The migration process has been started and will be executed offline. The response body contains the new PID assigned
  to the member. Note however that the migration process is not instantaneous, and the caller will have to check later for its outcome: if
  unsuccessful, no changes will be applied to the current data of the member, and the original PID will remain valid.`,
        type: Identifier
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async migrateMember(@User() user: UserInfo, @Param('pid') pid: string, @Body() req: LegalEntity): Promise<Identifier> {
        Logger.log('Member migration requested for PID ' + pid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        Logger.log('Call forwarded to INTERNAL service endpoint');
        // We need to delegate to the GOV INTERNAL process because:
        // 1. This is a stateful workflow, and the state is managed in-memory.
        // 2. The second and final step of the workflow will run within the internal process,
        //    because it is triggered by P&T module calling a GOV internal service endpoint.
        // If we don't delegate, we will end up having the workflow state residing this process
        // instead of the internal one, and the callback will fail.
        try {
            const { data } = await this.internalProxy.post(`/members/${pid}/migrate`, req);
            return data;
        } catch (error) {
            if (error.response) {
                // Forward the exact status and error message from the external service.
                throw new HttpException(error.response.data, error.response.status);
            }
            // Fallback for errors that don't have a response.
            throw new HttpException('Internal server error', HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @Delete('/members/:pid')
    @HttpCode(202)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Starts the offboarding of an existing member',
        description: `This operation starts the offboarding process for an existing member. This operation is irreversible, although the member data
  will be kept for auditing purposes. The process is executed offline and the caller will have to check later for its outcome.
  This operation is only available to administrators.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member to offboard'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The offboarding process has been started and will be executed offline`
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async offboardMember(@User() user: UserInfo, @Param('pid') pid: string): Promise<void> {
        Logger.log('Member offboarding requested for PID ' + pid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        Logger.log('Call forwarded to INTERNAL service endpoint');
        // See comment in onboardMember() for explanation of why we need to delegate this operation.
        try {
            // Pass caller context via headers
            const callerContext = {
                source: 'external_api',
                userInfo: {
                    uid: user.uid,
                    issuerDID: user.issuerDID,
                    role: user.role,
                    affiliation: user.affiliation,
                    region: user.region,
                    nickname: user.nickname
                },
                endpoint: '/api/v1.0/members/{pid}',
                timestamp: Date.now()
            };

            await this.internalProxy.delete('/members/' + pid, {
                headers: {
                    'X-Caller-Context': JSON.stringify(callerContext)
                }
            });
        } catch (error) {
            if (error.response) {
                // Forward the exact status and error message from the external service.
                throw new HttpException(error.response.data, error.response.status);
            }
            // Fallback for errors that don't have a response.
            throw new HttpException('Internal server error', HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @Get('/members')
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'List members',
        description: `This operation provides the list of registered members, which can be filtered by various parameters.
  This operation is only available to administrators and operators.`
    })
    @ApiQuery({
        name: 'nam',
        required: false,
        type: String,
        description: 'Legal name to filter members by (partial match, case insensitive)'
    })
    @ApiQuery({
        name: 'typ',
        required: false,
        enum: OrganizationType,
        description: `Organization type to filter members by, accepted values: [${Object.values(OrganizationType).join(', ')}]`,
    })
    @ApiQuery({
        name: 'cnt',
        required: false,
        type: String,
        description: 'Country code to filter members by'
    })
    @ApiQuery({
        name: 'sta',
        required: false,
        enum: EntityStatus,
        description: `Status to filter members by, accepted values: [${Object.values(EntityStatus).join(', ')}]`,
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains a list of references to members`,
        type: MemberRef,
        isArray: true
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listMembers(@Query() query?: MemberFilter): Promise<MemberRef[]> {
        return await this.membership.listMembers(query);
    }

    @Get('/members/:pid')
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Resolve the PID of a member',
        description: `This operation resolves the PID of a member. The information set includes both trusted
  (Blockchain, from P&T module) and non-trusted (local DB) data. This operation is only available to administrators and operators.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID to be resolved'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains the full information set of the target member`,
        type: Member
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async retrieveMember(@Param('pid') pid: string): Promise<Member> {
        return await this.membership.retrieveMember(pid);
    }

    @Get('/active-members')
    @HttpCode(200)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_ADMIN, ROLE_OPERATOR)
    @ApiOperation({
        summary: 'List active members',
        description: `This operation provides the full list of active members.`
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains a list of references to members`,
        type: MemberRef,
        isArray: true
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listActiveMembers(@Query() pagination?: MemberFilter): Promise<MemberRef[]> {
        const query = new MemberFilter();
        query.sta = EntityStatus.ACTIVE;
        if (pagination?.l !== undefined) query.l = pagination.l;
        if (pagination?.o !== undefined) query.o = pagination.o;
        return await this.membership.listMembers(query);
    }

    @Get('/active-members/:pid')
    @HttpCode(200)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_ADMIN, ROLE_OPERATOR)
    @ApiOperation({
        summary: 'Resolve the PID of an active member',
        description: `This operation resolves a PID into the name and basic details of an active member.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID to be resolved'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains the basic details of the target member`,
        type: MemberRef
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveActiveMemberRef(@Param('pid') pid: string): Promise<MemberRef> {
        const member = await this.membership.retrieveActiveMemberRef(pid);
        if (!member) {
            throw new NotFoundException('Member not found');
        }
        return member;
    }

    @Get('/authorities')
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'List authorities',
        description: `This operation provides the list of registered authorities, which can be filtered by various parameters.
  This operation is only available to administrators and operators.`
    })
    @ApiQuery({
        name: 'own',
        required: false,
        type: String,
        description: 'Owner PID to filter authorities by'
    })
    @ApiQuery({
        name: 'sta',
        required: false,
        enum: EntityStatus,
        description: `Status to filter authorities by, accepted values: [${Object.values(EntityStatus).join(', ')}]`,
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains a list of authorities`,
        type: Authority,
        isArray: true
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listAuthorities(@Query() query?: AuthorityFilter): Promise<Authority[]> {
        return await this.membership.listAuthorities(query);
    }

    @Get('/authorities/:did')
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Resolve the DID of an authority',
        description: `This operation resolves the DID of an authority.
  This operation is only available to administrators and operators.`
    })
    @ApiParam({
        name: 'did',
        required: true,
        type: String,
        description: 'DID to be resolved'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains the details of the target authority`,
        type: Authority
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async retrieveAuthority(@Param('did') did: string): Promise<Authority> {
        return await this.membership.retrieveAuthority(did);
    }

    @Post('/accounts')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Starts the process of enrolling a new trading account',
        description: `This operation starts the enrolment process for a new trading account. The operation
  immediately returns a response indicating that the request has been accepted, but the actual processing is
  executed offline: the caller will have to check later for its outcome. Once enrolled, the account will be
  exclusively owned the caller.`
    })
    @ApiBody({
        type: AccountIdentifier,
        description: 'The request body contains the TID of the trading account to be enrolled',
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: 'The enrolment process has been started and will be executed offline',
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async enrolAccount(@User() user: UserInfo, @Body() args: AccountIdentifier): Promise<void> {
        Logger.log('Account enrolment requested: ' + args.tid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        Logger.log('Call forwarded to INTERNAL service endpoint');

        // See comment in onboardMember() for explanation of why we need to delegate this operation.
        const req = new AccountEnrolmentRequest();
        req.tid = args.tid;
        req.did = user.issuerDID;
        req.uid = user.uid;
        if (user.affiliation && user.affiliation.trim().length > 0) {
            req.affiliation = user.affiliation.trim();
        }
        try {
            await this.internalProxy.post('/accounts', req);
        } catch (error) {
            if (error.response) {
                // Forward the exact status and error message from the external service.
                throw new HttpException(error.response.data, error.response.status);
            }
            // Fallback for errors that don't have a response.
            throw new HttpException('Internal server error', HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @Delete('/accounts/:tid')
    @HttpCode(202)
    @ApiOperation({
        summary: 'Starts the process of disenrolling an existing trading account',
        description: `This operation starts the disenrolment process for an existing trading account. The operation
  immediately returns a response indicating that the request has been accepted, but the actual processing is
  executed offline: the caller will have to check later for its outcome. This operation cannot be reverted.
   If the caller is an operator or an administrator, the target account can be disenrolled regardless of its owner;
  otherwise, only accounts owned by the caller can be disenrolled.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: 'The offboarding process has been started and will be executed offline',
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async disenrolAccount(@User() user: UserInfo, @Param('tid') tid: string): Promise<void> {
        Logger.log('Account disenrolment requested for TID ' + tid);
        Logger.log('Calling user: ' + JSON.stringify(user));

        // Check ownership consistency here, bacause we will not forward the user info to the internal service
        let ok: boolean = false;
        if (user.role === ROLE_ADMIN || user.role === ROLE_OPERATOR) {
            // If the caller is an admin or an operator, we just check the account status
            ok = await this.membership.checkAccountStatus(tid, true);
        } else {
            // If the caller is a user, we ALSO check that the account is owned by the caller
            ok = await this.membership.checkAccountOwnership(tid, user.issuerDID, user.uid);
        }
        if (!ok) {
            Logger.warn('Bad target: disenrolment request rejected');
            throw new NotFoundException('Trading account does not exist');
        }

        // See comment in onboardMember() for explanation of why we need to delegate this operation.
        Logger.log('Call forwarded to INTERNAL service endpoint');
        try {
            await this.internalProxy.delete('/accounts/' + tid);
        } catch (error) {
            if (error.response) {
                // Forward the exact status and error message from the external service.
                throw new HttpException(error.response.data, error.response.status);
            }
            // Fallback for errors that don't have a response.
            throw new HttpException('Internal server error', HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @Get('/active-accounts')
    @ApiOperation({
        summary: 'List active trading accounts owned by the caller',
        description: `This operation provides the list of active trading accounts owned by the caller.`
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The response body contains a list of trading accounts',
        type: Account,
        isArray: true
    })
    async listOwnedAccounts(@User() user: UserInfo, @Query('l') l?: number, @Query('o') o?: number): Promise<Account[]> {
        const pagination = (l !== undefined || o !== undefined) ? { l, o } : undefined;
        return await this.membership.listActiveAccountForUser(user, pagination);
    }

    @Get('/accounts')
    @ApiOperation({
        summary: 'List all trading accounts',
        description: `This operation provides the list of registered (both active and not active) trading accounts,
  which can be filtered by various parameters. If the caller is an operator or an administrator, all accounts are visible;
  otherwise, only accounts owned by the caller are visible.`
    })
    @ApiQuery({
        name: 'usr',
        required: false,
        type: String,
        description: 'Owner UID (user) to filter accounts by'
    })
    @ApiQuery({
        name: 'own',
        required: false,
        type: String,
        description: 'Owner PID (user affiliation) to filter accounts by'
    })
    @ApiQuery({
        name: 'aut',
        required: false,
        type: String,
        description: 'Onboarding Authority DID (user onboarder) to filter accounts by'
    })
    @ApiQuery({
        name: 'sta',
        required: false,
        enum: EntityStatus,
        description: `Status to filter accounts by, accepted values: [${Object.values(EntityStatus).join(', ')}]`,
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The response body contains a list of trading accounts',
        type: Account,
        isArray: true
    })
    async listAllAccounts(@User() user: UserInfo, @Query() query?: AccountFilter): Promise<Account[]> {
        if (user && user.role !== ROLE_OPERATOR && user.role !== ROLE_ADMIN) {
            return await this.membership.listAccounts(query, user); // profiled operation
        } else {
            return await this.membership.listAccounts(query); // unprofiled operation
        }
    }

    @Get('/accounts/:tid')
    @ApiOperation({
        summary: 'Resolves the TID of a trading account',
        description: `This operation resolves the TID of a trading account. If the caller is an operator or an administrator,
  the operation works on any account; otherwise, it will work only for accounts owned by the caller.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID to be resolved'
    })
    @ApiResponse({
        status: 200,
        description: 'The response body contains the details of the target trading account',
        type: Account
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveAccount(@User() user: UserInfo, @Param('tid') tid: string): Promise<Account> {
        let account: Account = null;
        if (user && user.role !== ROLE_OPERATOR && user.role !== ROLE_ADMIN) {
            account = await this.membership.retrieveAccount(tid, user); // profiled operation
        } else {
            account = await this.membership.retrieveAccount(tid);
        }
        if (account) {
            return account;
        } else {
            throw new NotFoundException('Trading account not found');
        }
    }

    @Get('/accounts/:tid/balance')
    @ApiOperation({
        summary: 'Gets the current balance of a trading account',
        description: `This operation returns the current balance and status of a trading account. If the caller is an operator or an administrator,
  the operation works on any account; otherwise, it will work only for accounts owned by the caller.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the target trading account'
    })
    @ApiResponse({
        status: 200,
        description: 'The response body contains the balance and status of the target trading account',
        type: AccountBalance
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async retrieveAccountBalance(@User() user: UserInfo, @Param('tid') tid: string): Promise<AccountBalance> {
        const account = await this.retrieveAccount(user, tid); // not found exception will be raised if the account does not exist or is not accessible
        try {
            return {
                balance: await this.membership.retrieveAccountBalance(tid),
                status: account.disenrolled ? AccountStatus.INACTIVE : AccountStatus.ACTIVE
            };
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get('/accounts/:tid/transactions')
    @HttpCode(200)
    @ApiOperation({
        summary: 'Retrieves the token management transaction history for a given trading account.',
        description: `This operation retrieves all token management transactions (credit and debit operations) for the specified trading account,
  listed in descending chronological order. If the caller is an operator or an administrator, the operation works on any account; otherwise, it
  will only work for accounts owned by the caller.`
    })
    @ApiParam({
        name: 'tid',
        required: true,
        type: String,
        description: 'TID of the trading account'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The request has been processed successfully',
        type: TransactionList
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listAccountTransactions(@User() user: UserInfo, @Param('tid') tid: string): Promise<TransactionList> {
        await this.retrieveAccount(user, tid); // not found exception will be raised if the account does not exist or is not accessible
        try {
            return await this.membership.listAccountTransactions(tid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/accounts/:tid/credit')
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Mints a given amount of FDE tokens and transfers them to a given trading account.',
        description: `This operation mints a given amount of FDE tokens and transfers them to the specified trading account.
  The operation is only available to administrators and operators. Upon successful completion, the operation returns
  a transaction record including a unique transaction ID and timestamp.`
    })
    @ApiBody({
        type: Amount,
        description: `Amount of FDE tokens to be minted and transferred (REQUIRED)`,
        required: true,
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The request has been processed successfully',
        type: TransactionResult
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async creditAccount(@User() user: UserInfo, @Param('tid') tid: string, @Body() req: Amount): Promise<TransactionResult> {
        Logger.log(`Token minting requested for account ${tid}: ${JSON.stringify(req)}`);
        Logger.log('Calling user: ' + JSON.stringify(user));
        try {
            return await this.membership.creditAccount(tid, req.amount);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/accounts/:tid/debit')
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Burns a given amount of FDE tokens in a given trading account.',
        description: `This operation burns a given amount of FDE tokens in the specified trading account.
  The operation is only available to administrators and operators. Upon successful completion, the operation returns
  a transaction record including a unique transaction ID and timestamp.`
    })
    @ApiBody({
        type: Amount,
        description: `Amount of FDE tokens to be burned (REQUIRED)`,
        required: true,
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The request has been processed successfully',
        type: TransactionResult
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async debitAccount(@User() user: UserInfo, @Param('tid') tid: string, @Body() req: Amount): Promise<TransactionResult> {
        Logger.log(`Token burning requested for account ${tid}: ${JSON.stringify(req)}`);
        Logger.log('Calling user: ' + JSON.stringify(user));
        try {
            return await this.membership.debitAccount(tid, req.amount);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/offboarding-notifications')
    @HttpCode(204)
    @Roles(ROLE_ADMIN, ROLE_SUPERUSER)
    @ApiOperation({
        summary: 'Receives the notification that a user has been offboarded by their Onboarding Authority',
        description: `Disenrols all accounts, if any, owned by a given user and adds it to the user blacklist.
        User in the blacklist are locked out at login. This operation is only available to administrators and to superusers.
        The latter can only send notifications for users that have been onboarded by the organization they are affiliated to.`
    })
    @ApiBody({
        type: UserQualifiedId,
        description: `The request body contains a reference to a user, which is identified by their UID and by the DID of their Onboarding Authoritity.`,
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The operation was completed successfully.`,
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async processUserOffboardingNotification(@User() user: UserInfo, @Body() target: UserQualifiedId): Promise<void> {
        // retireve the OA of the target user by resolving its DID reference
        const oa = await this.membership.retrieveAuthorityOwnerRef(target.oa);
        if (oa) {
            // the reference is valid: it matches an active OA
            // (we cannot determine if the UID is valid or not, but we don't care)
            if (user.role === ROLE_SUPERUSER && user.affiliation !== oa.pid) {
                // if the caller is not an admin user, it must be a superuser affiliated to the target user's OA
                // (meaning: it must have authority over the target user)
                throw new UnauthorizedException('This operation is not permitted');
            }
        } else {
            // the reference is not valid: it is linked to an an unknown or offboarded OA
            throw new BadRequestException("Unknown OA");
        }
        try {
            // Create caller context for the disenrollment operation
            const callerContext: CallerContext = {
                source: 'external_api',
                endpoint: '/api/v1.0/users/offboarding-notification',
                timestamp: Date.now(),
                userInfo: user
            };

            // Process user account disenrollment (blacklist + disenroll accounts)
            await this.membership.processUserAccountDisenrollment(target, callerContext);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/blacklists/users')
    @HttpCode(204)
    @Roles(ROLE_ADMIN)
    @ApiOperation({
        summary: 'Adds/removes a user to/from the blacklist',
        description: `Depending on the argument provided, either adds a new entry to the user blacklist, or removes an existing entry from it.
  User in the blacklist are locked out at login. This operation is only available to administrators.`
    })
    @ApiBody({
        type: UserBlacklistOp,
        description: `The request body contains a reference to a user, which is identified by their UID and by the DID of their Onboarding Authoritity,
  and the command to be executed on the blacklist (either "ADD" or "REMOVE"). The specified Onboarding Authority must exist and be active.`,
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The operation was completed successfully.`,
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async updateUserBlacklist(@User() user: UserInfo, @Body() op: UserBlacklistOp): Promise<void> {
        // resolve the DID reference of the target's OA
        const owner = await this.membership.retrieveAuthorityOwnerRef(op.target.oa);
        if (!owner) {
            throw new BadRequestException("Bad Onboarding Authority");
        }
        const id: UserQualifiedId = op.target;
        if (op.operation === ListOperation.ADD) {
            try {
                await this.blacklist.addUserBan(id.oa, id.uid);
            } catch (err) {
                translateToHttpException(err);
            }
        } else if (op.operation === ListOperation.REMOVE) {
            let removed: boolean = false;
            try {
                removed = await this.blacklist.removeUserBan(id.oa, id.uid);
            } catch (err) {
                translateToHttpException(err);
            }
            if (!removed) {
                throw new NotFoundException('User not found in blacklist');
            }
        } else {
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, "Unknown operation");
        }
    }

    @Get('/blacklists/users')
    @Roles(ROLE_ADMIN) // privileged operation
    @HttpCode(200)
    @ApiOperation({
        summary: 'Retrieves the blacklist',
        description: `Returns the full contents of the blacklist.`
    })
    @ApiQuery({
        name: 'oa',
        required: false,
        type: String,
        description: `Onboarding Authority DID to filter entries by`,
    })
    @ApiQuery({
        name: 'uid',
        required: false,
        type: String,
        description: 'User UID to filter entries by',
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains an array of fully-qualified user IDs.`,
        type: UserQualifiedId,
        isArray: true
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async getUserBlacklist(@Query() query?: UserBlacklistFilter): Promise<UserQualifiedId[]> {
        try {
            return this.blacklist.listBannedUsers(query);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/blacklists/members')
    @HttpCode(204)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Adds a member to the blacklist',
        description: `Adds a new entry to the member blacklist. User affiliated to a blacklisted member are locked out at login.
  This operation is only available to administrators.`
    })
    @ApiBody({
        type: Identifier,
        description: `The request body contains a reference to a member, which is identified by its PID.
  The specified member must exist and have onboarded status.`,
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The operation was completed successfully.`,
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async addMemberToBlacklist(@Body() target: Identifier): Promise<void> {
        const member = await this.membership.retrieveActiveMemberRef(target.id);
        if (!member) {
            throw new BadRequestException("Unknown member");
        }
        try {
            await this.blacklist.addMemberBan(target.id);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Delete('/blacklists/members/:pid')
    @HttpCode(204)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Removes a member from the blacklist',
        description: `Removes an existing entry from the member blacklist. This operation is only available to administrators.`
    })
    @ApiParam({
        name: 'pid',
        required: true,
        type: String,
        description: 'PID of the member to remove from the blacklist'
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The operation was completed successfully.`,
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async removeMemberFromBlacklist(@Param('pid') pid: string): Promise<void> {
        let removed: boolean = false;
        try {
            removed = await this.blacklist.removeMemberBan(pid);
        } catch (err) {
            translateToHttpException(err);
        }
        if (!removed) {
            throw new NotFoundException('Member not found in blacklist');
        }
    }

    @Get('/blacklists/members')
    @Roles(ROLE_ADMIN) // privileged operation
    @HttpCode(200)
    @ApiOperation({
        summary: 'Retrieves the member blacklist',
        description: `Returns the full contents of the member blacklist.`
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains an array of member identifiers (PID)`,
        type: Identifier,
        isArray: true
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async getMemberBlacklist(@Query() query?: MemberBlacklistFilter): Promise<Identifier[]> {
        try {
            return this.blacklist.listBannedMembers(query);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/blacklists/authorities')
    @HttpCode(204)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Adds an authority to the blacklist',
        description: `Adds a new entry to the authority blacklist. User onboarded by a blacklisted authority are locked out at login.
  This operation is only available to administrators.`
    })
    @ApiBody({
        type: Identifier,
        description: `The request body contains a reference to an authority, which is identified by its DID.
  The specified authority must exist and have active status.`,
        required: true,
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The operation was completed successfully.`,
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async addAuthorityToBlacklist(@Body() target: Identifier): Promise<void> {
        const authority = await this.membership.retrieveAuthority(target.id, true);
        if (!authority) {
            throw new BadRequestException("Unknown authority");
        }
        try {
            await this.blacklist.addAuthorityBan(target.id);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Delete('/blacklists/authorities/:did')
    @HttpCode(204)
    @Roles(ROLE_ADMIN) // privileged operation
    @ApiOperation({
        summary: 'Removes an authority from the blacklist',
        description: `Removes an existing entry from the authority blacklist. This operation is only available to administrators.`
    })
    @ApiParam({
        name: 'did',
        required: true,
        type: String,
        description: 'DID of the authority to remove from the blacklist'
    })
    @ApiResponse({
        status: 204, // No Content
        description: `The operation was completed successfully.`,
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async removeAuthorityFromBlacklist(@Param('did') did: string): Promise<void> {
        let removed: boolean = false;
        try {
            removed = await this.blacklist.removeAuthorityBan(did);
        } catch (err) {
            translateToHttpException(err);
        }
        if (!removed) {
            throw new NotFoundException('Authority not found in blacklist');
        }
    }

    @Get('/blacklists/authorities')
    @Roles(ROLE_ADMIN) // privileged operation
    @HttpCode(200)
    @ApiOperation({
        summary: 'Retrieves the authority blacklist',
        description: `Returns the full contents of the authority blacklist.`
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The response body contains an array of authority identifiers (PID)`,
        type: Identifier,
        isArray: true
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async getAuthorityBlacklist(@Query() query?: AuthorityBlacklistFilter): Promise<Identifier[]> {
        try {
            return this.blacklist.listBannedAuthorities(query);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/machine-identities')
    @HttpCode(201)
    @Roles(ROLE_ADMIN, ROLE_SUPERUSER) // privileged operation
    @ApiOperation({
        summary: 'Creates a machine identity token',
        description: `Generates and retrieves JWT with a long expiration time for accessing the Marketplace API and UI.
  JWT generated here have a fixed "machine_identity" UID. This operation is only available to administrators or to super-users.
  The latter can only generate tokens with user attributes (affiliation and region) matching those of their organization.`
    })
    @ApiBody({
        type: MachineIdentityRequest,
        description: `Attributes of the requested JWT`,
        required: true,
    })
    @ApiResponse({
        status: 201, // Created
        type: String
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async createMachineIdentity(@User() user: UserInfo, @Body() req: MachineIdentityRequest): Promise<string> {
        Logger.log('Requesting new machine identity: ' + JSON.stringify(req));
        Logger.log('Calling user: ' + JSON.stringify(user));
        if (user.role === ROLE_SUPERUSER) {
            // ignoring any custom parameters sent by the caller
            const member = await this.membership.retrieveActiveMemberRef(user.affiliation);
            if (member) {
                req.affiliation = member.pid;
                req.region = member.country;
                req.role = ROLE_USER;
            } else {
                // the caller is not affiliated with an active member !
                throw new UnauthorizedException("No active affiliation for the calling user");
            }
        }
        try {
            return await this.membership.createMachineIdentity(req.affiliation, req.role, req.region);
        } catch (err) {
            translateToHttpException(err);
        }
    }
}
