import { OrganizationType } from 'src/constants';
import {
    Entity,
    PrimaryColumn,
    Column,
    ManyToOne,
    OneToMany,
    JoinColumn,
    Index,
} from 'typeorm';

@Entity('member')
export class MemberEntity {
    @PrimaryColumn({ type: 'char', length: 22, nullable: false, update: false })
    pid: string;

    @Column({ type: 'varchar', length: 160, nullable: false, update: false })
    nam: string;

    @Column({ type: 'char', length: 2, nullable: false, update: false })
    cnt: string;

    @Column({
        type: 'enum',
        enum: OrganizationType,
        default: OrganizationType.UKN,
        nullable: false,
    })
    typ: OrganizationType;

    // If we use eager loading, this will throw typeorm in an infinite loop
    @ManyToOne(() => MemberEntity, { nullable: true, lazy: true })
    @JoinColumn({ name: 'aut', referencedColumnName: 'pid' })
    onboarder?: Promise<MemberEntity>;
    
    @Column({ type: 'varchar', length: 1000, nullable: true })
    rep?: string;

    @Column({ type: 'datetime', nullable: false, update: false })
    onb: Date;

    @Column({ type: 'datetime', nullable: true })
    ofb?: Date;

    // Lists all the authorities owned by the member, regardless of their status.
    // Note that only one active authority is allowed at any time, but here all authorities are listed.
    // Parent-to-children relation with lazy loading.
    @OneToMany(() => AuthorityEntity, authority => authority.owner)
    authorities?: Promise<AuthorityEntity[]>;

    // Lists all the accounts directly owned by the member, regardless of their status.
    // Parent-to-children relation with lazy loading.
    @OneToMany(() => AccountEntity, account => account.owner)
    accounts?: Promise<AccountEntity[]>;
}

@Entity('authority')
export class AuthorityEntity {
    @PrimaryColumn({ type: 'varchar', length: 255, nullable: false, update: false })
    did: string;

    // Points to the member who owns the authority.
    // Child-to-parent relation with eager loading.
    @ManyToOne(() => MemberEntity, member => member.authorities, { nullable: false, eager: true })
    @JoinColumn({ name: 'own' })
    owner: MemberEntity;

    @Column({ type: 'datetime', nullable: false, update: false })
    act: Date;

    @Column({ type: 'datetime', nullable: true })
    dct?: Date;

    // Lists all the accounts indirectly linked to this authority, regardless of their status
    // Parent-to-children relation with lazy loading.
    @OneToMany(() => AccountEntity, account => account.authority)
    accounts?: Promise<AccountEntity[]>;
}

@Entity('account')
export class AccountEntity {
    @PrimaryColumn({ type: 'char', length: 42, nullable: false, update: false })
    tid: string;

    // We set the maximum length to 255 characters because the length of externally-assigned UIDs is unknonwn
    @Column({ type: 'varchar', length: 255, nullable: false, update: false })
    usr: string;

    // Points to the member who directly owns the account, if any (for non-affilated users, there is no owner)
    // Child-to-parent relation (nullable) with eager loading
    @ManyToOne(() => MemberEntity, member => member.accounts, { nullable: true, eager: true })
    @JoinColumn({ name: 'own' })
    owner?: MemberEntity;

    // Points to the member who (indirectly, as the user's onboarding authority) is reponsible for the account
    // Child-to-parent relation (nullable) with eager loading
    @ManyToOne(() => AuthorityEntity, authority => authority.accounts, { nullable: true, eager: true })
    @JoinColumn({ name: 'aut' })
    authority: AuthorityEntity;

    @Column({ type: 'datetime', nullable: false, update: false })
    act: Date;

    @Column({ type: 'datetime', nullable: true })
    dct?: Date;
}

@Entity('transaction')
@Index('idx_transaction_tms', ['tms'])
@Index('idx_transaction_acc', ['acc'])
export class TransactionEntity {
    @PrimaryColumn({ type: 'char', length: 36, nullable: false, update: false })
    trx: string;

    @Column({ type: 'datetime', nullable: false, update: false })
    tms: Date;

    @Column({ type: 'char', length: 42, nullable: false, update: false })
    acc: string;

    @Column({ type: 'tinyint', nullable: false, update: false })
    dir: number;

    @Column({ type: 'decimal', precision: 20, scale: 2, nullable: false, update: false })
    amt: string;
}
