import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { MemberRef } from '../dtos/member-ref';
import { MembershipStorage } from './membership-storage';
import { MemberOnboardingRequest } from 'src/dtos/member-onboarding-request';
import { Identifier } from 'src/dtos/identifier';
import { ErrorCondition, isDID, isShortUUID, throwExceptionWithErrorCondition } from 'src/utils';
import { Member } from 'src/dtos/member';
import { ProcessState } from '../integration/process-state';
import { BackgroundJobState, BackgroundJobContext } from '../integration/background-job-state';
import { BackgroundJobNotifier } from '../integration/background-job-notifier';
import axios, { AxiosInstance, AxiosResponse } from 'axios';
import { LegalEntity } from 'src/dtos/legal-entity';
import { AccountEntity, AuthorityEntity, MemberEntity } from 'src/membership/membership-entities';
import { Source } from 'src/dtos/source';
import { plainToInstance } from 'class-transformer';
import { ContactInfo } from 'src/dtos/contact-info';
import { MemberFilter } from 'src/dtos/member-filter';
import { EntityStatus, ROA_PID } from 'src/constants';
import { AccountFilter } from 'src/dtos/account-filter';
import { Account } from 'src/dtos/account';
import { AccountEnrolmentRequest } from 'src/dtos/account-enrolment-request';
import { AccountIdentifier } from 'src/dtos/account-identifier';
import { AuthorityFilter } from 'src/dtos/authority-filter';
import { Authority } from 'src/dtos/authority';
import { UserInfo } from 'src/auth/usser-info';
import * as short from 'short-uuid';
import { MemberOffboardingRequest } from 'src/dtos/member-offboarding-request';
import { MemberMigrationRequest } from 'src/dtos/member-migration-request';
import { SecurityContext } from 'src/dtos/security-context';
import { MemberUpdateRequest } from 'src/dtos/member-update-request';
import { BlacklistService } from './blacklist-service';
import { AccountDisenrolmentRequest } from 'src/dtos/account-disenrolment-request';
import { CallerContext } from 'src/dtos/member-offboarding-request';
import { UserQualifiedId } from 'src/dtos/user-qualified-id';
import { OAChangeContext } from 'src/dtos/oa-change-context';
import { v4 as uuidv4 } from 'uuid';

@Injectable()
export class MembershipService {

    private readonly ptProxy: AxiosInstance = null;
    private readonly tmProxy: AxiosInstance = null;
    private readonly aaiProxy: AxiosInstance = null;

    constructor(
        private readonly config: ConfigService,
        private readonly blacklist: BlacklistService,
        private readonly db: MembershipStorage,
        private readonly requests: ProcessState,
        private readonly backgroundJobs: BackgroundJobState,
        private readonly jobNotifier: BackgroundJobNotifier
    ) {
        this.ptProxy = axios.create({
            baseURL: this.config.get<string>('PT_URL'),
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
        this.tmProxy = axios.create({
            baseURL: this.config.get<string>('TM_URL'),
            timeout: 15000, // 15 seconds timeout for every request (this is a long one!)
            headers: { 'Content-Type': 'application/json' },
        });
        this.aaiProxy = axios.create({
            baseURL: this.config.get<string>('JWT_URL'),
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });

        // Set up background job failure notification callback
        this.backgroundJobs.setJobCompletionCallback((job) => {
            this.jobNotifier.notifyJobCompletedWithFailures(job);
        });
    }

    async createMachineIdentity(affiliation: string, role: string, region: string): Promise<string> {
        let response = null;
        try {
            response = await this.aaiProxy.post('', { affiliation: affiliation, role: role, region: region });
        } catch (err) {
            Logger.error(`Error calling AAI machine identity service: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to AAI machine identity service failed')
        }

        return JSON.stringify(response.data);
    }

    async startOnboardingProcess(req: MemberOnboardingRequest): Promise<Identifier> {

        // Before doing anything with persistent effects on external systems
        // (in this case, the P&T module), we do a careful validation of user input:
        // specifically, we want to ensure that the provided reference to the Onboarding Authority
        // (either a DID or a PID) is valid according to our rules. The same checks will be
        // repeated once again at the start of the second phase, but at that time the trusted
        // source will be already registered, and there would be no turning back!
        await this.checkAuthorityReference(req.oa);

        // PID is assigned randomly
        const pid: string = short.generate();
        Logger.debug('Onboarding member, assigned PID is ' + pid);

        // Instruct the P&T module to register this member as a trusted source.
        // This is the first step of the onboarding process: the second and
        // final one will be triggred by the P&T module calling our callback
        // function sourceRegistrationConfirmed().
        await this.requestSourceRegistration(pid, req);

        // Return the assigned PID to the caller: they will have to check back later
        // if the overall onboariding process was completed successfully.
        return new Identifier(pid);
    }

    async startOffboardingProcess(pid: string, callerContext?: CallerContext): Promise<void> {

        // Intercept attempts to offboard the ROA: this is not allowed.
        if (pid === ROA_PID) {
            Logger.warn('Attempt at offboarding the ROA blocked');
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Offboarding the ROA is not allowed');
        }

        // Before doing anything with persistent effects on external systems
        // (in this case, the P&T module), we check that the target is valid.
        await this.checkMembershipStatus(pid);

        // We also check that the target member is not set as the OA of active members, beside itself.
        // Note that is that is the case, the other members must be either offboarded or switched
        // to a different OA before we can proceed with the offboarding of this member.
        await this.checkMembershipDependencies(pid);

        // Instruct the P&T module to unregister this member as a trusted source.
        // This is the first step of the onboarding process: the second and
        // final one will be triggred by the P&T module calling our callback
        // function sourceDeregistrationConfirmed().
        await this.requestSourceDeregistration(pid, callerContext);
    }

    async startMigrationProcess(pid: string, req: LegalEntity, callerContext?: CallerContext): Promise<Identifier> {

        // Intercept attempts to migrate the ROA: this is not allowed.
        if (pid === ROA_PID) {
            Logger.warn('Attempt at migrating the ROA blocked');
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Migrating the ROA is not allowed');
        }

        // Before doing anything with persistent effects on external systems
        // (in this case, the P&T module), we check that the target is valid.
        await this.checkMembershipStatus(pid);

        // Assign a new PID for the migrated member
        const newPid: string = short.generate();
        Logger.debug('Migrating member from PID ' + pid + ' to new PID ' + newPid);

        // Instruct the P&T module to register this member with the new legal entity.
        // This is the first step of the migration process: the second and
        // final one will be triggered by the P&T module calling our callback
        // function sourceMigrationConfirmed().
        await this.requestSourceMigration(pid, newPid, req, callerContext);

        // Return the new assigned PID to the caller: they will have to check back later
        // if the overall migration process was completed successfully.
        return new Identifier(newPid);
    }

    // callback executed by the P&T module when a source registration transaction is successful
    sourceRegistrationConfirmed(pid: string): boolean {

        // synchronous feedback if the PID does not match any pending request
        const req = this.requests.pop(pid);
        if (!(req instanceof MemberOnboardingRequest)) {
            Logger.warn('Source registration callback: no pending onboarding process exists with PID ' + pid);
            return false;
        }

        // fire and forget the async jobs
        this.completeOnboarding(pid, req);
        return true;
    }

    // callback executed by the P&T module when a source deregistration transaction is successful
    sourceDeregistrationConfirmed(pid: string): boolean {
        // synchronous feedback if the PID does not match any pending request
        const req = this.requests.pop(pid);
        if (!(req instanceof MemberOffboardingRequest)) {
            Logger.warn('Source deregistration callback: no pending process exists with PID ' + pid);
            return false;
        }

        // fire and forget the async jobs, passing the stored caller context
        this.completeOffboarding(pid, req.callerContext);
        return true;
    }

    // callback executed by the P&T module when a source migration transaction is successful
    sourceMigrationConfirmed(oldPid: string): boolean {
        // synchronous feedback if the PID does not match any pending request
        // Note: The P&T callback uses the OLD PID, not the new one
        const req = this.requests.pop(oldPid);
        if (!(req instanceof MemberMigrationRequest)) {
            Logger.warn('Source migration callback: no pending process exists with old PID ' + oldPid);
            return false;
        }

        // fire and forget the async jobs
        this.completeMigration(req);
        return true;
    }

    // callback executed by the P&T module when an account permissioning transaction is successful
    accountPermissionGrantConfirmed(tid: string): boolean {
        // synchronous feedback if the TID does not match any pending request
        const req = this.requests.pop(tid);
        if (!(req instanceof AccountEnrolmentRequest)) {
            Logger.warn('Account permission grant callback: no pending process exists with TID ' + tid);
            return false;
        }

        // fire and forget the async jobs
        this.completeEnrolment(req);
        return true;
    }

    // callback executed by the P&T module when an account de-permissioning transaction is successful
    accountPermissionRevocationConfirmed(tid: string): boolean {
        // Check if this is part of a background job first
        const jobId = this.findBackgroundJobForAccount(tid);
        if (jobId) {
            // This is part of a background job
            this.backgroundJobs.markAccountCompleted(jobId, tid);

            // Process next account in the background job
            this.processNextAccountInBackground(jobId);
            return true;
        }

        // Otherwise, handle as a regular disenrollment request
        const req = this.requests.pop(tid);
        if (!(req instanceof AccountDisenrolmentRequest)) {
            Logger.warn('Account permissioning revocation callback: no pending process exists with TID ' + tid);
            return false;
        }

        // fire and forget the async jobs
        this.completeDisenrolment(tid);
        return true;
    }

    async listMembers(query?: MemberFilter): Promise<MemberRef[]> {
        const list: MemberEntity[] = await this.db.getMembers(query);
        return list.map(item => {
            return {
                pid: item.pid,
                name: item.nam,
                type: item.typ,
                country: item.cnt,
                active: item.ofb ? false : true,
            };
        });
    }

    async retrieveMember(pid: string): Promise<Member> {
        if (!isShortUUID(pid)) {
            return null; // shortcut
        }
        let result: Member = null;
        const match: MemberEntity = await this.db.getMember(pid);
        if (match) {
            const onboarder: MemberEntity = await match.onboarder;
            result = {
                pid: match.pid,
                org: await this.loadSource(pid), // we get LegalEntity from the P&T module
                type: match.typ,
                rep: this.parseRepresentative(match.rep), // ContactInfo from JSON string, if any
                authority: onboarder.pid === match.pid ? 'SELF' : onboarder.pid,
                onboarded: match.onb.toISOString(),
                offboarded: match.ofb ? match.ofb.toISOString() : null,
            };
        }
        return result; // null if not found
    }

    async retrieveActiveMemberRef(pid: string): Promise<MemberRef> {
        if (!isShortUUID(pid)) {
            return null; // shortcut
        }
        let result: MemberRef = null;
        const match: MemberEntity = await this.db.getActiveMember(pid);
        if (match) {
            result = {
                pid: match.pid,
                name: match.nam,
                type: match.typ,
                country: match.cnt,
                active: true,
            }
        }
        return result; // null if not found
    }

    async updateMember(pid: string, req: MemberUpdateRequest): Promise<void> {
        // Step 1: Validate that at least one field is provided
        if (!req.type && !req.rep && !req.oa) {
            Logger.debug('Bad user-provided input: no fields provided in update request');
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'At least one field must be provided');
        }

        // Step 2: Check that the target member exists and is active
        const currentMember = await this.checkMembershipStatus(pid);

        // Step 3: Check if any values are actually changing
        let hasChanges = false;

        if (req.type !== undefined) {
            if (req.type !== currentMember.typ) {
                hasChanges = true;
            }
        }

        if (req.rep !== undefined) {
            // Handle explicit null to clear the field
            const newRep = req.rep === null ? null : JSON.stringify(req.rep);
            const currentRep = currentMember.rep;
            if (newRep !== currentRep) {
                hasChanges = true;
            }
        }

        // OA change detection is more complex - we'll handle it separately
        let oaIsChanging = false;
        if (req.oa !== undefined) {
            // Determine current OA state
            const currentOnboarder = await currentMember.onboarder;
            if (currentOnboarder.pid !== pid) {
                // Member currently delegates the OA role to another member (SCENARIO 1)
                if (isDID(req.oa)) {
                    // SCENARIO 1B: Assuming OA role
                    oaIsChanging = true;
                    hasChanges = true;
                } else {
                    // SCENARIO 1A: Delegating to different member
                    if (req.oa !== currentOnboarder.pid) {
                        oaIsChanging = true;
                        hasChanges = true;
                    }
                }
            } else {
                // Member currently has the OA role (SCENARIO 2)
                // Check if new value is a different DID
                const currentAuthority = await this.db.getActiveAuthorityByOwner(pid);
                if (!currentAuthority) {
                    throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Cannot resolve the OA for the target member');
                }

                if (isDID(req.oa)) {
                    // SCENARIO 2B: Switching to new DID
                    if (req.oa !== currentAuthority.did) {
                        oaIsChanging = true;
                        hasChanges = true;
                    }
                } else {
                    // SCENARIO 2A: Delegating to another member
                    if (req.oa === pid) {
                        throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Cannot self-delegate the OA role');
                    }
                    oaIsChanging = true;
                    hasChanges = true;
                }
            }
        }

        if (!hasChanges) {
            Logger.debug('No changes detected in update request for PID ' + pid);
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'No changes detected');
        }

        // Step 4: Validate the authority reference if OA is changing
        if (req.oa) {
            await this.checkAuthorityReference(req.oa);
        }

        // Step 5: Prepare OA change context with all resolved data (if OA is changing)
        let oaContext: OAChangeContext | null = null;
        if (oaIsChanging && req.oa) {
            const currentOnboarder = await currentMember.onboarder;
            const currentOAIsSelf = currentOnboarder.pid === pid;
            const newOAIsSelf = isDID(req.oa);

            oaContext = {
                currentOAIsSelf,
                newOAIsSelf,
            };

            // For SCENARIO 2: Member currently has self-OA
            if (currentOAIsSelf) {
                const currentAuthority = await this.db.getActiveAuthorityByOwner(pid);
                if (!currentAuthority) {
                    throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Cannot resolve the OA for the target member');
                }
                oaContext.currentAuthority = currentAuthority;

                // For SCENARIO 2A: Delegating to another member
                if (!newOAIsSelf) {
                    const delegateMember = await this.db.getAuthorityOwner(req.oa);
                    if (!delegateMember) {
                        throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Could not resolve delegate OA member');
                    }
                    oaContext.delegateMember = delegateMember;

                    const delegateAuthority = await this.db.getActiveAuthorityByOwner(delegateMember.pid);
                    if (!delegateAuthority) {
                        throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Delegate member has no active authority');
                    }
                    oaContext.delegateAuthority = delegateAuthority;
                }
            } else {
                // For SCENARIO 1A: Delegating to different member
                if (!newOAIsSelf) {
                    const delegateMember = await this.db.getAuthorityOwner(req.oa);
                    if (!delegateMember) {
                        throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Could not resolve delegate OA member');
                    }
                    oaContext.delegateMember = delegateMember;
                }
            }
        }

        // Step 6: Perform the update in the database
        Logger.log('Updating member with PID ' + pid + (oaIsChanging ? ' (OA change detected)' : ''));

        const deactivatedDID = await this.db.updateMember(currentMember, req, oaContext);

        // Step 7: If an authority was deactivated, add it to blacklist
        if (deactivatedDID) {
            Logger.log('Deactivated authority with DID ' + deactivatedDID + ', adding to blacklist');
            await this.blacklist.addAuthorityBan(deactivatedDID);
        }

        Logger.log('Member with PID ' + pid + ' updated successfully');
    }

    async listAuthorities(query?: AuthorityFilter): Promise<Authority[]> {
        const list = await this.db.getAuthorities(query);
        return list.map(item => {
            return {
                did: item.did,
                pid: item.owner.pid,
                name: item.owner.nam,
                activated: item.act.toISOString(),
                deactivated: item.dct ? item.dct.toISOString() : null,
                active: item.dct ? false : true,
            };
        });
    }

    async retrieveAuthority(did: string, active?: boolean): Promise<Authority> {
        if (!isDID(did)) {
            return null; // shortcut
        }
        const match: AuthorityEntity = active ? await this.db.getActiveAuthority(did) : await this.db.getAuthority(did);
        if (match) {
            return {
                did: match.did,
                pid: match.owner.pid,
                name: match.owner.nam,
                activated: match.act.toISOString(),
                deactivated: match.dct ? match.dct.toISOString() : null,
                active: match.dct ? false : true
            }
        } else {
            return null;
        }
    }

    async retrieveAuthorityOwnerRef(did: string): Promise<MemberRef> {
        const match: AuthorityEntity = await this.db.getActiveAuthority(did);
        if (match) {
            return {
                pid: match.owner.pid,
                name: match.owner.nam,
                type: match.owner.typ,
                country: match.owner.cnt,
                active: true
            }
        } else {
            return null;
        }
    }

    async checkLoginContext(did: string, pid: string): Promise<boolean> {
        const authority: AuthorityEntity = await this.db.getActiveAuthority(did);
        if (authority) {
            // authority check ok
            if (pid && pid.trim().length > 0 && pid !== 'NA') {
                // affiliation check requested
                const member = await this.db.getActiveMember(pid);
                if (member) {
                    // affiliation check ok, check for consistency with authority
                    const onboarder = await member.onboarder; // lazy loading
                    if (onboarder && onboarder.pid === authority.owner.pid) {
                        // authority / affiliation consistency check ok, all good
                        return true;
                    } else {
                        // authority / affiliation consistency check failed
                        Logger.debug('Failed authority / affiliation consistency check - DID: ' + did + ' / PID: ' + pid);
                        return false;
                    }
                } else {
                    // affiliation check failed
                    Logger.debug('Failed affiliation check - PID: ' + pid);
                    return false;
                }
            } else {
                // no affiliation check requested, we are ok
                return true;
            }
        } else {
            // authority check failed
            Logger.debug('Failed authority check - DID: ' + did);
            return false;
        }
    }

    public async startAccountEnrolmentProcess(req: AccountEnrolmentRequest): Promise<void> {

        // Before doing anything with persistent effects on external systems
        // (in this case, the P&T module), we do a careful validation of user input:
        // specifically, we want to ensure that the provided references to the Onboarding Authority
        // and to the affiliating member are valid according to our rules, and that the
        // account is not already enrolled.
        await this.checkEnrolmentConditions(req);

        // Instruct the P&T module to grant blockchain permission to this trading account.
        // This is the first step of the process: the second and final one will be triggred
        // by the P&T module calling our callback function accountPermissionGrantConfirmed().
        await this.requestAccountPermissionGrant(req);
    }

    public async startAccountDisenrolmentProcess(tid: string): Promise<void> {
        // Instruct the P&T module to grant/revoke blockchain permission to this trading account.
        // This is the first step of the process: the second and final one will be triggred
        // by the P&T module calling our callback function accountPermissionRevocationConfirmed().
        await this.requestAccountPermissionRevocation(tid);
    }

    public async listAccounts(query?: AccountFilter, user?: UserInfo): Promise<Account[]> {
        const accounts: AccountEntity[] = await this.db.getAccounts(query, user);
        return accounts.map(account => {
            return {
                tid: account.tid,
                owningUser: account.usr,
                owningMember: account.owner?.pid ?? null, // convert undefined to null
                userAuthority: account.authority.owner.pid,
                enrolled: account.act.toISOString(),
                disenrolled: account.dct ? account.dct.toISOString() : null,
            };
        });
    }

    public async listActiveAccountForUser(user: UserInfo, pagination?: { l?: number; o?: number }): Promise<Account[]> {
        const accounts: AccountEntity[] = await this.db.getActiveAccountsForUser(user, pagination);
        return accounts.map(account => {
            return {
                tid: account.tid,
                owningUser: account.usr,
                owningMember: account.owner?.pid ?? null, // convert undefined to null
                userAuthority: account.authority.owner.pid,
                enrolled: account.act.toISOString(),
                disenrolled: account.dct ? account.dct.toISOString() : null,
            };
        });
    }

    public async retrieveAccount(tid: string, user?: UserInfo): Promise<Account> {
        const accountEntity: AccountEntity = await this.db.getAccount(tid);
        if (accountEntity) {
            if (user && (accountEntity.usr !== user.uid || accountEntity.authority?.did !== user.issuerDID)) {
                return null; // profiled query + invalid target
            }
            const account = new Account();
            account.tid = accountEntity.tid;
            account.owningUser = accountEntity.usr;
            account.owningMember = accountEntity.owner?.pid ?? null, // convert undefined to null
                account.userAuthority = accountEntity.authority.owner.pid
            account.enrolled = accountEntity.act.toISOString();
            account.disenrolled = accountEntity.dct ? accountEntity.dct.toISOString() : null;
            return account;
        } else {
            return null;
        }
    }

    public async retrieveAccountBalance(tid: string): Promise<string> {
        try {
            const response: AxiosResponse = await this.tmProxy.get('/payment-tokens/balances/accounts/' + tid);
            return parseFloat(response.data.balance).toFixed(2); // convert number to string with 2 decimal places
        } catch (err) {
            Logger.error(`Error calling the T&M token balance service: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to T&M token balance service failed');
        }
    }

    public async checkAccountStatus(tid: string, active?: boolean): Promise<boolean> {
        const account: AccountEntity = await this.db.getAccount(tid);
        if (account) { // The account exists
            if (active) { // The caller is checking if the account is active
                return account.dct ? false : true;
            } else { // The caller is checking if the account exists
                return true;
            }
        } else { // The account does not exist
            return false;
        }
    }

    public async checkAccountOwnership(tid: string, did: string, uid: string): Promise<boolean> {
        const account: AccountEntity = await this.db.getAccount(tid);
        if (account) {
            return (account.authority.did === did && account.usr === uid);
        } else {
            return false;
        }
    }

    public async creditAccount(tid: string, amount: string): Promise<any> {
        // Check if account exists and is active
        const accountActive = await this.checkAccountStatus(tid, true);
        if (!accountActive) {
            Logger.warn(`Credit operation rejected: account ${tid} does not exist or is not active`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Trading account does not exist or is not active');
        }

        // Generate transaction ID and timestamp
        const trx = uuidv4();
        const tms = new Date();

        Logger.debug(`Starting credit operation for account ${tid}, amount: ${amount}, transaction ID: ${trx}`);

        // Insert transaction record first
        try {
            await this.db.insertAccountTransaction(trx, tms, tid, 0, amount);
            Logger.debug(`Transaction record inserted for credit operation: ${trx}`);
        } catch (err) {
            Logger.error(`Error inserting transaction record for credit operation: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Failed to insert transaction record');
        }

        try {
            // Call external service
            await this.tmProxy.post('/payment-tokens/mint', { tradingAccount: tid, amount: amount });
            Logger.debug(`External service call successful for credit operation: ${trx}`);
        } catch (err) {
            // If external service fails, remove the transaction record
            Logger.warn(`External service call failed for credit operation ${trx}, rolling back transaction record`);
            await this.db.deleteAccountTransaction(trx);
            Logger.error(`Error calling the T&M token minting service: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to T&M token minting service failed');
        }

        Logger.log(`Credit operation completed successfully for account ${tid}, transaction ID: ${trx}`);
        return {
            marketplace_transaction_id: trx,
            timestamp: tms.toISOString()
        };
    }

    public async debitAccount(tid: string, amount: string): Promise<any> {
        // Check if account exists
        const accountExists = await this.checkAccountStatus(tid);
        if (!accountExists) {
            Logger.warn(`Debit operation rejected: account ${tid} does not exist`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Trading account does not exist');
        }

        // Generate transaction ID and timestamp
        const trx = uuidv4();
        const tms = new Date();

        Logger.debug(`Starting debit operation for account ${tid}, amount: ${amount}, transaction ID: ${trx}`);

        // Insert transaction record first
        try {
            await this.db.insertAccountTransaction(trx, tms, tid, 1, amount);
            Logger.debug(`Transaction record inserted for debit operation: ${trx}`);
        } catch (err) {
            Logger.error(`Error inserting transaction record for debit operation: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Failed to insert transaction record');
        }

        try {
            // Call external service
            await this.tmProxy.delete('/payment-tokens/burn', { data: { tradingAccount: tid, amount: amount } });
            Logger.debug(`External service call successful for debit operation: ${trx}`);
        } catch (err) {
            // If external service fails, remove the transaction record
            Logger.warn(`External service call failed for debit operation ${trx}, rolling back transaction record`);
            await this.db.deleteAccountTransaction(trx);
            Logger.error(`Error calling the T&M token burning service: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to T&M token burning service failed');
        }

        Logger.log(`Debit operation completed successfully for account ${tid}, transaction ID: ${trx}`);
        return {
            marketplace_transaction_id: trx,
            timestamp: tms.toISOString()
        };
    }

    public async listAccountTransactions(tid: string): Promise<any> {
        // Check if account exists
        const accountExists = await this.checkAccountStatus(tid);
        if (!accountExists) {
            Logger.warn(`Transaction list rejected: account ${tid} does not exist`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Trading account does not exist');
        }

        Logger.debug(`Retrieving transactions for account ${tid}`);

        const transactions = await this.db.getTransactionsByAccount(tid);

        Logger.debug(`Found ${transactions.length} transactions for account ${tid}`);

        return {
            total_transactions: transactions.length,
            transactions: transactions.map(tx => ({
                marketplace_transaction_id: tx.trx,
                type: tx.dir === 0 ? 'purchase' : 'redeem',
                amount: tx.amt,
                timestamp: tx.tms.toISOString()
            }))
        };
    }

    public async checkSecurityContext(sc: SecurityContext): Promise<void> {

        // check that an active OA with this DID exists
        const oa: AuthorityEntity = await this.db.getActiveAuthority(sc.did);
        if (!oa) {
            Logger.warn('Login attempt rejected: unknown Onboarding Authority with DID ' + sc.did);
            throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Unknown Onboarding Authority')
        }

        if (await this.blacklist.isAuthorityBanned(sc.did)) {
            Logger.warn('Login attempt rejected: Onboarding Authority with DID ' + sc.did + ' is suspended');
            throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Onboarding Authority is suspended')
        }

        if (sc.uid) { // an user was specified
            if (await this.blacklist.isUserBanned(sc.did, sc.uid)) {
                Logger.warn('Login attempt rejected: user ' + sc.uid + ' is banned for Onboarding Authority with DID ' + sc.did);
                throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Access denied for user')
            }
        }

        if (sc.pid) { // an affiliation was specified

            // check that an active member with this PID exists
            const member = await this.db.getActiveMember(sc.pid);
            if (!member) {
                Logger.warn('Login attempt rejected: unknown member with PID ' + sc.pid);
                throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Unknown affiliation')
            }

            if (await this.blacklist.isMemberBanned(sc.pid)) {
                Logger.warn('Login attempt rejected: member with PID ' + sc.pid + ' is suspended');
                throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Affilation membership is suspended')
            }

            // check that this member's designated OA matches the provided DID
            const onboarder: MemberEntity = await member.onboarder; // lazy loading
            if (onboarder.pid !== oa.owner.pid) {
                Logger.warn('Login attempt rejected: Onboarding Authority with DID ' + sc.did + ' does not match the member\'s designated OA with PID ' + onboarder.pid);
                throwExceptionWithErrorCondition(ErrorCondition.AUTH, 'Inconsistent Onboarding Authority for affiliation')
            }
        }
    }

    private async checkAuthorityReference(id: string): Promise<void> {
        if (isDID(id)) {
            // provided DID must not match an existing authority, either active or revoked
            const match = await this.db.getAuthority(id);
            if (match) {
                Logger.debug('Bad user-provided input: DID ' + id + ' is already registered');
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Onboarding Authority with DID ' + id + ' is already registered');
            }
        } else if (isShortUUID(id)) {
            // provided PID must match an onboarded member that owns (at least) one active authority
            const match = await this.db.getAuthorityOwner(id);
            if (!match) {
                Logger.debug('Bad user-provided input: PID ' + id + ' does not match any active member with OA role');
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'No Onboarding Authority with PID ' + id + ' was found');
            }
        } else {
            Logger.debug('Bad user-provided input: value ' + id + ' is not recognized as a DID or PID');
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Value in "oa" (' + id + ') is not recognized as a DID or PID');
        }
    }

    private async checkMembershipStatus(pid: string): Promise<MemberEntity> {
        // Use getMember() instead of getActiveMember() so that we can differentiate between specific error cases
        const match = await this.db.getMember(pid);
        if (match) {
            if (match.ofb) {
                Logger.debug('Bad user-provided input: member with PID ' + pid + ' is already offboarded');
                throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Member with PID ' + pid + ' is already offboarded');
            }
        } else {
            Logger.debug('Bad user-provided input: PID ' + pid + ' does not match any existing member');
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Member with PID ' + pid + ' does not exist');
        }
        return match;
    }

    private async checkMembershipDependencies(pid: string): Promise<void> {
        const match = await this.db.getActiveMember(pid);
        if (match) {
            const dependent: MemberEntity[] = await this.db.getActiveMembersUnderAuthority(pid);
            for (const member of dependent) {
                if (member.pid !== pid) { // skip the member itself, which may be its own OA
                    Logger.debug('Bad user-provided input: PID ' + pid + ' has active members under its authority');
                    throwExceptionWithErrorCondition(ErrorCondition.INPUT, `One or more active memnbers exist under the authority of member with PID ${pid}`);
                }
            }
        }
    }

    private async requestSourceRegistration(pid: string, req: MemberOnboardingRequest): Promise<void> {

        // Store the request locally so that we can get back to this process
        // once we receive the callback from the P&T module.
        // (note that ProcessState is a 'volatile' storage with 60" of max life)
        try {
            this.requests.push(pid, req);
        } catch {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Another process for the same entity is pending: try again later')
        }

        // We call the P&T service to register the trusted source under the assigned PID.
        // If registration is successful, the P&T module should call our sourceRegistrationConfirmed() callback.
        let response = null;
        try {
            response = await this.ptProxy.post('/sources', { pid: pid, descriptor: req.org });
        } catch (err) {
            // we failed to call the P&T service: log the error and remove the request
            Logger.error('Error calling P&T source registration service', err);
            this.requests.pop(pid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T source registration service failed')
        }

        if (response.status == 202) {
            Logger.debug('Call to P&T source registration service successful for PID ' + pid);
        } else {
            // the call to the P&T service was unsuccessful: log the error and remove the request
            Logger.warn('Call to P&T source registration service FAILED for PID ' + pid + ', status code is ' + response.status);
            this.requests.pop(pid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T source registration service failed')
        }
    }

    private async requestSourceDeregistration(pid: string, callerContext?: CallerContext): Promise<void> {

        // Store the request locally so that we can get back to this process
        // once we receive the callback from the P&T module. In the deregistration
        // process, we store the caller context so we can include it in failure reports.
        // (note that ProcessState is a 'volatile' storage with 60" of max life).
        try {
            this.requests.push(pid, new MemberOffboardingRequest(callerContext));
        } catch {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Another process for the same entity is pending: try again later')
        }

        // We call the P&T service to unregister the trusted source with this PID.
        // If deregistration is successful, the P&T module should call our sourceDeregistrationConfirmed() callback.
        let response = null;
        try {
            response = await this.ptProxy.delete('/sources/' + pid);
        } catch (err) {
            // we failed to call the P&T service: log the error and remove the request
            Logger.error('Error calling P&T source deregistration service', err);
            this.requests.pop(pid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T source deregistration service failed')
        }

        if (response.status == 202) {
            Logger.debug('Call to P&T source deregistration service successful for PID ' + pid);
        } else {
            // the call to the P&T service was unsuccessful: log the error and remove the request
            Logger.warn('Call to P&T source deregistration service FAILED for PID ' + pid + ', status code is ' + response.status);
            this.requests.pop(pid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T source deregistration service failed')
        }
    }

    private async requestSourceMigration(oldPid: string, newPid: string, legalEntity: LegalEntity, callerContext?: CallerContext): Promise<void> {

        // Store the request locally so that we can get back to this process
        // once we receive the callback from the P&T module. In the migration
        // process, we store both PIDs and the caller context.
        // IMPORTANT: The request is keyed by OLD PID because the P&T callback uses the old PID.
        // (note that ProcessState is a 'volatile' storage with 60" of max life).
        try {
            this.requests.push(oldPid, new MemberMigrationRequest(oldPid, newPid, legalEntity, callerContext));
        } catch {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Another process for the same entity is pending: try again later')
        }

        // We call the P&T service to migrate the source. This is a PUT operation that:
        // 1. Deactivates the old source (oldPid)
        // 2. Registers the new source (newPid with new legalEntity)
        // If migration is successful, the P&T module will call our callback at /members/{oldPid}/confirm-migration
        let response = null;
        try {
            response = await this.ptProxy.put('/sources/' + oldPid, { pid: newPid, descriptor: legalEntity });
        } catch (err) {
            // we failed to call the P&T service: log the error and remove the request
            Logger.error('Error calling P&T source migration service', err);
            this.requests.pop(oldPid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T source migration service failed')
        }

        if (response.status == 202) {
            Logger.debug('Call to P&T source migration service successful: migrating from PID ' + oldPid + ' to new PID ' + newPid);
        } else {
            // the call to the P&T service was unsuccessful: log the error and remove the request
            Logger.warn('Call to P&T source migration service FAILED for PID ' + oldPid + ', status code is ' + response.status);
            this.requests.pop(oldPid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T source migration service failed')
        }
    }

    private async loadSource(pid: string): Promise<LegalEntity> {
        let response: AxiosResponse = null;
        try {
            response = await this.ptProxy.get('/sources/' + pid);
        } catch (err) {
            Logger.error('Error calling P&T source lookup service', err);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Could not communicate with the P&T source lookup service');
        }

        try {
            return plainToInstance(Source, response.data).descriptor;
        } catch (err) {
            Logger.error('Error parsing response from P&T source lookup service', err);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Unexpected response from the P&T source lookup service');
        }
    }

    private parseRepresentative(rep: string): ContactInfo {
        if (!rep || rep.trim().length == 0) // shortcut
            return null;

        try {
            return plainToInstance(ContactInfo, JSON.parse(rep));
        } catch (err) { // swallow any error
            Logger.warn('Error parsing representative contact information from member record', err);
            return null;
        }
    }

    private async checkEnrolmentConditions(req: AccountEnrolmentRequest): Promise<void> {
        if (isDID(req.did)) {
            // provided DID must match an active authority
            const match = await this.db.getActiveAuthority(req.did);
            if (!match) {
                Logger.debug('Bad user-provided input: DID ' + req.did + ' does not match any active authority');
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Value in "did" (' + req.did + ') does not match any active Onboarding Authority')
            }
        } else {
            Logger.debug('Bad user-provided input: value ' + req.did + ' is not recognized as a DID');
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Value in "did" (' + req.did + ') is not recognized as a DID')
        }

        if (req.affiliation && req.affiliation.length > 0) { // user affiliation is optional
            if (isShortUUID(req.affiliation)) {
                // provided PID must match an active member
                const match = await this.db.getActiveMember(req.affiliation);
                if (!match) {
                    Logger.debug('Bad user-provided input: PID ' + req.affiliation + ' does not match any active member');
                    throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Value in "affiliation" (' + req.affiliation + ') does not match any active member')
                }
            } else {
                Logger.debug('Bad user-provided input: value ' + req.affiliation + ' is not recognized as a DID');
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Value in "affiliation" (' + req.affiliation + ') is not recognized as a PID')
            }
        }

        const match = await this.db.getAccount(req.tid);
        if (match) {
            Logger.debug('Bad user-provided input: TID ' + req.tid + ' matches an existing account');
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Value in "tid" (' + req.tid + ') matches an existing account')
        }
    }

    private async requestAccountPermissionGrant(req: AccountEnrolmentRequest): Promise<void> {

        // store the request locally so that we can get back to this process
        // once we receive the callback from the P&T module
        // (note this is a 'volatile' storage with 60" of max life)
        try {
            this.requests.push(req.tid, req);
        } catch {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Another process for the same entity is pending: try again later')
        }

        // We call the P&T service to grant blockchain permission for the trading account.
        // If the blockchain transaction is successful, the P&T module should call our accountPermissionGrantConfirmed() callback.
        let response = null;
        try {
            Logger.debug('Calling P&T account permissioning service for granting permission to TID ' + req.tid);
            response = await this.ptProxy.post('/permissions', new AccountIdentifier(req.tid)); // grant permission
        } catch (err) {
            // we failed to call the P&T service: log the error and remove the request
            Logger.error('Error calling P&T account permissioning service', err);
            this.requests.pop(req.tid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T account permissioning service failed')
        }
        Logger.debug('Call to P&T account permissioning service successful for TID ' + req.tid);
    }

    private async requestAccountPermissionRevocation(tid: string): Promise<void> {

        // store the request locally so that we can get back to this process
        // once we receive the callback from the P&T module
        // (note this is a 'volatile' storage with 60" of max life)
        try {
            this.requests.push(tid, new AccountDisenrolmentRequest());
        } catch {
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Another process for the same entity is pending: try again later')
        }

        // We call the P&T service to revoke blockchain permission for the trading account.
        // If the blockchain transaction is successful, the P&T module should call our accountPermissionRevocationConfirmed() callback.
        let response = null;
        try {
            Logger.debug('Calling P&T account permissioning service for revoking permission from TID ' + tid);
            response = await this.ptProxy.delete('/permissions/' + tid); // revoke permission
        } catch (err) {
            // we failed to call the P&T service: log the error and remove the request
            Logger.error('Error calling P&T account permissioning service', err);
            this.requests.pop(tid);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to P&T account permissioning service failed')
        }
        Logger.debug('Call to P&T account permissioning service successful for TID ' + tid);
    }

    // this is where we actually register the member and, if needed, the authority
    private async completeOnboarding(pid: string, req: MemberOnboardingRequest): Promise<void> {
        try {
            // all database operations are done in the context of an atomic transaction
            await this.db.createMembership(
                pid, // pid
                req.org.legalName, // nam
                req.org.legalAddress.country, // cnt
                (req.rep ? JSON.stringify(req.rep) : null), // rep
                req.type, // typ
                req.oa, // onboarder
            );
            Logger.log('Member successfully onboarded with PID ' + pid);
        } catch (err) {
            // log and swallow the error
            Logger.error('Could not save onboarding data for PID ' + pid, err.message);
        }
    }

    // this is where we actually mark the member as offboarded and, if needed, deactivate all its dependent
    // accounts and authorities
    private async completeOffboarding(member: string, callerContext?: CallerContext): Promise<void> {
        try {
            // Get member details before offboarding for context
            const memberEntity = await this.db.getMember(member);

            // all database operations are done in the context of an atomic transaction
            const disenrolledAccounts: Set<string> = await this.db.offboardMember(member);
            Logger.log('Member with PID ' + member + ' successfully offboarded, ' + disenrolledAccounts.size + ' accounts disenrolled from database');

            // Start background job to process account disenrollment from blockchain
            if (disenrolledAccounts.size > 0) {
                const jobId = short.generate();
                const accountList = Array.from(disenrolledAccounts);

                try {
                    // Create context information for the background job
                    const context: BackgroundJobContext = {
                        initiatedBy: callerContext?.source === 'external_api' ? 'external_api' : 'internal_api',
                        sourceEndpoint: callerContext?.endpoint || '/gov/v1.0/members/{pid}',
                        timestamp: callerContext?.timestamp || Date.now(),
                        memberDetails: memberEntity ? {
                            pid: memberEntity.pid,
                            name: memberEntity.nam,
                            country: memberEntity.cnt,
                            type: memberEntity.typ
                        } : undefined,
                        callerContext: callerContext
                    };

                    this.backgroundJobs.createJob(jobId, member, 'member', accountList, context);
                    Logger.log(`Started background job ${jobId} to disenroll ${accountList.length} accounts for member ${member}`);

                    // Start processing the first account
                    this.processNextAccountInBackground(jobId);
                } catch (err) {
                    Logger.error(`Failed to create background job for member ${member}`, err.message);
                }
            }
        } catch (err) {
            // log and swallow the error
            Logger.error('Could not save offboarding data for PID ' + member, err.message);
        }
    }

    private async completeMigration(req: MemberMigrationRequest): Promise<void> {
        try {
            // Retrieve the old member data to get metadata we need to preserve
            const oldMember = await this.db.getMember(req.oldPid);
            if (!oldMember) {
                Logger.error('Migration failed: old member with PID ' + req.oldPid + ' not found');
                return;
            }

            // Get the onboarder reference to handle three cases:
            // 1. Self-reference (onboarder.pid == oldPid): Update to newPid
            // 2. ROA reference (onboarder.pid == ROA_PID): Preserve as-is
            // 3. Delegated OA (onboarder.pid == another member): Preserve as-is
            const onboarder = await oldMember.onboarder;
            const autRef = onboarder && onboarder.pid === req.oldPid ? req.newPid : onboarder.pid;

            Logger.log('Starting atomic migration from PID ' + req.oldPid + ' to new PID ' + req.newPid);

            // Perform atomic migration in a single database transaction:
            // 1. Set offboarding date on old member record
            // 2. Create new member record with new PID and legal entity
            // 3. Transfer all authorities from old member to new member
            // 4. Transfer all directly-owned accounts from old member to new member
            // 5. Accounts linked via authority are automatically correct (eager loading)
            await this.db.migrateMember(
                req.oldPid, // old PID to mark as offboarded
                req.newPid, // new PID for the member
                req.legalEntity.legalName, // new legal name
                req.legalEntity.legalAddress.country, // new country
                oldMember.rep, // same representative
                oldMember.typ, // same organization type
                autRef, // same onboarding authority (or self if was self-onboarded)
            );

            Logger.log('Member migration completed successfully from PID ' + req.oldPid + ' to new PID ' + req.newPid);
            Logger.log('All authorities and accounts transferred to new PID in single atomic transaction');
        } catch (err) {
            // log and swallow the error
            Logger.error('Could not complete migration from PID ' + req.oldPid + ' to new PID ' + req.newPid + ': ' + err.message);
        }
    }

    /**
     * Core logic for processing user offboarding notification.
     * This method blacklists the user and disenrolls all their accounts.
     * Used by both external and internal API endpoints.
     */
    async processUserAccountDisenrollment(target: UserQualifiedId, callerContext: CallerContext): Promise<void> {
        // Add user to blacklist
        await this.blacklist.addUserBan(target.oa, target.uid);

        // Disenroll all accounts owned by the user
        await this.disenrollUserAccounts(target, callerContext);
    }

    private async disenrollUserAccounts(user: UserQualifiedId, callerContext?: CallerContext): Promise<void> {
        // Disenroll all accounts owned by the specified user
        const disenrolledAccounts: Set<string> = await this.db.disenrolAccountsByOwner(user);
        Logger.log(`${disenrolledAccounts.size} accounts successfully disenrolled from database for user ${user.oa}:${user.uid}`);

        // Start background job to process account disenrollment from blockchain
        if (disenrolledAccounts.size > 0) {
            const jobId = short.generate();
            const accountList = Array.from(disenrolledAccounts);
            const userIdentifier = `${user.oa}:${user.uid}`;

            try {
                // Create context information for the background job
                const context: BackgroundJobContext = {
                    initiatedBy: callerContext?.source === 'external_api' ? 'external_api' : 'internal_api',
                    sourceEndpoint: callerContext?.endpoint || '/gov/v1.0/users/{oa}/{uid}/accounts',
                    timestamp: callerContext?.timestamp || Date.now(),
                    userDetails: {
                        oa: user.oa,
                        uid: user.uid
                    },
                    callerContext: callerContext
                };

                this.backgroundJobs.createJob(jobId, userIdentifier, 'user', accountList, context);
                Logger.log(`Started background job ${jobId} to disenroll ${accountList.length} accounts for user ${userIdentifier}`);

                // Start processing the first account
                this.processNextAccountInBackground(jobId);
            } catch (err) {
                Logger.error(`Failed to create background job for user ${userIdentifier}`, err.message);
            }
        }
    }

    // this is where we actually register the account
    private async completeEnrolment(req: AccountEnrolmentRequest): Promise<void> {
        try {
            await this.db.createAccount(req.tid, req.did, req.uid, req.affiliation);
            Logger.log('Account with TID ' + req.tid + ' successfully enrolled');
        } catch (err) {
            // log and swallow the error
            Logger.error('Could not save enrolment data for TID ' + req.tid, err.message);
        }
    }

    // this is where we actually mark the account as disenrolled
    private async completeDisenrolment(tid: string): Promise<void> {
        try {
            await this.db.deactivateAccount(tid);
            Logger.log('Account with TID ' + tid + ' successfully disenrolled');
        } catch (err) {
            // log and swallow the error
            Logger.error('Could not save disenrolment data for TID ' + tid, err.message);
        }
    }

    private async processNextAccountInBackground(jobId: string): Promise<void> {
        try {
            const nextTid = this.backgroundJobs.getNextAccount(jobId);
            if (!nextTid) {
                // No more accounts to process
                return;
            }

            // Use the existing account permission revocation logic
            await this.requestAccountPermissionRevocationForBackground(nextTid, jobId);
        } catch (err) {
            Logger.error(`Background job ${jobId}: Failed to process next account`, err.message);

            // Mark the current account as failed and continue
            const currentItem = this.backgroundJobs.getCurrentItem(jobId);
            if (currentItem) {
                this.backgroundJobs.markAccountFailed(jobId, currentItem.tid, err.message);
            }

            // Try to process the next account
            this.processNextAccountInBackground(jobId);
        }
    }

    private async requestAccountPermissionRevocationForBackground(tid: string, jobId: string): Promise<void> {
        // We call the P&T service to revoke blockchain permission for the trading account.
        // If the blockchain transaction is successful, the P&T module should call our accountPermissionRevocationConfirmed() callback.
        let response = null;
        try {
            Logger.debug(`Background job ${jobId}: Calling P&T account permissioning service for revoking permission from TID ${tid}`);
            response = await this.ptProxy.delete('/permissions/' + tid); // revoke permission
        } catch (err) {
            // we failed to call the P&T service: mark as failed and continue
            Logger.error(`Background job ${jobId}: Error calling P&T account permissioning service for TID ${tid}`, err);
            this.backgroundJobs.markAccountFailed(jobId, tid, 'Call to P&T account permissioning service failed');

            // Process next account
            this.processNextAccountInBackground(jobId);
            return;
        }

        if (response.status !== 202) {
            // the call to the P&T service was unsuccessful: mark as failed and continue
            Logger.warn(`Background job ${jobId}: Call to P&T account permissioning service FAILED for TID ${tid}, status code is ${response.status}`);
            this.backgroundJobs.markAccountFailed(jobId, tid, `P&T service returned status ${response.status}`);

            // Process next account
            this.processNextAccountInBackground(jobId);
            return;
        }

        Logger.debug(`Background job ${jobId}: Call to P&T account permissioning service successful for TID ${tid}`);
        // Success - the callback will handle marking as completed and processing the next account
    }

    private findBackgroundJobForAccount(tid: string): string | null {
        // Iterate through all background jobs to find which one is currently processing this account
        for (const [jobId, job] of this.backgroundJobs.getAllJobs().entries()) {
            if (job.currentItem?.tid === tid) {
                return jobId;
            }
        }
        return null;
    }
}
