import { Injectable, Logger } from '@nestjs/common';
import { InjectDataSource, InjectRepository } from '@nestjs/typeorm';
import { DataSource, EntityManager, IsNull, Repository } from 'typeorm';
import { AccountEntity, AuthorityEntity, MemberEntity, TransactionEntity } from './membership-entities';
import { EntityStatus, OrganizationType } from 'src/constants';
import { MemberFilter } from 'src/dtos/member-filter';
import { ErrorCondition, isDID, isShortUUID, throwExceptionWithErrorCondition } from 'src/utils';
import { AccountFilter } from 'src/dtos/account-filter';
import { AuthorityFilter } from 'src/dtos/authority-filter';
import { UserInfo } from 'src/auth/usser-info';
import { UserQualifiedId } from 'src/dtos/user-qualified-id';
import { MemberUpdateRequest } from 'src/dtos/member-update-request';
import { OAChangeContext } from 'src/dtos/oa-change-context';

@Injectable()
export class MembershipStorage {

    constructor(
        @InjectRepository(MemberEntity)
        private memberRepo: Repository<MemberEntity>,
        @InjectRepository(AuthorityEntity)
        private authorityRepo: Repository<AuthorityEntity>,
        @InjectRepository(AccountEntity)
        private accountRepo: Repository<AccountEntity>,
        @InjectRepository(TransactionEntity)
        private transactionRepo: Repository<TransactionEntity>,
        @InjectDataSource()
        private dataSource: DataSource
    ) { }

    async createMembership(
        pid: string,
        nam: string,
        cnt: string,
        rep: string,
        typ: OrganizationType,
        autRef: string,
    ): Promise<void> {

        // Atomic transaction: no persistent effects if any error occurs
        await this.dataSource.transaction(async manager => {
            let newAuthority: boolean = false;
            if (isDID(autRef)) {
                // the new member is going to have onboarding authority status
                newAuthority = true;
            } else if (!isShortUUID(autRef)) {
                // the argument is neither a DID nor a PID
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Invalid format for authority reference: ' + autRef)
            }

            let existingAuthority: MemberEntity = null;
            if (!newAuthority) {
                // The intention of the caller is to set a pre-existing member
                // as the onboarding authority for this new member (aut is a PID):
                // find the matching MemberEntity, ensuring that it has onboarded
                // status and that it is the owner of an authority with active status
                const resolved: MemberEntity = await this.resolveAuthorityFromPid(autRef, manager);
                if (resolved) {
                    existingAuthority = resolved;
                } else {
                    throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Could not resolve any active authority from PID ' + autRef)
                }
            }

            // Insert a new record in member table. If existingAuthority is null,
            // the member will be created with a self-reference as the onboarding authority,
            // while the next step will create a new child record in the authority table.
            const member = await this.createMember(manager, pid, nam, cnt, typ, rep, existingAuthority);

            if (!existingAuthority) {
                // The intention of the caller is to set the new member as its own
                // onboarding authority (aut is a DID): create a new record
                // in the authority table with this member as its parent.
                await this.createAuthority(autRef, member, manager);
            }
        });
    }

    async offboardMember(pid: string): Promise<Set<string>> {
        const disenrolledAccounts: Set<string> = new Set<string>();

        // Atomic transaction: no persistent effects if any error occurs
        await this.dataSource.transaction(async manager => {
            const params = {
                where: {
                    pid: pid,
                    ofb: IsNull(),
                },
            };
            const member = await manager.findOne(MemberEntity, params);
            if (!member) {
                throw new Error('No onboarded member with PID ' + pid + ' exists');
            }

            const offboardingDate = new Date();
            member.ofb = offboardingDate;
            await manager.save(member);

            // Disenroll all accounts directly owned by this member
            const accounts = await member.accounts;
            for (const account of accounts) {
                if (!account.dct) {
                    account.dct = offboardingDate;
                    await manager.save(account);
                    disenrolledAccounts.add(account.tid); // keep track of disenrolled accounts
                }
            }

            // Deactivate all authorities owned by this member.
            // Note that we did previously check that no other current members depend on this member as their OA.
            const authorities = await member.authorities;
            for (const authority of authorities) {
                if (!authority.dct) {
                    authority.dct = offboardingDate;

                    // Disenroll all accounts that are NOT directly owned by this member but that
                    // are linked to this member as the owner's OA.
                    // Note however that if we block the offboarding of members because of dangling dependencies,
                    // when we reach this point there should not be any accounts matching this condition. 
                    const accounts = await authority.accounts;
                    for (const account of accounts) {
                        if (!account.dct) {
                            account.dct = offboardingDate;
                            await manager.save(account);
                            disenrolledAccounts.add(account.tid); // keep track of disenrolled accounts
                        }
                    }
                    await manager.save(authority);
                }
            }
        });

        return disenrolledAccounts;
    }

    async disenrolAccountsByOwner(own: UserQualifiedId): Promise<Set<string>> {
        const disenrolledAccounts: Set<string> = new Set<string>();

        // Atomic transaction: no persistent effects if any error occurs
        await this.dataSource.transaction(async manager => {
            const params = {
                where: {
                    usr: own.uid,
                    authority: { did: own.oa },
                    dct: IsNull(),
                },
            }
            const accounts = await manager.find(AccountEntity, params);
            const disenrollmentDate = new Date();
            for (const account of accounts) {
                account.dct = disenrollmentDate;
                await manager.save(account);
                disenrolledAccounts.add(account.tid); // keep track of disenrolled accounts
            }
        });

        return disenrolledAccounts;
    }

    async migrateMember(
        oldPid: string,
        newPid: string,
        nam: string,
        cnt: string,
        rep: string,
        typ: OrganizationType,
        autRef: string,
    ): Promise<void> {

        // Atomic transaction: all database operations in single transaction
        await this.dataSource.transaction(async manager => {

            // Step 1: Retrieve and validate the old member
            const oldMember = await manager.findOne(MemberEntity, {
                where: {
                    pid: oldPid,
                    ofb: IsNull(),
                },
            });
            if (!oldMember) {
                throw new Error('No onboarded member with PID ' + oldPid + ' exists');
            }

            // Step 2: Set offboarding date on old member
            const migrationDate = new Date();
            oldMember.ofb = migrationDate;
            await manager.save(oldMember);

            // Step 3: Resolve the authority reference for the new member
            let newAuthority: boolean = false;
            if (isDID(autRef)) {
                // the new member is going to have onboarding authority status
                newAuthority = true;
            } else if (!isShortUUID(autRef)) {
                // the argument is neither a DID nor a PID
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Invalid format for authority reference: ' + autRef)
            }

            let existingAuthority: MemberEntity = null;
            if (!newAuthority) {
                // The intention is to set a pre-existing member as the onboarding authority
                const resolved: MemberEntity = await this.resolveAuthorityFromPid(autRef, manager);
                if (resolved) {
                    existingAuthority = resolved;
                } else {
                    throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Could not resolve any active authority from PID ' + autRef)
                }
            }

            // Step 4: Create the new member record
            const newMember = await this.createMember(manager, newPid, nam, cnt, typ, rep, existingAuthority);

            if (!existingAuthority) {
                // The new member is its own onboarding authority: create a new authority record
                await this.createAuthority(autRef, newMember, manager);
            }

            // Step 5: Transfer all authorities from old member to new member
            const authorities = await oldMember.authorities;
            for (const authority of authorities) {
                authority.owner = newMember;
                await manager.save(authority);
            }

            // Step 6: Transfer all directly-owned accounts from old member to new member
            const accounts = await oldMember.accounts;
            for (const account of accounts) {
                account.owner = newMember;
                await manager.save(account);
            }

            // Note: Accounts linked through authority references are automatically correct
            // because we updated the authority ownership in Step 5, and the authority
            // entity has eager loading, so the account.authority.owner will now point
            // to the new member.
        });
    }

    async updateMember(member: MemberEntity, req: MemberUpdateRequest, oaContext: OAChangeContext | null): Promise<string | null> {
        let deactivatedDID: string | null = null;

        // Validate consistency between req.oa and oaContext
        if (req.oa !== undefined && !oaContext) {
            throw new Error('Internal error: OA change context not provided when OA is being updated');
        }
        if (req.oa === undefined && oaContext) {
            throw new Error('Internal error: OA change context provided but OA is not being updated');
        }

        // Atomic transaction: all updates are applied together or none at all
        await this.dataSource.transaction(async manager => {

            // Update organization type if provided
            if (req.type !== undefined) {
                member.typ = req.type;
            }

            // Update representative contact info if provided
            if (req.rep !== undefined) {
                // Handle explicit null to clear the field
                member.rep = req.rep === null ? null : JSON.stringify(req.rep);
            }

            // Update onboarding authority reference if provided
            if (req.oa !== undefined) {

                if (!oaContext.currentOAIsSelf) {

                    // SCENARIO 1: Member currently delegates OA role
                    Logger.log(`Member ${member.pid} is changing its OA role delegation`);
                    if (oaContext.newOAIsSelf) {
                        // SCENARIO 1B: assume OA role
                        Logger.log(`Assuming OA role with DID ${req.oa}`);
                        await this.createAuthority(req.oa, member, manager);
                        member.onboarder = Promise.resolve(member);
                    } else {
                        // SCENARIO 1A: Delegating to different member (use pre-resolved data)
                        const delegateMember = oaContext.delegateMember!;
                        Logger.log(`Delegating OA role to member ${delegateMember.pid}`);
                        member.onboarder = Promise.resolve(delegateMember);
                    }
                    // no need for account updates in this scenario

                } else {

                    // SCENARIO 2: Member currently has OA role
                    const currentAuthority = oaContext.currentAuthority!;
                    Logger.log(`Member ${member.pid} is changing its own OA role (current DID: ${currentAuthority.did})`);

                    // Deactivate the current OA DID
                    Logger.log(`Deactivating OA with DID ${currentAuthority.did}`);
                    currentAuthority.dct = new Date();
                    await manager.save(currentAuthority);
                    deactivatedDID = currentAuthority.did;

                    // Determine new DID for account updates
                    let newDID: string;
                    let newAuthority: AuthorityEntity;
                    if (oaContext.newOAIsSelf) {
                        // SCENARIO 2B: keep OA role but change DID (value provided by user)
                        newDID = req.oa;
                        Logger.log(`Creating new OA with DID ${newDID}`);
                        newAuthority = await this.createAuthority(newDID, member, manager);
                        member.onboarder = Promise.resolve(member);
                    } else {
                        // SCENARIO 2A: delegate OA role to another member (reference provided by user)
                        const delegateMember = oaContext.delegateMember!;
                        const delegateAuthority = oaContext.delegateAuthority!;
                        newDID = delegateAuthority.did;
                        newAuthority = delegateAuthority;
                        Logger.log(`Delegating OA role to member ${delegateMember.pid} with DID ${newDID}`);
                        member.onboarder = Promise.resolve(delegateMember);
                    }

                    // Update all active accounts linked to the old DID
                    const affectedAccounts = await manager.find(AccountEntity, {
                        where: {
                            authority: { did: deactivatedDID },
                            dct: IsNull(), // only active accounts
                        },
                        relations: ['authority'],
                    });
                    if (affectedAccounts.length > 0) {
                        Logger.log(`Updating ${affectedAccounts.length} active account(s) linked to deactivated DID ${deactivatedDID}`);
                        for (const account of affectedAccounts) {
                            account.authority = newAuthority;
                            Logger.log(`Re-linking account ${account.tid} to DID ${newDID}`);
                            await manager.save(account);
                        }
                    }
                
                }
            }

            // Save the updated member record
            await manager.save(member);
        });

        return deactivatedDID; // null if no authority was deactivated
    }

    getMembers(query?: MemberFilter): Promise<MemberEntity[]> {
        if (!query) {
            return this.memberRepo.find(); // no filters shortcut
        }

        const queryBuilder = this.memberRepo.createQueryBuilder('member');

        if (query.nam) {
            // Partial match, case insensitive using LIKE with LOWER()
            queryBuilder.andWhere('LOWER(member.nam) LIKE LOWER(:nam)', {
                nam: `%${query.nam}%`
            });
        }
        if (query.typ) {
            queryBuilder.andWhere('member.typ = :typ', { typ: query.typ });
        }
        if (query.cnt) {
            queryBuilder.andWhere('member.cnt = :cnt', { cnt: query.cnt });
        }
        if (query.sta === EntityStatus.ACTIVE) {
            queryBuilder.andWhere('member.ofb IS NULL');
        } else if (query.sta === EntityStatus.INACTIVE || query.sta === EntityStatus.TERMINATED) {
            queryBuilder.andWhere('member.ofb IS NOT NULL');
        }

        // Apply sorting
        queryBuilder.orderBy('member.onb', 'DESC');

        // Apply pagination if provided
        if (query.l !== undefined) {
            queryBuilder.limit(query.l);
        }
        if (query.o !== undefined) {
            queryBuilder.offset(query.o);
        }

        return queryBuilder.getMany();
    }

    getActiveMembersUnderAuthority(oaPid: string): Promise<MemberEntity[]> {
        return this.memberRepo
            .createQueryBuilder('member')
            .where('member.aut = :oaPid', { oaPid })
            .andWhere('member.ofb IS NULL')
            .getMany();
    }

    getMember(pid: string): Promise<MemberEntity | null> {
        return this.memberRepo.findOneBy({ pid });
    }

    getActiveMember(pid: string): Promise<MemberEntity | null> {
        return this.memberRepo.findOne({
            where: {
                pid: pid,
                ofb: IsNull(),
            },
        });
    }

    getAuthorities(query?: AuthorityFilter): Promise<AuthorityEntity[]> {
        const queryBuilder = this.authorityRepo.createQueryBuilder('authority')
            .leftJoinAndSelect('authority.owner', 'owner');

        if (query) {
            if (query.own) {
                queryBuilder.andWhere('owner.pid = :own', { own: query.own });
            }

            if (query.sta === EntityStatus.ACTIVE) {
                queryBuilder.andWhere('authority.dct IS NULL');
            } else if (query.sta === EntityStatus.INACTIVE || query.sta === EntityStatus.TERMINATED) {
                queryBuilder.andWhere('authority.dct IS NOT NULL');
            }
        }

        // Apply sorting
        queryBuilder.orderBy('authority.act', 'DESC');

        // Apply pagination if provided
        if (query?.l !== undefined) {
            queryBuilder.limit(query.l);
        }
        if (query?.o !== undefined) {
            queryBuilder.offset(query.o);
        }

        return queryBuilder.getMany();
    }

    getAuthority(did: string): Promise<AuthorityEntity | null> {
        return this.authorityRepo.findOneBy({ did });
    }

    getActiveAuthority(did: string): Promise<AuthorityEntity | null> {
        return this.authorityRepo.findOne({
            where: {
                did: did,
                dct: IsNull(),
            },
        });
    }

    getActiveAuthorityByOwner(pid: string): Promise<AuthorityEntity | null> {
        return this.authorityRepo.findOne({
            where: {
                owner: { pid: pid },
                dct: IsNull(),
            },
        });
    }

    getAuthorityOwner(pid: string): Promise<MemberEntity | null> {
        return this.resolveAuthorityFromPid(pid);
    }

    async createAccount(tid: string, aut: string, usr: string, own?: string): Promise<AccountEntity> {
        const account: AccountEntity = new AccountEntity();
        account.tid = tid;
        account.usr = usr;
        account.act = new Date();
        const authority: AuthorityEntity = await this.getActiveAuthority(aut);
        if (authority) {
            account.authority = authority;
        } else {
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'No active authority with DID ' + aut);
        }

        if (own) {
            const owner: MemberEntity = await this.getActiveMember(own);
            if (owner) {
                account.owner = owner;
            } else {
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'No active member with PID ' + own);
            }
        }

        await this.accountRepo.insert(account); // throws exception if duplicate TID
        return await this.accountRepo.findOne({ where: { tid } });
    }

    async deactivateAccount(tid: string): Promise<AccountEntity> {
        const account: AccountEntity = await this.getAccount(tid);
        if (!account) {
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Account with TID ' + tid + ' does not exist');
        }
        if (account.dct) {
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Account with TID ' + tid + ' is already deactivated');
        }
        account.dct = new Date();
        return await this.accountRepo.save(account);
    }

    getAccounts(query?: AccountFilter, user?: UserInfo): Promise<AccountEntity[]> {
        const queryBuilder = this.accountRepo.createQueryBuilder('account')
            .leftJoinAndSelect('account.owner', 'owner')
            .leftJoinAndSelect('account.authority', 'authority');

        if (query) {
            if (query.usr) {
                queryBuilder.andWhere('account.usr = :usr', { usr: query.usr });
            }

            if (query.own) {
                queryBuilder.andWhere('owner.pid = :own', { own: query.own });
            }

            if (query.sta === EntityStatus.TERMINATED) {
                // condition on authority, if set, conflicts with status == "disabled" and is ignored
                queryBuilder.andWhere('account.authority IS NULL');
                queryBuilder.andWhere('account.dct IS NOT NULL');
            } else {
                // any other condition on status is compatible with the condition on authority
                if (query.aut) {
                    queryBuilder.andWhere('authority.did = :aut', { aut: query.aut });
                } else {
                    queryBuilder.andWhere('account.authority IS NOT NULL');
                }
                if (query.sta === EntityStatus.ACTIVE) {
                    queryBuilder.andWhere('account.dct IS NULL');
                } else if (query.sta === EntityStatus.INACTIVE) {
                    queryBuilder.andWhere('account.dct IS NOT NULL');
                }
            }
        }

        // if user is defined, this is a profiled query: user profile overrides any conflicting conditions
        if (user) {
            // as this comes from our auth guard, we assume both values are always set
            queryBuilder.andWhere('account.usr = :userUid', { userUid: user.uid });
            if (query?.sta === EntityStatus.TERMINATED) {
                queryBuilder.andWhere('account.authority IS NULL');
            } else {
                queryBuilder.andWhere('authority.did = :issuerDID', { issuerDID: user.issuerDID });
            }
        }

        // Apply sorting
        queryBuilder.orderBy('account.act', 'DESC');

        // Apply pagination if provided
        if (query?.l !== undefined) {
            queryBuilder.limit(query.l);
        }
        if (query?.o !== undefined) {
            queryBuilder.offset(query.o);
        }

        return queryBuilder.getMany();
    }

    getActiveAccountsForUser(user: UserInfo, pagination?: { l?: number; o?: number }): Promise<AccountEntity[]> {
        const queryBuilder = this.accountRepo.createQueryBuilder('account')
            .leftJoinAndSelect('account.authority', 'authority')
            .where('account.usr = :usr', { usr: user.uid })
            .andWhere('authority.did = :issuerDID', { issuerDID: user.issuerDID })
            .andWhere('account.dct IS NULL')
            .orderBy('account.act', 'DESC');

        // Apply pagination if provided
        if (pagination?.l !== undefined) {
            queryBuilder.limit(pagination.l);
        }
        if (pagination?.o !== undefined) {
            queryBuilder.offset(pagination.o);
        }

        return queryBuilder.getMany();
    }

    getAccount(tid: string): Promise<AccountEntity> {
        return this.accountRepo.findOne({ where: { tid } })
    }

    private async createMember(
        manager: EntityManager,
        pid: string,
        nam: string,
        cnt: string,
        typ: OrganizationType,
        rep?: string,
        onboarder?: MemberEntity
    ): Promise<MemberEntity> {

        // Due to the need of managing a self-reference, we are forced to use save()
        // instead of insert(), so we must check for a potential key clash that would
        // turn this operation into an unwanted update.
        if (await manager.findOne(MemberEntity, { where: { pid } })) {
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Duplicate key detected for member with PID ' + pid)
        }

        const member = new MemberEntity();
        member.pid = pid;
        member.nam = nam;
        member.cnt = cnt;
        member.typ = typ;

        // The onboarder property uses lazy loading, so we have to wrap any assigned value into a resolved Promise
        if (onboarder) {
            member.onboarder = Promise.resolve(onboarder); // reference to another member
        } else {
            member.onboarder = Promise.resolve(member); // self-reference
        }

        if (rep) {
            member.rep = rep;
        }

        member.onb = new Date();

        return await manager.save(member);
    }

    private async createAuthority(did: string, owner: MemberEntity, manager?: EntityManager): Promise<AuthorityEntity> {
        const authority: AuthorityEntity = new AuthorityEntity();
        authority.did = did;
        authority.owner = owner;
        authority.act = new Date();
        const params = { where: { did } };

        if (manager) {
            // in-transaction context (managed by the caller)
            await manager.insert(AuthorityEntity, authority); // throws exception if duplicate DID
            return await manager.findOne(AuthorityEntity, params);
        } else {
            await this.authorityRepo.insert(authority); // throws exception if duplicate DID
            return await this.authorityRepo.findOne(params);
        }
    }

    private async resolveAuthorityFromPid(pid: string, manager?: EntityManager): Promise<MemberEntity | null> {
        let member: MemberEntity = null;
        const params = {
            where: {
                pid: pid,
                ofb: IsNull(),
            },
        };

        if (manager) {
            // in-transaction context (managed by the caller)
            member = await manager.findOne(MemberEntity, params);
        } else {
            member = await this.memberRepo.findOne(params);
        }

        if (member) {
            // Member should own at least one active authority.
            const authorities = await member.authorities; // lazy loading
            const activeAuthorities = authorities.filter(authority => authority.dct === null);
            return activeAuthorities.length > 0 ? member : null;
        } else {
            return null;
        }

    }

    async insertAccountTransaction(trx: string, tms: Date, acc: string, dir: number, amt: string): Promise<void> {
        const transaction = new TransactionEntity();
        transaction.trx = trx;
        transaction.tms = tms;
        transaction.acc = acc;
        transaction.dir = dir;
        transaction.amt = amt;
        await this.transactionRepo.insert(transaction);
    }

    async deleteAccountTransaction(trx: string): Promise<void> {
        await this.transactionRepo.delete({ trx });
    }

    async getTransactionsByAccount(acc: string): Promise<TransactionEntity[]> {
        return this.transactionRepo.find({
            where: { acc },
            order: { tms: 'DESC' }
        });
    }
}
