import { Body, Controller, HttpCode, Post, } from '@nestjs/common';
import { ApiBody, ApiOperation, ApiResponse, ApiTags } from "@nestjs/swagger";
import { OAService } from "./oa-service";
import { translateToHttpException } from 'src/utils';
import { CredentialClaim } from 'src/dtos/credential-claim';
import { CredentialOfferingState } from 'src/dtos/credential-offering-state';
import { HTTP_ERR_400, HTTP_ERR_404, HTTP_ERR_500, HTTP_ERR_502 } from 'src/constants';

// Anything here is open to public access by non-authenticated users

@ApiTags('Open API: Invitation Hub')
@Controller('/api/v1.0/invitations')
export class OAControllerOpen {

    constructor(private readonly oa: OAService) {
    }

    @Post()
    @HttpCode(201)
    @ApiOperation({
        summary: 'Accept onboarding invitation',
        description: `Completes the user onbarding process. This endpoint is called by the user after following the link
 in the invitation message. The call must include the invitation ID and the one-time password (OTP) provided in the invitation
 message. Once the OTP is verified, the caller receives a URI that can be used to claim the Verifiable Onboarding Credential
 (VOC) from the FAME Root Onboarding Authority's VOC issuance service.`
    })
    @ApiBody({
        type: CredentialClaim,
        description: `The request body contains the invitation ID and the one-time password (OTP) provided in the invitation message.`,
        required: true,
    })
    @ApiResponse({
        status: 201, // Created
        description: `The invitation has been accepted and the user is now registered. The response body contains the state
 the information required to retireve the VOC from the issuer.`, 
        type: CredentialOfferingState
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async acceptInvitation(@Body() inv: CredentialClaim): Promise<CredentialOfferingState> {
        try {
            return await this.oa.acceptInvitation(inv);
        } catch (err) {
            translateToHttpException(err);
        }
    }
}