import { Post, HttpCode, Controller, UseGuards, Body, Delete, Get, Param, Query } from '@nestjs/common';
import { ApiOperation, ApiBody, ApiResponse, ApiTags, ApiBearerAuth, ApiParam, ApiQuery } from '@nestjs/swagger';
import { ROLE_ADMIN, ROLE_OPERATOR } from 'src/auth/auth-constants';
import { Roles } from 'src/auth/roles-decorator';
import { HTTP_ERR_400, HTTP_ERR_401, HTTP_ERR_403, HTTP_ERR_404, HTTP_ERR_500, HTTP_ERR_502, UserRole, UserStatus } from 'src/constants';
import { Identifier } from 'src/dtos/identifier';
import { UserFilter } from 'src/dtos/user-filter';
import { UserOnboardingRequest } from 'src/dtos/user-onboarding-request';
import { UserRef } from 'src/dtos/user-ref';
import { User } from 'src/dtos/user';
import { translateToHttpException } from 'src/utils';
import { OAService } from './oa-service';
import { AuthGuard } from 'src/auth/auth-guard';
import { RolesGuard } from 'src/auth/roles-guard';

@ApiTags('Open API: User Management')
@Controller('/api/v1.0/users')
@ApiBearerAuth()
@UseGuards(AuthGuard, RolesGuard)
export class OAController {

    constructor(private readonly oa: OAService) { }

    @Post()
    @HttpCode(202)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Starts the onboarding process of a candidate user',
        description: `Initiates the process for onboarding a new user of the FAME Marketplace, by creating a record
 in the database of users of the FAME Root Onboarding Authority and sending an invitation message to the email address
 provided as part of the user's contact information. The invitation message contains a link to a web page where the user
 can start claiming their Verifiable Oboarding Credential (VOC), which will be used to authenticate their identity in
 any future interaction with the FAME Marketplace platform.`
    })
    @ApiBody({
        type: UserOnboardingRequest,
        description: `The request body contains the user information and contact details, including the email address to
 which the invitation message will be sent.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The user record has been created and an invitation message has been sent.
 The response body contains the UID assigned to the new user by the system.`,
        type: Identifier
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async onboardUser(@Body() user: UserOnboardingRequest): Promise<Identifier> {
        try {
            return await this.oa.createAndInviteUser(user);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/:uid/reinvite')
    @HttpCode(204)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Restarts from scratch the onboarding process of a candidate user',
        description: `Resends the invitation message to the email address in the user's contact information, in case the
 user has lost the original message or the link has expired. It also resets the status of the user record in the
 database, so that the user can start the onboarding process from scratch. A new invitation ID is generated.`
    })
    @ApiParam({
        name: 'uid',
        required: true,
        type: String,
        description: 'UID of the target user'
    })
    @ApiResponse({
        status: 204, // No Content
        description: 'The user record has been updated and a new invitation message has been sent.',
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async reinviteUser(@Param('uid') uid: string): Promise<void> {
        try {
            await this.oa.reinviteUser(uid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get()
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'List users',
        description: `Returns a list of references pointing to all the matching user record in the FAME Root Onboarding
 Authority database.`,
    })
    @ApiQuery({
        name: 'iid',
        description: `Filter by invitation ID (OPTIONAL, 22-characters short UUID)`,
        type: String,
        required: false
    })
    @ApiQuery({
        name: 'rol',
        description: `Filter by user role (OPTIONAL, accepted values: [${Object.values(UserRole).join(', ')}])`,
        enum: UserRole,
        required: false
    })
    @ApiQuery({
        name: 'aff',
        description: `Filter by affiliation PID (OPTIONAL, 22-characters short UUID)`,
        type: String,
        required: false
    })
    @ApiQuery({
        name: 'cnt',
        description: 'Filter by country of residence (OPTIONAL, ISO 3166-1 alpha-2 code)',
        type: String,
        required: false
    })
    @ApiQuery({
        name: 'lname',
        description: 'Filter by last name (OPTIONAL, partial match, case insensitive)',
        type: String,
        required: false
    })
    @ApiQuery({
        name: 'email',
        description: 'Filter by email address (OPTIONAL, partial match, case insensitive)',
        type: String,
        required: false
    })
    @ApiQuery({
        name: 'sta',
        description: `Filter by user status (OPTIONAL, accepted values: [${Object.values(UserStatus).join(', ')}])`,
        enum: UserStatus,
        required: false
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return (default: no limit, max: 1000)'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return (default: 0)'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The response body conytains an array of references to users matching the given criteria',
        type: UserRef,
        isArray: true
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async listUsers(@Query() query?: UserFilter): Promise<UserRef[]> {
        try {
            return await this.oa.listUser(query);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get('/:uid')
    @HttpCode(200)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Retrieve user',
        description: `Retrieves the database record of a given user.`
    })
    @ApiParam({
        name: 'uid',
        required: true,
        type: String,
        description: 'UID of the user to retrieve'
    })
    @ApiResponse({
        status: 200, // OK
        description: `The user record has been found and is returned in the response body.`,
        type: User
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async retrieveUser(@Param('uid') uid: string): Promise<User> {
        try {
            return await this.oa.retrieveUser(uid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Post('/:uid/offboard')
    @HttpCode(204)
    @Roles(ROLE_ADMIN, ROLE_OPERATOR) // privileged operation
    @ApiOperation({
        summary: 'Offboards a user',
        description: `Updates the record in the database of the FAME Root Onboarding Authority to reflect that the user
 has been offboarded. The user record is not deleted. WARNING: this operation is irreversible. NOTE: currently, any VOCs
 already issued to the user are NOT revoked, as this feature is yet to be implemented.`
    })
    @ApiParam({
        name: 'uid',
        required: true,
        type: String,
        description: 'UID of the target user'
    })
    @ApiResponse({
        status: 204, // No Content
        description: 'The user record has been updated and a new invitation message has been sent.',
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async offboardUser(@Param('uid') uid: string): Promise<void> {
        try {
            await this.oa.offboardUser(uid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Delete('/:uid')
    @HttpCode(204)
    @Roles(ROLE_ADMIN) // admin-only operation
    @ApiOperation({
        summary: 'Deletes a user',
        description: `Physically deletes a user record from the database of the FAME Root Onboarding Authority. This operation
 can only be done oN users that have OFFBOARDED status. WARNING: this operation is irreversible.`
    })
    @ApiParam({
        name: 'uid',
        required: true,
        type: String,
        description: 'UID of the target user'
    })
    @ApiResponse({
        status: 204, // No Content
        description: 'The user record has been deleted.',
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    async deleteUser(@Param('uid') uid: string): Promise<void> {
        try {
            await this.oa.deleteUser(uid);
        } catch (err) {
            translateToHttpException(err);
        }
    }
}
