import { UserOnboardingRequest } from "../dtos/user-onboarding-request";
import { UserEntity } from "./user-entity";
import { OAStorage } from "./oa-storage";
import { Injectable, Logger } from "@nestjs/common";
import { EmailService } from "../integration/email-service";
import { ConfigService } from "@nestjs/config";
import * as short from 'short-uuid';
import axios, { AxiosInstance } from 'axios';
import { ErrorCondition, throwExceptionWithErrorCondition } from 'src/utils';
import { CredentialClaim } from 'src/dtos/credential-claim';
import { CredentialOfferingState } from 'src/dtos/credential-offering-state';
import { Identifier } from 'src/dtos/identifier';
import { UserRef } from 'src/dtos/user-ref';
import { User } from 'src/dtos/user';
import { UserFilter } from 'src/dtos/user-filter';
import { MembershipService } from 'src/membership/membership-service';
import { ROA_DID } from 'src/auth/auth-constants';

@Injectable()
export class OAService {

    private readonly MAX_ATTEMPTS: number = 5; // five OTP attempts is the maximum before invalidating an invitation
    private readonly MAX_TIME: number = 72 * 60 * 60 * 1000; // three days is the maximum time allowed to accept an invitation
    private readonly agentProxy: AxiosInstance = null;
    private readonly membershipProxy: AxiosInstance = null;

    constructor(
        private readonly userDb: OAStorage,
        private readonly membership: MembershipService,
        private readonly mailer: EmailService,
        private readonly config: ConfigService) {
        this.agentProxy = axios.create({
            baseURL: this.config.get<string>('VOC_ISSUANCE_URL'), // Base URL from environment variables
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
        this.membershipProxy = axios.create({
            baseURL: this.config.get<string>('INTERNAL_URL'), // Internal membership service URL
            timeout: 5000, // 5 seconds timeout for every request
            headers: { 'Content-Type': 'application/json' },
        });
    }

    public async createAndInviteUser(input: UserOnboardingRequest): Promise<Identifier> {

        // Affiliation is optional, but if provided it must match an active member
        if (input.affiliation && input.affiliation.trim().length > 0) {
            if (!await this.checkAffiliation(input.affiliation)) {
                Logger.warn(`Bad user request: PID ${input.affiliation} does not match any active member`);
                throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Invalid affiliation');
            }
        }

        const user: UserEntity = {
            uid: short.generate(),
            iid: short.generate(),
            inv: new Date(),
            otp: Math.floor(10000 + Math.random() * 90000).toString(),
            att: 0,
            rol: input.role,
            cnt: input.country,
            aff: input.affiliation,
            fname: input.contact.firstName,
            lname: input.contact.lastName,
            email: input.contact.email,
            phone: input.contact.phone,
            nck: input.nickname
        };

        try {
            await this.userDb.saveUser(user);
        } catch (err) {
            Logger.error(`Error creating user database record with UID ${user.uid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not create user record');
        }

        try {
            await this.sendInvitation(user);
        } catch (err) {
            Logger.error(`Error sending invitation to user with UID ${user.uid}, email ${user.email}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Could not send invitation to user');
        }

        return { id: user.uid };
    }

    public async reinviteUser(uid: string): Promise<void> {
        const user = await this.userDb.getUserById(uid);
        if (!user) {
            Logger.warn(`Bad user request: UID ${uid} does not match any user`);
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'User not found');
        }

        if (user.ofb) {
            Logger.warn(`Bad user request: UID ${uid} corresponds to an offboarded user`);
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'User has been offboarded');
        }

        user.iid = short.generate();
        user.inv = new Date();
        user.otp = Math.floor(10000 + Math.random() * 90000).toString();
        user.onb = null;
        user.att = 0;
        try {
            await this.userDb.saveUser(user);
        } catch (err) {
            Logger.error(`Error updating user database record with UID ${user.uid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not update user record');
        }

        try {
            await this.sendInvitation(user);
        } catch (err) {
            Logger.error(`Error sending invitation to user with UID ${user.uid}, email ${user.email}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Could not send invitation to user');
        }
    }

    public async acceptInvitation(inv: CredentialClaim): Promise<CredentialOfferingState> {
        const user = await this.userDb.getUserByInvitation(inv.iid);
        if (!user) {
            Logger.warn(`Bad user request: IID ${inv.iid} does not match any invitation`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Invitation not found');
        }

        if (user.onb || user.ofb) {
            Logger.warn(`Bad user request: IID ${inv.iid} corresponds to an already registered user`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'Invitation not found');
        }

        const now = new Date();
        if (user.inv < new Date(now.getTime() - this.MAX_TIME) || user.att >= this.MAX_ATTEMPTS) {
            Logger.warn(`Bad user request: IID ${inv.iid} corresponds to an expired invitation`);
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'Invitation has expired: contact the issuer for a new one');
        }

        if (user.att < this.MAX_ATTEMPTS) user.att++;
        if (user.otp !== inv.otp) {
            Logger.warn(`Bad user request: OTP value ${inv.otp} mismatch for invitation with IID ${inv.iid}`);
            try {
                await this.userDb.saveUser(user);
            } catch (err) {
                // log and swallow
                Logger.error(`Error updating number of attempts for user with UID ${user.uid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            }
            return {
                uri: null,
                preAuthCode: null,
                attemptsLeft: this.MAX_ATTEMPTS - user.att
            };
        }

        const payload = {
            uid: user.uid,
            affiliation: user.aff ? user.aff : "NA",
            region: user.cnt ? user.cnt : "NA",
            role: user.rol ? user.rol : "user",
            nickname: user.nck ? user.nck : "Anonymous",
        };
        const offering: CredentialOfferingState = await this.createOffering(payload);

        user.onb = now;
        try {
            await this.userDb.saveUser(user);
        } catch (err) {
            Logger.error(`Error onboarding user with UID ${user.uid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not update user record');
        }

        return offering;
    }

    public async listUser(query?: UserFilter): Promise<UserRef[]> {
        const results: UserEntity[] = await this.userDb.getUsers(query);
        return results.map(user => {
            return {
                uid: user.uid,
                role: user.rol,
                affiliation: user.aff,
                lastName: user.lname,
                country: user.cnt,
                active: (user.onb && !user.ofb) ? true : false,
            };
        });
    }

    public async retrieveUser(uid: string): Promise<User> {
        const result: UserEntity = await this.userDb.getUserById(uid);
        if (result) {
            return {
                uid: result.uid,
                iid: result.iid,
                otp: result.otp,
                otpAttempts: result.att,
                role: result.rol,
                country: result.cnt,
                affiliation: result.aff,
                nickname: result.nck,
                firstName: result.fname,
                lastName: result.lname,
                email: result.email,
                phone: result.phone,
                invited: result.inv ? result.inv.toISOString() : null,
                onboarded: result.onb ? result.onb.toISOString() : null,
                offboarded: result.ofb ? result.ofb.toISOString() : null,
            };
        } else {
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'User not found');
        }
    }

    public async offboardUser(uid: string): Promise<void> {
        const user = await this.userDb.getUserById(uid);
        if (!user) {
            Logger.warn(`Bad user request: UID ${uid} does not match any user`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'User not found');
        }

        if (user.ofb) {
            Logger.warn(`Bad user request: UID ${uid} corresponds to an already offboarded user`);
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'User already offboarded');
        }

        user.ofb = new Date();
        try {
            await this.userDb.saveUser(user);
        } catch (err) {
            Logger.error(`Error offboarding user with UID ${user.uid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not update user record');
        }

        // Now disenroll all accounts owned by this user by calling the internal membership endpoint
        try {
            const userQualifiedId = {
                oa: ROA_DID,
                uid: user.uid
            };

            // Create caller context to track this operation
            const callerContext = {
                source: 'internal_api',
                endpoint: '/api/v1.0/users/{uid}/offboard',
                timestamp: Date.now()
            };

            Logger.log(`Requesting account disenrollment for user ${ROA_DID}:${user.uid}`);

            await this.membershipProxy.post('/users/offboarding-notification', userQualifiedId, {
                headers: {
                    'x-caller-context': JSON.stringify(callerContext)
                }
            });

            Logger.log(`Account disenrollment request completed for user ${ROA_DID}:${user.uid}`);
        } catch (err) {
            Logger.error(`Failed to disenroll accounts for user ${uid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            // Don't throw - user is already offboarded, but log the failure
        }
    }

    public async deleteUser(uid: string): Promise<boolean> {
        const user = await this.userDb.getUserById(uid);
        if (!user) {
            Logger.warn(`Bad user request: UID ${uid} does not match any user`);
            throwExceptionWithErrorCondition(ErrorCondition.NOTFOUND, 'User not found');
        }

        if (!user.ofb) {
            Logger.warn(`Bad user request: user with UID ${uid} has not offboarded status`);
            throwExceptionWithErrorCondition(ErrorCondition.INPUT, 'User must be offboarded before deletion');
        }

        try {
            return (await this.userDb.deleteUser(uid)).affected > 0;
        } catch (err) {
            Logger.error(`Error deleting user with UID ${uid}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.INTERNAL, 'Could not delete user record');
        }
    }

    private async createOffering(payload: Record<string, string>): Promise<CredentialOfferingState> {
        const code = short.generate();

        const body = JSON.stringify({
            grants: {
                'urn:ietf:params:oauth:grant-type:pre-authorized_code': {
                    'pre-authorized_code': code,
                    user_pin_required: false,
                },
            },
            credentials: ["FAMEv1"],
            credentialDataSupplierInput: { ...payload }
        });

        try {
            const response: any = await this.agentProxy.post('/credential-offers', body);
            Logger.debug(`Credential issuer service response: ${JSON.stringify(response.data)}`);
            return {
                uri: response.data.uri,
                preAuthCode: code,
                attemptsLeft: -1
            };
        } catch (err) {
            Logger.error(`Error calling credential issuer service for user with UID ${payload.uid}, auth code ${code}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to credential issuer service failed')
        }
    };

    private async sendInvitation(user: UserEntity): Promise<void> {

        // Phase 1: send email with invitation link
        const to: string = user.email;
        const subject: string = 'Your FAME Onboarding request';
        const claimUrl = `${this.config.get<string>('VOC_CLAIM_URL')}${user.iid}`;

        const html = `
<!DOCTYPE html>
<html>
<head>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .button { display: inline-block; padding: 12px 24px; background-color: #007bff; color: #ffffff !important; font-weight: bold; text-decoration: none; border-radius: 4px; margin: 20px 0; }
        .button:visited { color: #ffffff !important; }
        .button:hover { background-color: #0056b3; color: #ffffff !important; }
        .button:active { color: #ffffff !important; }
        .warning { background-color: #fff3cd; border-left: 4px solid #ffc107; padding: 12px; margin: 16px 0; }
        .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 0.9em; color: #666; }
    </style>
</head>
<body>
    <div class="container">
        <h2>Welcome to the FAME Federation!</h2>

        <p>This message contains some important instructions on how to complete your onboarding process.</p>

        <p>
            Before you can claim your FAME Onboarding Credential, which is required to access the FAME Marketplace, you need two things:
            <ul>
                <li>A one-time-password (<strong>OTP</strong>), which has been sent to you separately.</li>
                <li>The <strong>FAME Identity Wallet</strong> app installed on your personal <u>Android</u> device - iOS is not yet supported, sorry.</li>
            </ul>
        </p>

        <p>
            The FAME Identity Wallet is used to securely store and manage your FAME Onboarding Credential. The app cannot be installed from the Google Play Store:
            instead, you need to download the APK file and install it manually, and also disable any future automated updates.
            The full procedure is explained in detail on the <a href="https://marketplace.fame-horizon.eu/about-us" target="_blank">FAME Marketplace website</a>,
            under the "Onboard Organization & Users" section. Read those instructions carefully, because there are some critical steps to be followed.
        </p>

        <p>Once the wallet app is installed and properly set up, you can claim your FAME Onboarding Credential by clicking the button below:</p>

        <div style="text-align: center;">
            <a href="${claimUrl}" class="button">Go claim your credential!</a>
        </div>

        <p>Or, you can just copy and paste this link into your browser: <a href="${claimUrl}">${claimUrl}</a></p>

        <div class="warning">
            <strong>Important!</strong> This invitation will expire in 72 hours. After that time, you will need to request a new invitation.
        </div>

        <div class="footer">
            <p>With best regards,<br/>
            <strong>The FAME Federation team</strong></p>

            <p><em>(This is an automated message, sent by a bot. PLEASE DO NOT REPLY: this email address is not monitored!)</em></p>
        </div>
    </div>
</body>
</html>`;

        await this.mailer.sendHtmlMail(to, subject, html);


        // Phase 2: send OTP
        // TODO this should be sent by SMS
        const otpHtml = `
<!DOCTYPE html>
<html>
<head>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .otp { font-size: 32px; font-weight: bold; color: #007bff; text-align: center; padding: 20px; background-color: #f8f9fa; border-radius: 8px; margin: 20px 0; letter-spacing: 4px; }
        .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 0.9em; color: #666; }
    </style>
</head>
<body>
    <div class="container">
        <h2>Your One-Time Password (OTP)</h2>

        <p>Your one-time-password (OTP) for onboarding is:</p>

        <div class="otp">${user.otp}</div>

        <p>Use this code when claiming your FAME Onboarding Credential.</p>

        <div class="footer">
            <p>With best regards,<br/>
            <strong>The FAME Federation team</strong></p>

            <p><em>(This is an automated message, sent by a bot. PLEASE DO NOT REPLY: this email address is not monitored!)</em></p>
        </div>
    </div>
</body>
</html>`;

        await this.mailer.sendHtmlMail(to, subject, otpHtml);
    }

    private async checkAffiliation(aff: string): Promise<boolean> {
        try {
            return (null !== await this.membership.retrieveActiveMemberRef(aff));
        } catch (err) {
            Logger.error(`Error calling membership service for PID ${aff}: ${err.name}, Message: ${err.message}, Stack: ${err.stack}`);
            throwExceptionWithErrorCondition(ErrorCondition.EXTERNAL, 'Call to membership service failed');
        }
    }
}
