import { Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { DeleteResult, IsNull, Not, Repository } from "typeorm";
import { UserEntity } from "./user-entity";
import { UserFilter } from 'src/dtos/user-filter';
import { UserStatus } from 'src/constants';

@Injectable()
export class OAStorage {

    constructor(
        @InjectRepository(UserEntity)
        private repo: Repository<UserEntity>,
    ) { }

    getUsers(query: UserFilter): Promise<UserEntity[]> {
        if (!query) {
            return this.repo.find(); // no filters shortcut
        }

        const queryBuilder = this.repo.createQueryBuilder('user');

        if (query.iid) {
            queryBuilder.andWhere('user.iid = :iid', { iid: query.iid });
        }
        if (query.rol) {
            queryBuilder.andWhere('user.rol = :rol', { rol: query.rol });
        }
        if (query.aff) {
            queryBuilder.andWhere('user.aff = :aff', { aff: query.aff });
        }
        if (query.cnt) {
            queryBuilder.andWhere('user.cnt = :cnt', { cnt: query.cnt });
        }
        if (query.lname) {
            // Partial match, case insensitive using LIKE with LOWER()
            queryBuilder.andWhere('LOWER(user.lname) LIKE LOWER(:lname)', {
                lname: `%${query.lname}%`
            });
        }
        if (query.email) {
            // Partial match, case insensitive using LIKE with LOWER()
            queryBuilder.andWhere('LOWER(user.email) LIKE LOWER(:email)', {
                email: `%${query.email}%`
            });
        }
        if (query.sta === UserStatus.INVITED) {
            queryBuilder.andWhere('user.inv IS NOT NULL');
            queryBuilder.andWhere('user.ofb IS NULL');
            queryBuilder.andWhere('user.onb IS NULL');
        } else if (query.sta === UserStatus.ONBOARDED) {
            queryBuilder.andWhere('user.onb IS NOT NULL');
            queryBuilder.andWhere('user.ofb IS NULL');
        } else if (query.sta === UserStatus.OFFBOARDED) {
            queryBuilder.andWhere('user.ofb IS NOT NULL');
        }

        // Apply sorting
        queryBuilder.orderBy('user.inv', 'DESC');

        // Apply pagination if provided
        if (query.l !== undefined) {
            queryBuilder.limit(query.l);
        }
        if (query.o !== undefined) {
            queryBuilder.offset(query.o);
        }

        return queryBuilder.getMany();
    }

    getUserById(uid: string): Promise<UserEntity | null> {
        return this.repo.findOneBy({ uid });
    }

    getUserByInvitation(iid: string): Promise<UserEntity | null> {
        return this.repo.findOneBy({ iid });
    }

    saveUser(userDb: UserEntity): Promise<UserEntity> {
        return this.repo.save(userDb);
    }

    deleteUser(uid: string): Promise<DeleteResult> {
        return this.repo.delete(uid);
    }
}
