import {
    BadRequestException, Body,
    Controller,
    Get,
    HttpStatus, NotFoundException,
    Param,
    Post,
    Put,
    Query
} from '@nestjs/common';
import {RequestsService} from './requests-service';
import {ApiOperation, ApiResponse, ApiTags} from "@nestjs/swagger";
import {Request} from "../dtos/request";
import {RequestQueue} from "../dtos/request-queue";

export const PENDING = 'PENDING';
export const PROCESSED = 'PROCESSED';
export const EXPIRED = 'EXPIRED';
export const FAILED = 'FAILED';

@ApiTags('Request Management')
@Controller()
export class RequestsInternalController {
    constructor(private readonly requestService: RequestsService) {
    }

    @ApiOperation({summary: 'Enqueue Request'})
    @Post('/gov/v1.0/rqueue')
    async enqueue(@Body() request: Request) {
        if (!request.rid) throw new BadRequestException('Rid is empty');
        if (!request.requestor) throw new BadRequestException('Requestor is empty');
        try {
            const request_ = await this.requestService.retrieve(request.rid);
            if (request_ && Object.keys(request_).length > 0)
                throw new BadRequestException(`Request with this ${request.rid} already exists`);
        } catch (error) {
            console.log('rqueue - error into retrieve: ', error);
            throw error
        }

        try {
            const record = await this.requestService.enqueue(request.rid, request.requestor, request?.payload, request?.message);
            if (record) return HttpStatus.OK;
        } catch (error) {
            console.log('rqueue - error into enqueue: ', error);
            throw error
        }
    }

    @ApiOperation({summary: 'Retrieve Request'})
    @ApiResponse({
        status: 200,
        description: 'The found record',
        type: RequestQueue
    })
    @Get('/gov/v1.0/rqueue/:rid')
    async retrieve(@Param('rid') rid: string): Promise<RequestQueue> {
        if (!rid) throw new BadRequestException('Rid is empty');

        try {
            const requestQueue = new RequestQueue();
            const request = await this.requestService.retrieve(rid);
            if (!request || Object.keys(request).length > 0)
                throw new NotFoundException(`Request with this ${rid} does not exist`);
            requestQueue.requestor = request?.requestor;
            requestQueue.enqueued = request?.enqueue;
            requestQueue.payload = request?.payload;
            requestQueue.finalized = request?.finalized;
            requestQueue.status = request?.status;
            requestQueue.message = request?.message;
            return requestQueue;
        } catch (error) {
            console.log('rqueue - error into retrieve: ', error);
            throw error;
        }
    }

    @ApiOperation({summary: 'Finalize Request'})
    @Put('/gov/v1.0/rqueue/:rid')
    async finalize(
        @Param('rid') rid: string,
        @Query('message') message: string,
        @Query('status') status: string,
    ): Promise<string> {
        if (!rid) throw new BadRequestException('Rid is empty');
        if (!status) throw new BadRequestException('status is empty');
        if (
            status != PROCESSED &&
            status != EXPIRED &&
            status != FAILED
        )
            throw new BadRequestException(
                `Status is not valid only ${PROCESSED} and ${EXPIRED}  and ${FAILED} are allowed`,
            );
        try {
            const request = await this.requestService.retrieve(rid);
            if (!request || Object.keys(request).length === 0)
                throw new BadRequestException(`Request with this ${rid} does not exist`);
            if (request.status != PENDING)
                throw new BadRequestException(`Request with this ${rid} is not pending`);
            return this.requestService.finalize(rid, status, message);
        } catch (error) {
            console.log('finalize - error into retrieve: ', error);
            throw error;
        }
    }
}
