import { Injectable, Logger } from '@nestjs/common';
import { InjectConnection } from '@nestjs/mongoose';
import mongoose, { Connection } from 'mongoose';
import { RequestSchema } from './request-schema';
import * as process from 'process';
import { PENDING } from './requests-controller';

@Injectable()
export class RequestsService {
  private readonly RequestModel: any;
  private collection: any;

  constructor(@InjectConnection() private connection: Connection) {

    this.collection = this.connection.db.collection(
      process.env.MONGO_COLLECTION,
    );
    this.RequestModel = mongoose.model(
      'RequestModel',
      RequestSchema,
      'request',
    );
  }

  async retrieve(rid: string) {
    try {
      if (this.connection.readyState == 0) {
        Logger.error('Connection is not ready');
        return;
      }
      return await this.collection.findOne({
        rid: rid,
      });
    } catch (e) {
      Logger.error('MongoDB Error : ' + e);
      throw e;
    }
  }


  async enqueue(rid: string, requestor: string, payload: string, message: string) {
    try {
      if (this.connection.readyState == 0) {
        Logger.error('Connection is not ready');
        return;
      }
      const record = await this.collection.insertOne({
        payload: payload,
        timestamp: new Date().toISOString(),
        requestor: requestor,
        rid: rid,
        status: PENDING,
        message: message,
      });
      Logger.log('Request inserted with id ' + record.insertedId);
      return record.insertedId;
    } catch (e) {
      Logger.error(e);
      throw e;
    }
  }

  async finalize(rid: string, status: string, message: string) {
    try {
      if (this.connection.readyState == 0) {
        Logger.error('Connection is not ready');
        return;
      }
      const oldDoc = await this.retrieve(rid);
      if (oldDoc) {
        return await this.collection.updateOne(
          { _id: new mongoose.Types.ObjectId(oldDoc._id) },
          {
            $set: {
              status: status.toUpperCase(),
              message: message,
              finalized: new Date().toISOString(),
            },
          },
        );
      } else {
        throw new Error('Request to update not found');
      }
    } catch (e) {
      Logger.error(e);
      throw e;
    }
  }
}
