import { ConfigService } from '@nestjs/config';
import { UserInfo } from './auth/usser-info';
import { ROLE_ADMIN, ROLE_OPERATOR } from './auth/auth-constants';
import { BadGatewayException, BadRequestException, ForbiddenException, InternalServerErrorException, NotFoundException } from '@nestjs/common';

const NIL_PID = '0000000000000000000000';
const HTTP_EXCEPTION_SEP = '§§';
const HTTP_EXCEPTION_SEP_LEN = HTTP_EXCEPTION_SEP.length;

export enum ErrorCondition {
    AUTH,
    INPUT,
    NOTFOUND,
    EXTERNAL,
    INTERNAL,
    UNKNOWN,
}

export function getCurrentDateTime(): string {
    return new Date().toISOString().slice(0, 19).replace('T', ' ');
}

export function isAuthDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_AUTH');
}

export function isRoleCheckDisabled(configService: ConfigService): boolean {
    return isTrue(configService, 'DISABLE_ROLES');
}

export function getISOTimeFromString(seconds: string): string {
    return getISOTime(safeParseInt(seconds));
}

export function getISOTime(seconds: number): string {
    return seconds > 0 ? new Date(seconds * 1000).toISOString() : ''; // normalize non-positive values to empty string
}

export function safeParseInt(num: string) {
    return parseInt(num) || 0; // normalize empty or non-numerical value, which return 'NaN', to 0
}

export function isPrivilegedUser(user: UserInfo): boolean {
    return (user && (user.role === ROLE_ADMIN || user.role === ROLE_OPERATOR));
}

export function isAdminUser(user: UserInfo): boolean {
    return (user && user.role === ROLE_ADMIN);
}

export function isTrue(conf: ConfigService, key: string): boolean {
    let val: string = conf.get<string>(key);
    return val?.trim() === 'true';
}

export function isDID(str: string | null | undefined): boolean {
    const didRegex = /^did:[a-z0-9]+:.+$/;
    return didRegex.test(str);
}

export function isShortUUID(str: string | null | undefined): boolean {
    // The regex checks for exactly 22 characters from the Flickr Base58 alphabet:
    // Digits 1-9, uppercase A-H, J-N, P-Z, and lowercase a-k, m-z.
    if (NIL_PID === str) {
        return true;
    }
    const shortUuidRegex = /^[1-9A-HJ-NP-Za-km-z]{22}$/;
    return shortUuidRegex.test(str);
}

export function throwExceptionWithErrorCondition(type: ErrorCondition, desc: string) {
    throw new Error(type + HTTP_EXCEPTION_SEP + desc);
}

export function translateToHttpException(err: Error) {
    const condition: ErrorCondition = getErrorCondition(err);
    const message: string = getErrorConditionDescription(err);
    switch (condition) {
        case ErrorCondition.AUTH:
            throw new ForbiddenException(message);
        case ErrorCondition.INPUT:
            throw new BadRequestException(message);
        case ErrorCondition.NOTFOUND:
            throw new NotFoundException(message);
        case ErrorCondition.INTERNAL:
            throw new InternalServerErrorException(message);
        case ErrorCondition.EXTERNAL:
            throw new BadGatewayException(message);
        default:
            throw new InternalServerErrorException(message);
    }
}

function getErrorCondition(err: Error): ErrorCondition {
    const msg = err.message;
    if (!msg || msg.trim().length == 0) {
        return ErrorCondition.UNKNOWN;
    }
    const idx = msg.indexOf(HTTP_EXCEPTION_SEP);
    if (idx >= 0) {
        const typeString = msg.substring(0, idx);
        const typeNumber = Number(typeString);
        // Check if the parsed number is a valid enum member.
        if (!isNaN(typeNumber) && ErrorCondition[typeNumber] !== undefined) {
            return typeNumber as ErrorCondition;
        }
    }
    return ErrorCondition.UNKNOWN;
}

function getErrorConditionDescription(err: Error): string {
    const msg = err.message;
    if (!msg || msg.trim().length == 0) {
        return msg;
    }
    const idx = msg.indexOf(HTTP_EXCEPTION_SEP);
    if (idx >= 0) {
        return msg.substring(idx + HTTP_EXCEPTION_SEP_LEN);
    } else {
        return msg;
    }
}
