import {
    Controller,
    Get, NotFoundException,
    Query,
} from '@nestjs/common';

import { PatService } from '../pats/services/pat.service';
import { PatOfferingRepository } from '../pats/repositories/pat.offering-repository';
import { Questionnaire } from '../pats/entities/questionnaire.dto';
import { ApiBearerAuth, ApiOperation, ApiResponse, ApiTags } from "@nestjs/swagger";
import { Price } from '../pats/entities/price';
import { PatAnswersRepository } from '../pats/repositories/pat.answers-repository';
import {SatService} from "../pats/services/sat.service";
import {firstValueFrom} from "rxjs";
import {HttpService} from "@nestjs/axios";

@ApiBearerAuth()
@ApiTags('Pricing Advisory Tool - PAT')
@Controller()
export class PatController {
    constructor(
        private readonly patService: PatService,
        private readonly patOfferingsRepository: PatOfferingRepository,
        private readonly patAnswersRepository: PatAnswersRepository,
        private readonly satService: SatService,
        private readonly httpService: HttpService
    ) {}

    @ApiOperation({summary: 'Questionnaire'})
    @ApiResponse({
        status: 200,
        description: 'The found record',
        type: Questionnaire
    })

    //{"id":"3191-3686-b27a-cfcb","asset":"b3zMEzsygyWkMpj65","duration":"1M"}
    //2pR4RXhexhX6SShUjwEMrLEewa7nzdB8GbR6K5J2VkN2MXerkszy98ns5466N6CPd51Xgowvt9XzCmkHjQv84uZgWmRryq5MTE4
    @Get('/pat/v1.0/pricing-params')
    @ApiOperation({ summary: 'PricingParams'})
    async getParams(@Query('base58') offering: string): Promise<Questionnaire> {
        console.log("pricing-params")
        const decodedObject = this.patService.convertBase58ToJson(offering)
        console.log("decodedObject", decodedObject)
        const request = await this.patService.convertToOfferingRequest(decodedObject);
        console.log("request", request)
        const success = await this.patOfferingsRepository.saveOffering(request);
        console.log("success", success)
        return await this.patService.getQuestionnaireByAssetType(request?.dcterms_type, request?.cid);
    }


    // For testing!
    //{"cid":"3191-3686-b27a-cfcb","answers":[{"id":1,"value":"1"},{"id":2,"value":4},{"id":3,"value":"1"},{"id":4,"value":"1"},{"id":5,"value":4},{"id":6,"value":"1"},{"id":7,"value":"1"},{"id":8,"value":5},{"id":9,"value":"1"},{"id":10,"value":"1"},{"id":11,"value":"1"},{"id":12,"value":10},{"id":13,"value":100},{"id":14,"value":5},{"id":15,"value":"1"},{"id":16,"value":"0"},{"id":17,"value":"50"}]}
    // 5AvPvssBBk7bN3AtaqjzP93p7jjF8zWidzJxiAcQfxreFeLgvQks7yj3jZWssnjvnrgccAPspPADryZZ7JXUFqqK2tJTU8vLciv3CHu8HyeEDtGqnuivNqvLrHSvPhmHqzs37NCtHRbs16vGZwjH51hKinicoEXsDPohybe5Ar9pdTZ7nYQ5qksuLpG1T1scgUzxBv8vA6K8ntkuLLgLqALAt3sNter8wFCXERUiamvUYXyUowExquKgvKeuZKRrJewbhJeceEBYdBr5GT1ABjWdRCAX5ASt1uEvL3EWz6zGN6HGCfLLwqdAsL6mmVWS534jb6Ltk9ocMjX8LsokVbUTGWrxqpc1DM4sXFTVGwUt7Rd3C6As9MfK7hmJCARRVzo8eRFess5QJXuqS5TWqE6NxKTzCuGTqeVDk4wNoV5LJG2awetnRz6CMvWNChP3v1Sf9ordCda6MEFhDcUanE6fg4eNts2ctmdoCWcUnxGGQHQ1UMhcX4shajxsG89za1SpBgLiVn5fdkmJMwyh2jMJxSQeeaf6
    @ApiOperation({summary: 'PricingAdvice'})
    @ApiResponse({
        status: 200,
        description: 'The found record',
        type: Price
    })
    @Get('/pat/v1.0/pricing-advice')
    async getPricing(@Query('base58') params: string): Promise<Price> {
        const decodedObject = this.patService.convertBase58ToJson(params)

        console.log(`getPricing(decodedObject=${decodedObject})`)

        if (decodedObject == null || decodedObject?.cid == null || decodedObject?.answers == null) {
            throw new NotFoundException('Decoded object not found');
        }

        await this.patAnswersRepository.saveAnswer(decodedObject);

        //1. step - Calculate price based on questionnaire and answers
        const response = await this.patService.calculateQuestionnaireBasedPrice(decodedObject, this.patOfferingsRepository)

        const questionnaireBasedPrice = response?.qbp
        const q7 = response?.q7

        //DYNAMIC PRICING PART
        //2. step - Find similar offerings using BERT
        const similarOfferings= await this.satService.findSimilarOfferingsForOfferingId(decodedObject?.cid)

        //3. step - Calculate QBS score for similarOfferings
        const similarOfferingsWithQBSScore = await this.satService.calculateQBSScore(similarOfferings, decodedObject, this.patService)
        console.log("similarOfferingsWithQBSScore", similarOfferingsWithQBSScore);

        //4. step - Calculate Eigen Vector Centrality (EVC)
        const evcMap = this.satService.calculateEigenvectorCentrality(similarOfferingsWithQBSScore);
        console.log("evcMap", evcMap);

        /*
        //5. step - Find transactional prices for connected assets
        const assetPrices = await this.satService.getAssetPricingData(connectedAssets);
        */

        /*
        //6. step - Dynamic price calculation
        let weightedPriceSum = 0;
        let weightSum = 0;

        for (const asset of connectedAssets) {
            const price = this.patService.normalizePrice(asset, decodedObject);
            const transactions = assetPrices.get(asset.id)?.transactions ?? 1;
            const evc = evcMap.get(asset.id) ?? 1;

            const weight = transactions * evc;
            weightedPriceSum += price * weight;
            weightSum += weight;
        }

        if (weightSum === 0) {
            throw new Error("No valid connected assets for dynamic pricing.");
        }

        const P2 = weightedPriceSum / weightSum;
        */

        const P2 = 1

        //7. step - Applying business model to the calculated price
        const P1 = await this.patService.applyBmToTheQuestionnaireBasedPrice(decodedObject, this.patOfferingsRepository, questionnaireBasedPrice, q7)

        //8. step - Final Price
        //const finalPrice = Math.sqrt(P1 * P2)
        const price = new Price();
        price.price = Number(P1.toFixed(2));
        await this.patOfferingsRepository.updateOfferingByCid(decodedObject?.cid, {advised_price: price.price})
        return price;
    }

    /*
    @ApiOperation({ summary: 'Get Transactions by Offering IDs' })
    @ApiResponse({
        status: 200,
        description: 'Returns transaction history for specified offering IDs.',
    })
    @Get('/pat/v1.0/transactions')
    async getTransactions(@Query('oid') offeringIds: string[]): Promise<any> {
        try {
            // Check if offeringIds is an array or a single string (if only one oid is passed)
            if (typeof offeringIds === 'string') {
                offeringIds = [offeringIds]; // Convert single value to an array
            }

            const transactions = await this.patService.getTransactions(offeringIds);
            return Array.from(transactions.entries());
        } catch (error) {
            throw new NotFoundException(`Error fetching transactions: ${error.message}`);
        }
    }

    @Get('/pat/v1.0/asset-info')
    @ApiOperation({ summary: 'AssetInfo'})
    async getAssetInfo(@Query('assetId') assetId: string): Promise<string> {
        console.log("asset-info")
        const assetInfo = await this.patService.getAssetCanonicalDescription(assetId);
        return assetInfo;
    }
     */
}
