import {MongoClient, MongoServerError} from 'mongodb';
import {HttpException, HttpStatus, Injectable} from '@nestjs/common';
import {Answer} from '../entities/answer.dto';
import {QuestionnaireCompiled} from '../entities/questionnaire-compiled.dto';

console.log("PAT ANSWERS REPOSITORY");

const env = process.env.APP_ENV;
const dbName = process.env.DB_NAME;

let url: string;

if (env === 'PROD') {
  url = `mongodb://root:${process.env.MONGO_INITDB_ROOT_PASSWORD}@${dbName}:27017`;
} else {
  url = process.env.MONGO_DEV_URL;
}

console.log("mongodb url", url);
const collectionName = 'answers';

@Injectable()
export class PatAnswersRepository {
  private client: MongoClient;

  constructor() {
    this.client = new MongoClient(url);
  }

  private async connectToDb() {
    try {
      await this.client.connect();
      console.log(`Connected successfully to MongoDB, URL: ${url}, Database: ${dbName}, Collection: ${collectionName}`);
      return this.client.db(dbName).collection(collectionName);
    } catch (err) {
      console.error('Failed to connect to MongoDB', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to connect to MongoDB' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    }
  }

  private async closeConnection() {
    await this.client.close();
  }

  // Method to remove all answers
  async deleteAllAnswers(): Promise<boolean> {
    const collection = await this.connectToDb();
    try {
      const result = await collection.deleteMany({});
      console.log(`${result.deletedCount} answers deleted successfully`);
      return true;
    } catch (err) {
      console.error('Failed to delete all answers', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to delete all answers' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  // Method to remove answers by offeringId
  async deleteAnswersByOfferingId(offeringId: string): Promise<boolean> {
    const collection = await this.connectToDb();
    try {
      const result = await collection.deleteMany({ offering_id: offeringId });
      console.log(`${result.deletedCount} answers for offeringId ${offeringId} deleted successfully`);
      return true;
    } catch (err) {
      console.error(`Failed to delete answers for offeringId ${offeringId}`, err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: `Failed to delete answers for offeringId ${offeringId}` },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  async find(query: any): Promise<QuestionnaireCompiled[]> {
    const collection = await this.connectToDb();
    try {
      const dbData = await collection.find(query).toArray();
      if (dbData.length > 0) {
        console.log('Documents found:', dbData);
        return dbData.map(this.convertAnswersDataToObj);
      } else {
        console.log('No documents found with query:', query);
        return [];
      }
    } catch (err) {
      console.error('Failed to retrieve answers', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to retrieve answers' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  async findAnswersByOfferingIds(offeringIds: string[]): Promise<QuestionnaireCompiled[]> {
    const collection = await this.connectToDb();
    try {
      const dbData = await collection.find({ _id: { $in: offeringIds as any } }).toArray();
      if (dbData.length > 0) {
        console.log('Documents found:', dbData);
        return dbData.map(this.convertAnswersDataToObj);
      } else {
        console.log('No documents found for given offeringIds:', offeringIds);
        return [];
      }
    } catch (err) {
      console.error('Failed to retrieve answers', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to retrieve answers' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }


  async findAnswersByOfferingId(offeringId: string): Promise<QuestionnaireCompiled> {
    const collection = await this.connectToDb();
    try {
      const dbData = await collection.findOne({ _id: offeringId as any });
      if (dbData) {
        console.log('Document found:', dbData);
        return this.convertAnswersDataToObj(dbData);
      } else {
        console.log('findAnswersByOfferingId - No document found with offeringId:', offeringId);
        throw new HttpException(
            { status: HttpStatus.NOT_FOUND, error: `findAnswersByOfferingId - No document found with offeringId: ${offeringId}` },
            HttpStatus.NOT_FOUND
        );
      }
    } catch (err) {
      console.error('Failed to retrieve answers', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to retrieve answers' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  async saveAnswer(answers: QuestionnaireCompiled): Promise<boolean> {
    const collection = await this.connectToDb();
    try {
      await collection.insertOne(this.convertAnswersObjToDb(answers));
      console.log('Answer inserted successfully');
      return true;
    } catch (err) {
      if (err instanceof MongoServerError && err.code === 11000) {
        throw new HttpException(
            { status: HttpStatus.CONFLICT, error: `Duplicate key error for offeringId ${answers.cid}` },
            HttpStatus.CONFLICT
        );
      } else {
        console.error('Failed to save answer', err);
        throw new HttpException(
            { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to save answer' },
            HttpStatus.INTERNAL_SERVER_ERROR
        );
      }
    } finally {
      await this.closeConnection();
    }
  }

  private convertAnswersDataToObj(dbData: any): QuestionnaireCompiled {
    const questionnaireCompiled = new QuestionnaireCompiled();
    questionnaireCompiled.cid = dbData._id;
    if (Array.isArray(dbData.answers)) {
      questionnaireCompiled.answers = dbData.answers.map((jsonAnswer: any) => {
        const answer = new Answer();
        answer.id = jsonAnswer.id;
        answer.value = jsonAnswer.value;
        return answer;
      });
    } else {
      questionnaireCompiled.answers = [];
    }
    return questionnaireCompiled;
  }

  private convertAnswersObjToDb(answers: QuestionnaireCompiled): any {
    return {
      _id: answers.cid,
      answers: answers.answers.map((answer: Answer) => ({
        id: answer.id,
        value: answer.value
      }))
    };
  }
}
