import { MongoClient, MongoServerError } from 'mongodb';
import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { OfferingMain } from '../entities/offering-main.dto';
import * as dotenv from 'dotenv';
import dayjs from 'dayjs';

dotenv.config();

console.log("PAT OFFERING REPOSITORY");
const env = process.env.APP_ENV;
const dbName = process.env.DB_NAME;

let url: string;

if (env === 'PROD') {
  url = `mongodb://root:${process.env.MONGO_INITDB_ROOT_PASSWORD}@${dbName}:27017`;
} else {
  url = process.env.MONGO_DEV_URL;
}

console.log("mongodb url", url);

const collectionName = 'offerings';


@Injectable()
export class PatOfferingRepository {

  private client: MongoClient;

  constructor() {
    this.client = new MongoClient(url);
  }

  private async connectToDb() {
    try {
      await this.client.connect();
      console.log(`Connected successfully to MongoDB, URL: ${url}, Database: ${dbName}, Collection: ${collectionName}`);
      return this.client.db(dbName).collection(collectionName);
    } catch (err) {
      console.error('Failed to connect to MongoDB', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to connect to MongoDB' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    }
  }

  private async closeConnection() {
    await this.client.close();
  }

  // Method to remove an offering by its offeringId
  async deleteOfferingById(offeringId: string): Promise<boolean> {
    const collection = await this.connectToDb();

    try {
      const result = await collection.deleteOne({ offering_id: offeringId });
      if (result.deletedCount === 1) {
        console.log(`Offering with offeringId ${offeringId} deleted successfully`);
        return true;
      } else {
        console.log(`No offering found with offeringId ${offeringId}`);
        return false;
      }
    } catch (err) {
      console.error('Failed to delete offering', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to delete offering' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  // Method to remove all offerings
  async deleteAllOfferings(): Promise<boolean> {
    const collection = await this.connectToDb();

    try {
      const result = await collection.deleteMany({});
      console.log(`${result.deletedCount} offerings deleted successfully`);
      return true;
    } catch (err) {
      console.error('Failed to delete all offerings', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to delete all offerings' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  // Method to find offerings by query
  async find(query: any): Promise<OfferingMain[]> {
    const collection = await this.connectToDb();

    try {
      const dbData = await collection.find(query).toArray();
      if (dbData.length > 0) {
        console.log('Documents found:', dbData);
        return dbData.map(this.convertOfferingDataToObj);
      } else {
        console.log('No documents found with query:', query);
        return [];
      }
    } catch (err) {
      console.error('Failed to retrieve offerings', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to retrieve offerings' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  // Method to find an offering by offeringId
  async findOfferingById(offeringId: string): Promise<OfferingMain> {
    const collection = await this.connectToDb();

    try {
      const dbData = await collection.findOne({ offering_id: offeringId });
      if (dbData) {
        console.log('Document found:', dbData);
        return this.convertOfferingDataToObj(dbData);
      } else {
        console.log('findOfferingById - No document found with offeringId:', offeringId);
        throw new HttpException(
            { status: HttpStatus.NOT_FOUND, error: `findOfferingById - No document found with offeringId: ${offeringId}` },
            HttpStatus.NOT_FOUND
        );
      }
    } catch (err) {
      if (err instanceof HttpException) {
        throw err;
      } else {
        console.error('Failed to retrieve offering', err);
        throw new HttpException(
            { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to retrieve offering' },
            HttpStatus.INTERNAL_SERVER_ERROR
        );
      }
    } finally {
      await this.closeConnection();
    }
  }

  // Method to find an offering by offeringId
  async findOfferingByCid(offeringId: string): Promise<OfferingMain> {
    const collection = await this.connectToDb();

    try {
      const dbData = await collection.findOne({ cid: offeringId });
      if (dbData) {
        console.log('Document found:', dbData);
        return this.convertOfferingDataToObj(dbData);
      } else {
        console.log('findOfferingByCid - No document found with offeringId:', offeringId);
        throw new HttpException(
            { status: HttpStatus.NOT_FOUND, error: `findOfferingByCid - No document found with offeringId: ${offeringId}` },
            HttpStatus.NOT_FOUND
        );
      }
    } catch (err) {
      if (err instanceof HttpException) {
        throw err;
      } else {
        console.error('Failed to retrieve offering', err);
        throw new HttpException(
            { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to retrieve offering' },
            HttpStatus.INTERNAL_SERVER_ERROR
        );
      }
    } finally {
      await this.closeConnection();
    }
  }

  // Method to save a new offering
  async saveOffering(offering: OfferingMain): Promise<boolean> {
    const collection = await this.connectToDb();

    try {
      await collection.insertOne(this.convertOfferingObjToDb(offering));
      console.log('Offering inserted successfully');
      return true;
    } catch (err) {
      if (err instanceof MongoServerError && err.code === 11000) {
        console.error('Duplicate key error:', err.message);
        throw new HttpException(
            { status: HttpStatus.CONFLICT, error: `Duplicate key error for offeringId ${offering.offering_id}` },
            HttpStatus.CONFLICT
        );
      } else {
        console.error('Failed to save offering', err);
        throw new HttpException(
            { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to save offering' },
            HttpStatus.INTERNAL_SERVER_ERROR
        );
      }
    } finally {
      await this.closeConnection();
    }
  }

  async updateOfferingByCid(offeringId: Number, updateData: Partial<OfferingMain>): Promise<void> {
    const collection = await this.connectToDb();
    try {
      const result = await collection.updateOne({ cid: offeringId }, { $set: updateData });

      if (result.matchedCount === 0) {
        throw new HttpException(
            { status: HttpStatus.NOT_FOUND, error: 'Offering not found' },
            HttpStatus.NOT_FOUND
        );
      }
    } catch (error) {
      console.error('Error updating offering:', error);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Error updating offering' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    }
  }


  // Method to find offerings by dcterms_type
  async findOfferingsByDctermsType(dctermsType: string): Promise<OfferingMain[]> {
    const collection = await this.connectToDb();

    try {
      const dbDataArray = await collection.find({ dcterms_type: dctermsType }).toArray();
      if (dbDataArray.length > 0) {
        console.log(`${dbDataArray.length} documents found with dcterms_type: ${dctermsType}`);
        return dbDataArray.map(dbData => this.convertOfferingDataToObj(dbData));
      } else {
        console.log(`No documents found with dcterms_type: ${dctermsType}`);
        return [];
      }
    } catch (err) {
      console.error('Failed to fetch offerings by dcterms_type', err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to fetch offerings by dcterms_type' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    } finally {
      await this.closeConnection();
    }
  }

  private convertOfferingDataToObj(dbData: any): OfferingMain {
    const offeringObj = new OfferingMain();
    offeringObj.offering_id = dbData.offering_id as any;
    offeringObj.cid = dbData.cid;
    offeringObj.dcterms_identifier = dbData.dcterms_identifier;
    offeringObj.dcterms_type = dbData.dcterms_type;
    offeringObj.dcterms_title = dbData.dcterms_title;
    offeringObj.dcterms_description = dbData.dcterms_description;
    offeringObj.offering = dbData.offering;
    offeringObj.advised_price = dbData.advised_price;
    offeringObj.created = dbData.created;

    return offeringObj;
  }

  private convertOfferingObjToDb(offering: OfferingMain): any {
    return {
      offering_id: offering.offering_id,
      cid: offering.cid,
      dcterms_identifier: offering.dcterms_identifier,
      dcterms_type: offering.dcterms_type,
      dcterms_title: offering.dcterms_title,
      dcterms_description: offering.dcterms_description,
      offering: offering.offering,
      advised_price: offering.advised_price,
      created: dayjs().format('YYYY-MM-DD HH:mm:ss')
    };
  }
}
