import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { PatOfferingRepository } from '../repositories/pat.offering-repository';
import { PatService } from './pat.service';
import { PatAnswersRepository } from "../repositories/pat.answers-repository";
import { OfferingMain } from "../entities/offering-main.dto";
import { QuestionnaireCompiled } from "../entities/questionnaire-compiled.dto";

/**
 * @since June 2024
 */

@Injectable()
export class MongodbService {

  async offeringById(id: string, patOfferingsRepository: PatOfferingRepository, patService: PatService) {
    console.log(`offeringById: Searching for offering with ID: ${id}`);
    const jsonData = await patOfferingsRepository.findOfferingById(id);
    console.log(`offeringById: Found offering for ID: ${id}`);
    return patService.convertJsonToBase58(jsonData);
  }

  async answerByOfferingId(id: string, patAnswersRepository: PatAnswersRepository, patService: PatService) {
    console.log(`answerByOfferingId: Searching for answers with offeringId: ${id}`);
    const jsonData = await patAnswersRepository.findAnswersByOfferingId(id);
    console.log(`answerByOfferingId: Found answers for offeringId: ${id}`);
    return patService.convertJsonToBase58(jsonData);
  }

  async offeringsByDctermsType(dctermsType: string, patOfferingsRepository: PatOfferingRepository, patService: PatService) {
    console.log(`offeringsByDctermsType: Searching for offerings with dctermsType: ${dctermsType}`);
    const offerings = await patOfferingsRepository.findOfferingsByDctermsType(dctermsType);
    console.log(`offeringsByDctermsType: Found ${offerings.length} offerings with dctermsType: ${dctermsType}`);
    return offerings.map(offering => patService.convertJsonToBase58(offering));
  }

  async getAllOfferings(patOfferingsRepository: PatOfferingRepository): Promise<OfferingMain[]> {
    console.log('getAllOfferings: Retrieving all offerings');
    const offerings = await patOfferingsRepository.find({});
    console.log(`getAllOfferings: Found ${offerings.length} offerings`);
    return offerings;
  }

  async getAllAnswers(patAnswersRepository: PatAnswersRepository): Promise<QuestionnaireCompiled[]> {
    console.log('getAllAnswers: Retrieving all answers');
    const answers = await patAnswersRepository.find({});
    console.log(`getAllAnswers: Found ${answers.length} answers`);
    return answers;
  }

  // Method to remove all offerings
  async deleteAllOfferings(patOfferingsRepository: PatOfferingRepository) {
    console.log('deleteAllOfferings: Deleting all offerings');
    try {
      await patOfferingsRepository.deleteAllOfferings();
      console.log("deleteAllOfferings: All offerings removed successfully");
    } catch (err) {
      console.error("deleteAllOfferings: Failed to delete all offerings:", err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to delete all offerings' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    }
  }

  // Method to remove all answers
  async deleteAllAnswers(patAnswersRepository: PatAnswersRepository) {
    console.log('deleteAllAnswers: Deleting all answers');
    try {
      await patAnswersRepository.deleteAllAnswers();
      console.log("deleteAllAnswers: All answers removed successfully");
    } catch (err) {
      console.error("deleteAllAnswers: Failed to delete all answers:", err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to delete all answers' },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    }
  }

  // Method to delete answers by offeringId
  async deleteAnswersByOfferingId(offeringId: string, patAnswersRepository: PatAnswersRepository) {
    console.log(`deleteAnswersByOfferingId: Deleting answers for offeringId: ${offeringId}`);
    try {
      await patAnswersRepository.deleteAnswersByOfferingId(offeringId);
      console.log(`deleteAnswersByOfferingId: Answers for offeringId ${offeringId} removed successfully`);
    } catch (err) {
      console.error(`deleteAnswersByOfferingId: Failed to delete answers for offeringId ${offeringId}:`, err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: `Failed to delete answers for offeringId ${offeringId}` },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    }
  }

  // Method to delete an offering by offeringId
  async deleteOfferingById(offeringId: string, patOfferingsRepository: PatOfferingRepository) {
    console.log(`deleteOfferingById: Deleting offering with offeringId: ${offeringId}`);
    try {
      await patOfferingsRepository.deleteOfferingById(offeringId);
      console.log(`deleteOfferingById: Offering with offeringId ${offeringId} removed successfully`);
    } catch (err) {
      console.error(`deleteOfferingById: Failed to delete offering with offeringId ${offeringId}:`, err);
      throw new HttpException(
          { status: HttpStatus.INTERNAL_SERVER_ERROR, error: `Failed to delete offering with offeringId ${offeringId}` },
          HttpStatus.INTERNAL_SERVER_ERROR
      );
    }
  }
}
