import { Injectable, Logger, NotFoundException } from '@nestjs/common';
import baseX from 'base-x';
import { readFile } from 'fs/promises';
import { Questionnaire } from '../entities/questionnaire.dto';
import { Option } from '../entities/option.dto';
import { Question } from '../entities/question.dto';
import { Limit } from '../entities/limit.dto';
import { QuestionnaireCompiled } from '../entities/questionnaire-compiled.dto';
import { Answer } from '../entities/answer.dto';
import { PatOfferingRepository } from '../repositories/pat.offering-repository';
import {DcTermsType, OfferingMain} from "../entities/offering-main.dto";
import {firstValueFrom} from "rxjs";
import { HttpService } from '@nestjs/axios';
import * as dotenv from "dotenv";
import axios from "axios";
import {Offering} from "../entities/offering.dto";

/**
 * @author Adam Popernik, EUBA
 * @since November 2023
 */

@Injectable()
export class PatService {

  constructor(private readonly httpService: HttpService) {}

  static BASE58_ALPHABET = '123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz'
  static baseXCodec = baseX(PatService.BASE58_ALPHABET)
  convertBase58ToJson(base58) {
    // Decode the Base58 string back to a JSON string
    const charCodeArray = Array.from(PatService.baseXCodec.decode(base58));
    const decodedJSONString = charCodeArray.map(x => String.fromCharCode(x)).join('');

    try {
      return JSON.parse(decodedJSONString);
    } catch (error) {
      console.error('Error decoding Base58 data:', error);
    }

  }

  convertJsonToBase58(jsonData): String {
    const jsonStr = JSON.stringify(jsonData);
    return PatService.baseXCodec.encode(Buffer.from(jsonStr, 'utf8'));
  }

  async loadJsonFromFile(path) {
    try {
      const jsonData = await readFile(path, 'utf8');
      return JSON.parse(jsonData);
    } catch (error) {
      const errorMessage = 'Error reading ' + path + ':' + error
      throw new NotFoundException(errorMessage);
    }
  }

  async getQuestionnaireByAssetType(assetType, cid) {
    console.log(`getQuestionnaireByAssetType(assetType=${assetType}, cid=${cid})`)
    let jsonData = null

    const mappedType = String(assetType) === "Dataset" ? DcTermsType.DATASET : DcTermsType.DIGITAL_PRODUCT

    if (!mappedType) {
      throw new NotFoundException(`Asset type ${assetType} not found`)
    }

    switch (mappedType) {
      case DcTermsType.DATASET: {
        jsonData = await this.loadJsonFromFile('src/pats/data/response/DATASET-QUESTIONS.json')
        break;
      }
      case DcTermsType.DIGITAL_PRODUCT: {
        jsonData = await this.loadJsonFromFile('src/pats/data/response/DIGITAL-PRODUCT-QUESTIONS.json')
        break;
      }
      default: {
        throw new NotFoundException(`Asset type ${assetType} not found`);
      }
    }

    return this.convertJsonToQuestionnaireObject(jsonData, cid)
  }

  async calculateQuestionnaireBasedPrice(decodedObject: any, patRepository: PatOfferingRepository): Promise<{ qbp: number; q7: number }> {
    // CID received from JSON request
    const cid = decodedObject?.cid;
    const offering = await patRepository.findOfferingByCid(cid);
    const assetType = offering.dcterms_type;

    // Load questionnaire by CID and convert them into the questionnaire object
    const questionnaire = await this.getQuestionnaireByAssetType(assetType, decodedObject?.cid);
    console.log(`questionnaire=${JSON.stringify(questionnaire, null, 2)}`);

    // Load the answers/questionnaireCompiled from JSON and convert them into the object
    const questionnaireCompiled = this.convertJsonQuestionnaireCompiledObject(decodedObject);

    let answers = new Map<string, { weight: number | null; value: number }>();

    for (const answer of questionnaireCompiled.answers) {
      const question = questionnaire.findQuestionById(answer.id);
      if (!question) {
        console.warn(`Question with ID ${answer.id} not found in questionnaire.`);
        continue;
      }

      let weight: number | null;
      try {
        weight = eval(question.weight);
        if (isNaN(weight)) throw new Error('Invalid weight calculation');
      } catch (error) {
        weight = null;
      }

      console.log(`Question: ${question.alias}, Value: ${answer.value}, Weight: ${weight}`);

      answers.set(question.alias, { weight, value: answer.value });
    }

    const getValue = (key: string) => {
      const value = answers.get(key)?.value ?? 0;
      const weight = answers.get(key)?.weight ?? 0;
      const result = value * weight;
      console.log(`getValue(${key}): value=${value}, weight=${weight}, result=${result}`);
      return result;
    };

    const calculateGroup = (keys: string[]) => {
      const sum = keys.reduce((total, key) => total + getValue(key), 0);
      const weight = 1 / 5
      const result =  weight * sum;
      console.log(`calculateGroup(${JSON.stringify(keys)}): sum=${sum} * weight=${weight} => result=${result}`);
      return result;
    };

    // Mapping of asset types to relevant question groups
    const groupsConfig: Record<string, string[][]> = {
      [DcTermsType.DATASET]: [
        ['Q1.1', 'Q1.2', 'Q1.3'], // group1
        ['Q2.1', 'Q2.3', 'Q2.4', 'Q2.5', 'Q2.6', 'Q2.7'], // group2
        ['Q3.1', 'Q3.2'], // group3
        ['Q5.1', 'Q5.2'], // group5
        ['Q6.1'], // group6
        ['Q7.1'] // group7
      ],
      [DcTermsType.DIGITAL_PRODUCT]: [
        ['Q1.1', 'Q1.3', 'Q1.4'], // group1
        ['Q2.3', 'Q2.6', 'Q2.7'], // group2
        ['Q3.2'], // group3
        ['Q5.1', 'Q5.2'], // group5
        ['Q6.1'], // group6
        ['Q7.1'] // group7
      ]
    };

    // Get the mapped type from the dictionary
    const mappedType = String(assetType) === "Dataset" ? DcTermsType.DATASET : DcTermsType.DIGITAL_PRODUCT;
    const groups = groupsConfig[mappedType];

    // Validate the asset type
    if (!groups) {
      throw new NotFoundException(`Asset type ${assetType} not found`);
    }

    // Calculate values for each group
    const [group1, group2, group3, group5, group6] = groups.map(calculateGroup);

    console.log(`Group values: group1=${group1}, group2=${group2}, group3=${group3}, group5=${group5}, group6=${group6}`);

    //calculateGroup(["Q1.1","Q1.3","Q1.4"]): sum=0.9166666666666666 * weight=0.2 => result=0.18333333333333335
    //calculateGroup(["Q2.3","Q2.6","Q2.7"]): sum=0.5333333333333333 * weight=0.2 => result=0.10666666666666667
    // Prevent division by zero

    const mandayrate = 1000
    const q4_1 = answers.get('Q4.1')?.value * mandayrate ?? 0;
    const q4_2 = answers.get('Q4.2')?.value || 1;
    const unitPrice = q4_1 / q4_2;
    console.log(`Unit price calculation: Q4.1=${q4_1}, Q4.2=${q4_2}, unitPrice=${unitPrice}`);

    //console.log(`Price multiplier: isUniqueData=${isUniqueData}, priceMultiplier=${priceMultiplier}`);

    const qbp = unitPrice * (1 + 0.5 * (group1 + group2 + group3 + group5 + group6)) ;
    console.log(`Final price: ${qbp}`);

    const q7 = answers.get('Q7.1')?.value
    return { qbp, q7 };
  }


  private convertJsonQuestionnaireCompiledObject(jsonData: any): QuestionnaireCompiled {
    const questionnaireCompiled = new QuestionnaireCompiled();
    questionnaireCompiled.cid = jsonData.cid;
    if (Array.isArray(jsonData.answers)) {
      questionnaireCompiled.answers = jsonData.answers.map((jsonAnswer: any) => {
        const answer = new Answer();
        answer.id = jsonAnswer.id;
        answer.value = jsonAnswer.value;
        return answer;
      });
    } else {
      questionnaireCompiled.answers = [];
    }

    return questionnaireCompiled;
  }

  private convertJsonToQuestionnaireObject(jsonData: any, cid: any): Questionnaire {

    const questionnaire = new Questionnaire();
    questionnaire.cid = cid;

    if (Array.isArray(jsonData.questions)) {
      questionnaire.questions = jsonData.questions.map((jsonQuestion: any) => {
        const question = new Question();
        question.alias = jsonQuestion.alias;
        question.id = jsonQuestion.id;
        question.required = jsonQuestion.required;
        question.type = jsonQuestion.type;
        //TODO: Use enum for group
        question.group = jsonQuestion.group;
        question.weight = jsonQuestion.weight;
        question.label = jsonQuestion.label;

        if (jsonQuestion.options) {
          question.options = jsonQuestion.options.map((jsonOption: any) => {
            const option = new Option();
            option.label = jsonOption.label;
            option.value = jsonOption.value;
            return option;
          });
        }

        if (jsonQuestion.limits) {
          const limit = new Limit();
          limit.numberFormat = jsonQuestion.limits.numberFormat;
          limit.min = jsonQuestion.limits.min;
          limit.max = jsonQuestion.limits.max;
          question.limits = limit;
        }

        return question;
      });
    }

    Logger.log('questionnaire: ' + JSON.stringify(questionnaire, null, 2));

    return questionnaire;
  }

  private convertSubscriptionToDays(subscription: string): number {
    const unitMultipliers = { H: 1 / 24, D: 1, W: 7, M: 30, Y: 365 };

    const match = subscription.match(/(\d+)([HDWMY])/);
    if (!match) throw new Error('Invalid subscription format');

    const [_, num, unit] = match;
    return Number(num) * (unitMultipliers[unit] ?? 0);
  }

  private convertVolumeToGB(volume: string): number {
    const unitMultipliers = {
      B: 1 / (1024 ** 3),
      KB: 1 / (1024 ** 2),
      MB: 1 / 1024,
      GB: 1,
      TB: 1024,
    };

    const match = volume.match(/^(\d+(?:\.\d+)?)(B|KB|MB|GB|TB)$/i);
    if (!match) {
      throw new Error(`Invalid volume format: ${volume}`);
    }

    const [, numStr, unitRaw] = match;
    const unit = unitRaw.toUpperCase();

    const multiplier = unitMultipliers[unit];
    if (multiplier === undefined) {
      throw new Error(`Unknown unit: ${unit}`);
    }

    return Number(numStr) * multiplier;
  }

  async applyBmToTheQuestionnaireBasedPrice(decodedObject: any, patRepository: PatOfferingRepository, questionnaireBasedPrice: number, vx: number) {
    const offering = (await patRepository.findOfferingByCid(decodedObject?.cid))?.offering;
    console.log("applyBmToTheQuestionnaireBasedPrice - answers", decodedObject?.answers)
    const { subscription, downloads, volume } = offering || {};

    if (subscription) {
      const k = this.convertSubscriptionToDays(subscription);
      console.log("convertSubscriptionToDays - K = ", k)
      console.log("questionnaireBasedPrice", questionnaireBasedPrice)
      console.log("Math.log(1 + k)", Math.log(1 + k))
      return questionnaireBasedPrice * (k / (365 * Math.log(1 + k)));
    }

    if (downloads) {
      return questionnaireBasedPrice * (Number(downloads) / 250);
    }

    if (volume) {
      const gb = this.convertVolumeToGB(volume)
      return questionnaireBasedPrice * ((Number(gb) / vx) / 250);
    }

    return questionnaireBasedPrice;

  }


  async getTransactions(offeringIds: string[]): Promise<Map<string, any[]>> {
    if (!Array.isArray(offeringIds)) {
      throw new NotFoundException('The offeringIds must be an array.');
    }

    if (offeringIds.length === 0) {
      throw new NotFoundException('At least one offeringId (oid) must be provided.');
    }

    // Construct API URL with query parameters
    const apiUrl = `https://tm.fame-horizon.eu/api/v1.0/gov/v1.0/trading-history?${offeringIds
        .map((oid) => `oid=${oid}`)
        .join('&')}`;

    try {
      console.log("API URL:", apiUrl);  // Log the final API URL
      // Fetch transaction history
      const response = await firstValueFrom(
          this.httpService.get<Record<string, [number, string, number][]>>(apiUrl),
      );

      // Parse and format the response
      return this.parseTransactions(response.data);
    } catch (error) {
      throw new NotFoundException(`Failed to fetch transactions: ${error.message}`);
    }
  }


  private parseTransactions(response: Record<string, [number, string, number][]>): Map<string, any[]> {
    const offeringMap = new Map<string, any[]>();
    for (const [offeringId, entries] of Object.entries(response)) {
      if (Array.isArray(entries) && entries.length > 0) {
        // Map all transactions for the current offeringId
        const transactions = entries.map(([price, transactionId, timestamp]) => ({
          price,
          transactionId,
          timestamp,
          dateTime: new Date(timestamp * 1000).toISOString(), // Convert timestamp to ISO DateTime
        }));

        offeringMap.set(offeringId, transactions);
      }
    }

    return offeringMap;
  }

  public async getAssetCanonicalDescription(aid: string): Promise<string> {
    const config = {
      headers: {
        Authorization: `Bearer ${(process.env.FDAC_PUB_TOKEN)}`
      }
    };
    try {
      const { data } = await axios.get(`${process.env.FDAC_URL}/interoperability/canonical/components/${aid}`, config);
      return data;
    } catch (err) {
      if (err.response && err.response.status === 404) {
        return null;
      } else {
        throw err;
      }
    }
  }

  public async convertToOfferingRequest(decodedObject: any): Promise<OfferingMain> {
    const offeringMain: OfferingMain = new OfferingMain();
    offeringMain.cid = decodedObject?.id;
    offeringMain.dcterms_identifier = decodedObject?.asset;

    const offering: Offering = new Offering();
    offering.volume = decodedObject?.volume;
    offering.downloads = decodedObject?.retrievals;
    offering.subscription = decodedObject?.duration;

    offeringMain.offering = offering;

    const assetInfo = await this.getAssetCanonicalDescription(offeringMain.dcterms_identifier) as unknown as Record<string, any>;
    console.log("assetInfo", assetInfo);

    offeringMain.dcterms_description = assetInfo.longDescription ?? assetInfo.description;
    offeringMain.dcterms_type = assetInfo.type;
    offeringMain.dcterms_title = assetInfo.name;

    console.log("offeringMain", offeringMain);

    return offeringMain;
  }

}
