import {Injectable} from '@nestjs/common';
import {PatOfferingRepository} from 'src/pats/repositories/pat.offering-repository';
import axios from 'axios';
import {PatAnswersRepository} from "../repositories/pat.answers-repository";
import Graph, {DirectedGraph} from 'graphology';
import * as math from 'mathjs';

@Injectable()
export class SatService {
    private readonly HF_API_URL = process.env.HF_API_URL;
    private readonly HF_API_TOKEN = process.env.HF_API_TOKEN;

    constructor(
        private readonly patOfferingsRepository: PatOfferingRepository,
        private readonly patAnswersRepository: PatAnswersRepository,
    ) {}

    /*
     async buildSimilarityNetwork(similarAssets: { id: string; qbsScore: number }[]): Map<string, { id: string; similarity: number }[]> {
        const network = new Map<string, { id: string; similarity: number }[]>();

        for (let i = 0; i < similarAssets.length; i++) {
            for (let j = i + 1; j < similarAssets.length; j++) {
                const assetA = similarAssets[i];
                const assetB = similarAssets[j];

                if (assetA.qbsScore > 0.5 && assetB.qbsScore > 0.5) {
                    // Pridať obojsmernú hranu
                    network.set(assetA.id, [...(network.get(assetA.id) || []), { id: assetB.id, similarity: assetA.qbsScore }]);
                    network.set(assetB.id, [...(network.get(assetB.id) || []), { id: assetA.id, similarity: assetB.qbsScore }]);
                }
            }
        }

        return network;
    }
     */

    async findSimilarOfferingsForOfferingId(offeringId: string, threshold: number = 0.2): Promise<any[]> {
        const newOffering = await this.patOfferingsRepository.findOfferingByCid(offeringId);

        if (!newOffering) {
            throw new Error(`Offering with ID ${offeringId} not found.`);
        }

        const { dcterms_type, dcterms_title, dcterms_description } = newOffering;

        // 1. Fetch all offerings from MongoDB that match the same dcterms_type
        let offeringsFromDb = await this.patOfferingsRepository.findOfferingsByDctermsType(dcterms_type);

        if (!offeringsFromDb || offeringsFromDb.length === 0) {
            console.log(`No offerings found for dcterms_type: ${dcterms_type}`);
            return [];
        }

        // 2. Remove the offering itself from the list
        offeringsFromDb = offeringsFromDb.filter(offering => offering.offering_id !== offeringId);

        if (offeringsFromDb.length === 0) {
            console.log(`No other offerings found after removing offering with ID ${offeringId}`);
            return [];
        }

        // 3. Prepare sentences (title + description) for similarity comparison
        const sentences = offeringsFromDb.map(offering => `${offering.dcterms_title} ${offering.dcterms_description}`);
        const sourceSentence = `${dcterms_title} ${dcterms_description}`;

        // 4. Get similarity scores
        const similarityScores = await this.getSimilarityScores({ source_sentence: sourceSentence, sentences });

        if (!similarityScores || similarityScores.length === 0) {
            console.log(`No similarity scores returned for offering ID ${offeringId}`);
            return [];
        }

        const decimalPlaces = 3;

        // 5. Log each offering with its similarity score
        const similarOfferings = offeringsFromDb
            .map((offering, index) => {
                const rawScore = similarityScores[index];
                const factor = Math.pow(10, decimalPlaces);
                const roundedScore = Math.round(rawScore * factor) / factor; // Dynamické zaokrúhlenie
                console.log(`Offering ${offering.dcterms_title} (${offering.offering_id}) has similarity score: ${roundedScore}`);
                return { offering, similarityScore: roundedScore };
            })
            .filter(item => item.similarityScore >= threshold) // Filter až po zaokrúhlení
            .sort((a, b) => b.similarityScore - a.similarityScore); // Sort by similarity score

        console.log("similarOfferings", similarOfferings)
        // 6. Return only offering_ids
        return similarOfferings.map(item => item.offering.offering_id);
    }


    private async getSimilarityScores(data: any): Promise<number[]> {
        const requestConfig = {
            method: 'post',
            url: this.HF_API_URL,
            headers: {
                'Authorization': `Bearer ${this.HF_API_TOKEN}`,
                'Content-Type': 'application/json',
            },
            data: { inputs: data },
        };

        console.log("Sending request to Hugging Face API:", JSON.stringify(requestConfig, null, 2));

        try {
            const response = await axios(requestConfig);
            return response.data;
        } catch (error) {
            console.error('Error fetching similarity scores from Hugging Face:', error);
            return [];
        }
    }

    async calculateQBSScore(similarOfferings, decodedObject, patService) {
        const cid = decodedObject?.cid;
        const offering = await this.patOfferingsRepository.findOfferingByCid(cid);
        const assetType = offering.dcterms_type;

        const questionnaire = await patService.getQuestionnaireByAssetType(assetType, decodedObject?.cid);
        console.log(`questionnaire=${JSON.stringify(questionnaire, null, 2)}`);

        // Safe eval function for weight calculation
        function evaluateWeight(weight: string): number | null {
            try {
                const evaluatedWeight = eval(weight); // Using eval to evaluate the expression
                if (isNaN(evaluatedWeight)) throw new Error('Invalid weight calculation');
                return evaluatedWeight;
            } catch (error) {
                console.error('Error evaluating weight:', error.message);
                return null; // Return null if there's an error
            }
        }

        // Format user answers into a map structure: alias: {id, type, value, weight}
        const userAnswersFormatted = decodedObject?.answers.reduce((map, answer) => {
            const question = questionnaire.questions.find(q => q.id === answer.id);
            if (question) {
                map[question?.alias] = {
                    id: answer.id,
                    type: question?.type,
                    value: answer.value,
                    weight: evaluateWeight(question?.weight)
                };
            }
            return map;
        }, {});

        // Get database answers and format them into the same structure
        const dbAnswers = await this.patAnswersRepository.findAnswersByOfferingIds(similarOfferings);
        const dbAnswersFormatted = dbAnswers.map(entry => ({
            cid: entry.cid,
            answers: entry.answers.reduce((map, answer) => {
                const question = questionnaire.questions.find(q => q.id === answer.id);
                if (question) {
                    map[question?.alias] = {
                        id: answer.id,
                        type: question?.type,
                        value: answer.value,
                        weight: evaluateWeight(question?.weight)
                    };
                }
                return map;
            }, {})
        }));

        console.log("userAnswersFormatted", userAnswersFormatted);
        console.log("dbAnswersFormatted:", JSON.stringify(dbAnswersFormatted, null, 2));

        const allAnswersFormatted = [userAnswersFormatted, ...dbAnswersFormatted.map(dbAnswer => dbAnswer.answers)];

        let similarityResults = [];

        for (let i = 0; i < allAnswersFormatted.length; i++) {
            for (let j = i + 1; j < allAnswersFormatted.length; j++) {
                const answers1 = allAnswersFormatted[i];
                const answers2 = allAnswersFormatted[j];

                const cid1 = i === 0 ? cid : dbAnswersFormatted[i - 1].cid;
                const cid2 = j === 0 ? cid : dbAnswersFormatted[j - 1].cid;

                const logicalKeys = ['Q1.1', 'Q1.3', 'Q2.7', 'Q3.2', 'Q5.2', 'Q6.1'];
                const logicalSum = logicalKeys.reduce((sum, key) =>
                    sum + (answers1[key]?.value === answers2[key]?.value ? answers1[key]?.weight : 0), 0);
                const logicalTotalWeight = logicalKeys.reduce((sum, key) => sum + (answers1[key]?.weight || 0), 0);
                const logicalSimilarity = logicalSum / logicalTotalWeight;

                const continuousKeys = ['Q4.1', 'Q4.2'];
                const continuousValues = continuousKeys.map(key => ({
                    user: answers1[key]?.weight * Math.log(answers1[key]?.value),
                    answer: answers2[key]?.weight * Math.log(answers2[key]?.value)
                }));
                const continuousSimilarity = (continuousValues[0].user * continuousValues[0].answer + continuousValues[1].user * continuousValues[1].answer) /
                    (Math.sqrt(continuousValues[0].user ** 2 + continuousValues[1].user ** 2) *
                        Math.sqrt(continuousValues[0].answer ** 2 + continuousValues[1].answer ** 2));

                const ordinalKeys = ['Q1.4', 'Q2.3', 'Q2.6', 'Q5.1'];
                const ordinalValues = ordinalKeys.map(key => ({
                    user: answers1[key]?.weight * answers1[key]?.value,
                    answer: answers2[key]?.weight * answers2[key]?.value
                }));
                const ordinalSimilarity = ordinalValues.reduce((sum, { user, answer }) => sum + user * answer, 0) /
                    (Math.sqrt(ordinalValues.reduce((sum, { user }) => sum + user ** 2, 0)) *
                        Math.sqrt(ordinalValues.reduce((sum, { answer }) => sum + answer ** 2, 0)));

                const similarity = (logicalSimilarity + continuousSimilarity + ordinalSimilarity) / 3;

                similarityResults.push({
                    offering1: cid1,
                    offering2: cid2,
                    similarity
                });
            }
        }

        return similarityResults;
    }

    async calculateEigenvectorCentrality(similarOfferings: any[]): Promise<any> {
        const graph = new DirectedGraph();

        // Pridanie uzlov a hrán do grafu, ak similarity > 0.5
        similarOfferings.forEach(({ offering1, offering2, similarity }) => {
            if (similarity > 0.5) {
                if (!graph.hasNode(offering1)) graph.addNode(offering1);
                if (!graph.hasNode(offering2)) graph.addNode(offering2);
                graph.addEdge(offering1, offering2, { weight: similarity });
            }
        });

        console.log('Nodes:', graph.nodes());
        console.log('Edges:', graph.edges());

        // Výpočet Eigenvector Centrality
        const centralityScores = this.computeEigenvectorCentrality(graph);

        console.log('Centrality Scores:', centralityScores);

        return {
            networkSize: graph.order,
            edges: graph.size,
            centralityScores,
        };
    }

    private computeEigenvectorCentrality(graph: DirectedGraph, maxIter = 100, tolerance = 1e-6): Record<string, number> {
        const nodes = graph.nodes();
        const n = nodes.length;

        let centrality = Object.fromEntries(nodes.map(node => [node, Math.random()]));
        let diff = 1;
        let iter = 0;
        const epsilon = 1e-9;

        while (diff > tolerance && iter < maxIter) {
            const newCentrality = Object.fromEntries(nodes.map(node => [node, 0]));

            nodes.forEach(node => {
                graph.forEachOutboundNeighbor(node, (neighbor, attr) => {
                    const weight = attr?.weight ?? 1;
                    newCentrality[neighbor] += centrality[node] * weight;
                });

                graph.forEachInboundNeighbor(node, (neighbor, attr) => {
                    const weight = attr?.weight ?? 1;
                    newCentrality[neighbor] += centrality[node] * weight;
                });
            });

            const sum = Object.values(newCentrality).reduce((s, v) => s + v, 0);
            if (sum === 0) {
                console.warn("Eigenvector centrality collapsed to zero.");
                return centrality;
            }

            const norm = Math.sqrt(Object.values(newCentrality).reduce((sum, val) => sum + val ** 2, 0)) + epsilon;
            Object.keys(newCentrality).forEach(node => (newCentrality[node] /= norm));

            diff = Math.max(...nodes.map(node => Math.abs(newCentrality[node] - centrality[node])));
            centrality = newCentrality;
            iter++;
        }

        return centrality;
    }

}
