# FAME Provenance & Tracing Smart Contracts

This repository contains the **Solidity smart contracts** that provide blockchain-based data provenance and asset tracing capabilities for the **FAME Marketplace platform**. These contracts are deployed on a private Ethereum-compatible blockchain and are used by the Provenance & Tracing (PT) API to store immutable records of data sources and assets.

## Overview

The smart contracts enable:

- **Trusted Source Registration**: Immutable registration of data sources with descriptors and timestamps
- **Asset Tracing**: Blockchain-based tracking of data assets with cryptographic hashes
- **Asset Lineage**: Parent-child relationships between assets to track data transformations
- **Auditability**: Complete on-chain history of source and asset lifecycle events
- **Data Migration**: Administrative functions for migrating historical data to the blockchain

These contracts work in conjunction with the PT API, which handles off-chain logic, authentication, and provides RESTful interfaces for the FAME platform.

## Smart Contracts

### Provenance.sol

Manages the registration and lifecycle of **Trusted Sources** - entities that provide data to the FAME Marketplace.

**Key Functions:**
- `registerSource(pid, descriptor)` - Registers a new trusted source with a unique PID and JSON descriptor
- `deactivateSource(pid)` - Marks a source as inactive (soft delete)
- `getSource(pid)` - Retrieves source information by PID
- `getSources()` - Returns all registered sources

**Events:**
- `SourceEvent(pid, action)` - Emitted when sources are registered or deactivated

### Tracing.sol

Manages **Asset Records** - immutable records of data assets with cryptographic hashes and lineage tracking.

**Key Functions:**
- `postAsset(aid, hash, predecessor)` - Registers a new asset with optional predecessor linkage
- `createEntry(...)` - Administrative function for migrating historical asset records (owner only)
- `assets(aid)` - Public mapping to query asset records

**Events:**
- `AssetEvent(aid, action)` - Emitted when assets are registered or migrated

**Asset Lineage:**
Assets can form chains through predecessor/successor relationships, enabling tracking of data transformations and derivations.

## Prerequisites

- **Node.js** 14.x or higher
- **npm** or **yarn**
- **Truffle** - Ethereum development framework
- Access to an Ethereum-compatible blockchain (Besu, Ganache, etc.)
- **Private keys** for deployment accounts

## Installation

```bash
# Install dependencies
npm install
```

This installs:
- Truffle framework for compilation and deployment
- OpenZeppelin contracts (dependencies)
- HDWallet provider for blockchain connectivity
- dotenv for environment variable management

## Configuration

### Environment Variables

Create a `.env` file in the project root with the following variables:

```bash
# Blockchain RPC endpoint
PRODUCTION_ENDPOINT=http://<address:port of the Blockchain RPC endpoint>

# Blockchain network ID
PRODUCTION_ID=6666

# Private keys for deployment accounts (comma-separated)
PRIVATE_KEYS=0x1234...abc,0x5678...def
```

**Important Security Note**: Never commit the `.env` file containing real private keys to version control. The `.env` file should be added to `.gitignore`.

### Truffle Configuration

The [truffle-config.js](truffle-config.js) file defines network configurations:

- **development** - Local Ganache instance (127.0.0.1:7545)
- **production** - Production blockchain network (configured via environment variables)

## Compilation

Compile the smart contracts to generate ABI and bytecode:

```bash
npx truffle compile
```

This will:
- Compile all `.sol` files in the `contracts/` directory
- Generate artifacts in the `build/contracts/` directory
- Produce ABI (Application Binary Interface) files used by the PT API
- Validate Solidity syntax and dependencies

**Compiler Version**: Solidity 0.8.19 (configured in truffle-config.js)

## Testing

Run the Truffle test suite (if tests are defined):

```bash
npm test
```

Or use Truffle directly:

```bash
npx truffle test
```

For local testing with Ganache:

```bash
# Start Ganache on port 7545
ganache-cli -p 7545

# In another terminal, run tests
npx truffle test --network development
```

## Deployment

### Local Development (Ganache)

1. Start a local Ganache instance:

```bash
ganache-cli -p 7545 --networkId 5777
```

2. Deploy contracts to the local network:

```bash
npx truffle migrate --network development
```

### Production Blockchain

1. Ensure `.env` file is configured with production settings:

```bash
PRODUCTION_ENDPOINT=http://<address:port of the Blockchain RPC endpoint>
PRODUCTION_ID=6666
PRIVATE_KEYS=0xYourPrivateKey1,0xYourPrivateKey2
```

2. Deploy to the production network:

```bash
npx truffle migrate --network production
```

3. Note the deployed contract addresses from the output:

```
Deploying 'Provenance'
----------------------
> contract address:    0x6102a67CA8Bb97E0f5fA9991EB13f952e8cf9E4B
> block number:        12345
> account:             0xYourDeployerAddress
> balance:             100 ETH

Deploying 'Tracing'
-------------------
> contract address:    0x55F96411Ff9e424e172aB2Aa79fa5f19B09632Ce
> block number:        12346
```

4. **Important**: Update the PT API configuration with the new contract addresses:
   - Copy the deployed addresses
   - Update `.env` in the PT API project:
     ```bash
     CONTRACT_PROVENANCE_ADDR=0x6102a67CA8Bb97E0f5fA9991EB13f952e8cf9E4B
     CONTRACT_TRACING_ADDR=0x55F96411Ff9e424e172aB2Aa79fa5f19B09632Ce
     ```

### Migration Scripts

Deployment is handled by migration scripts in the `migrations/` directory:

- [1_deploy_provenance.js](migrations/1_deploy_provenance.js) - Deploys the Provenance contract
- [2_deploy_tracing.js](migrations/2_deploy_tracing.js) - Deploys the Tracing contract

## Interacting with Contracts

### Using Truffle Console

Access deployed contracts via the Truffle console:

```bash
# Connect to the network
npx truffle console --network production

# Get contract instances
let provenance = await Provenance.deployed()
let tracing = await Tracing.deployed()

# Call contract functions
await provenance.registerSource("source-123", '{"name":"My Source"}')
await tracing.postAsset("asset-456", "0x123abc...", "")

# Query data
let source = await provenance.getSource("source-123")
console.log(source)
```

### From PT API

The PT API interacts with these contracts through Web3.js. See the PT API project for integration details.

## Extracting ABI Files

After compilation, extract ABI files for use in the PT API:

```bash
# ABI files are located in build/contracts/
cp build/contracts/Provenance.json ../pt/src/integration/Provenance.abi
cp build/contracts/Tracing.json ../pt/src/integration/Tracing.abi
```

The PT API uses these ABI files to interact with the deployed contracts.

## Contract Ownership & Access Control

Both contracts implement ownership patterns:

- **Provenance**: Only the owner can register/deactivate sources
- **Tracing**:
  - Any address can post new assets via `postAsset()`
  - Only the owner can use `createEntry()` for migrations

Ownership is set to the deploying address (`msg.sender`) in the constructor.

## Gas Optimization

These contracts are optimized for gas efficiency:

- String length limits to prevent excessive gas costs
- Efficient storage patterns with mappings
- Minimal on-chain computation
- Events for off-chain indexing instead of expensive queries

## Security Considerations

1. **Access Control**: Owner-only functions are protected with modifiers
2. **Input Validation**: All inputs are validated for length and format
3. **Reentrancy**: No external calls that could enable reentrancy attacks
4. **Integer Overflow**: Solidity 0.8.x includes built-in overflow protection
5. **Immutable Records**: Once registered, records cannot be deleted (only deactivated)

## Deployment Checklist

When deploying to production:

1. **Pre-Deployment**
   - [ ] Review and audit smart contract code
   - [ ] Compile contracts: `npx truffle compile`
   - [ ] Run tests: `npm test`
   - [ ] Verify gas estimates for common operations
   - [ ] Create `.env` file with production configuration
   - [ ] Ensure deployer accounts have sufficient balance

2. **Deployment**
   - [ ] Deploy to production network: `npx truffle migrate --network production`
   - [ ] Verify deployment succeeded (check transaction receipts)
   - [ ] Record deployed contract addresses
   - [ ] Verify contract ownership is correct
   - [ ] Test basic contract functions via console

3. **Post-Deployment**
   - [ ] Update PT API configuration with new contract addresses
   - [ ] Copy ABI files to PT API integration directory
   - [ ] Restart PT API with new configuration
   - [ ] Verify PT API can interact with contracts
   - [ ] Document deployment details (addresses, block numbers, deployer)
   - [ ] Back up deployment artifacts and `.env` securely

4. **Production Verification**
   - [ ] Test source registration via PT API
   - [ ] Test asset posting via PT API
   - [ ] Verify events are emitted correctly
   - [ ] Monitor gas costs for transactions
   - [ ] Confirm blockchain explorer shows correct data (if available)

## Troubleshooting

### Common Issues

**Compilation Errors**
- Ensure Solidity version 0.8.19 is being used
- Check OpenZeppelin contracts are installed: `npm install`
- Verify syntax in `.sol` files

**Deployment Fails**
- Check blockchain RPC endpoint is accessible
- Verify deployer account has sufficient balance
- Ensure `PRIVATE_KEYS` in `.env` are correctly formatted (include `0x` prefix)
- Check `PRODUCTION_ID` matches the target blockchain network ID
- Review gas limits in [truffle-config.js](truffle-config.js)

**Transaction Reverts**
- Check function requirements (e.g., source must not already exist)
- Verify input parameters meet validation criteria (length limits, format)
- Ensure calling address has appropriate permissions (owner-only functions)
- Review event logs for detailed error messages

**Gas Estimation Errors**
- Increase `gas` limit in [truffle-config.js](truffle-config.js)
- Check string inputs are not exceeding length limits
- Verify blockchain node is synced and responsive

**ABI Mismatch in PT API**
- Ensure ABI files are copied after latest compilation
- Verify contract addresses in PT API `.env` match deployed addresses
- Restart PT API after updating configuration

## Integration with PT API

The PT API (Provenance & Tracing API) uses these contracts for:

1. **Source Management** - Registers and deactivates trusted sources
2. **Asset Tracking** - Posts new assets with cryptographic hashes
3. **Data Migration** - Uses owner-only functions to migrate historical records
4. **Event Listening** - Monitors blockchain events for real-time updates

See the PT API repository for integration implementation details.

## Development Workflow

Typical development workflow:

1. Modify smart contracts in `contracts/`
2. Compile: `npx truffle compile`
3. Test locally: Start Ganache → `npx truffle test`
4. Deploy to test network: `npx truffle migrate --network development`
5. Test integration with PT API on local environment
6. Deploy to production: `npx truffle migrate --network production`
7. Update PT API configuration
8. Verify production deployment

## Contract Upgrade Strategy

These contracts are **not upgradeable** by design - they use immutable deployment for maximum security and trust.

If contract upgrades are needed:
1. Deploy new contract versions
2. Update PT API configuration to use new addresses
3. Migrate historical data using `createEntry()` functions
4. Deprecate old contracts (keep for historical reference)

## License

This project is licensed under the MIT License. See [LICENSE](LICENSE) for details.

## Project Status

**Production** - These contracts are deployed and actively used in the FAME Marketplace platform.

## Additional Resources

- **Truffle Documentation**: https://trufflesuite.com/docs/
- **Solidity Documentation**: https://docs.soliditylang.org/
- **OpenZeppelin Contracts**: https://docs.openzeppelin.com/contracts/
- **Web3.js Documentation**: https://web3js.readthedocs.io/

## Contact

For issues, questions, or contributions, please contact the FAME development team.
