// SPDX-License-Identifier: MIT
pragma solidity ^0.8.0;

contract Provenance {

    address public immutable owner;

    modifier ownerOnly(){
        require(msg.sender == owner, "Only the contract owner can perform this action");
        _;
    }
    
    mapping (string => SourceRecord) private sources;
    string[] private pids;

    struct SourceRecord {
        string pid;
        string descriptor;
        uint registered;
        uint updated;
        uint deactivated;
    }

    event SourceEvent (string pid, string action);

    constructor() {
        owner = msg.sender;
    }

    /// @notice Registers a new Trusted Source.
    function registerSource(string calldata pid, string calldata descriptor) external ownerOnly() {
        require(bytes(pid).length > 0, "Source PID cannot be empty");
        require(bytes(pid).length <= 36, "Source PID exceeds length limit"); // UUID or shorter
        require(bytes(descriptor).length > 0, "Source descriptor cannot be empty");
        require(bytes(descriptor).length <= 1000, "Source descriptor exceeds length limit");

        SourceRecord storage rec = sources[pid];
        require(rec.registered == 0, "Source is already registered");

        sources[pid] = SourceRecord(pid, descriptor, block.timestamp, block.timestamp, 0);
        pids.push(pid);

        emit SourceEvent(pid, "REGISTERED");
    }

    /// @notice Deactivates an existing Trusted Source.
    function deactivateSource(string calldata pid) external ownerOnly() {
        require(bytes(pid).length > 0, "Source PID cannot be empty");

        SourceRecord storage rec = sources[pid];
        require(rec.registered != 0, "Source is not registered");
        require(rec.deactivated == 0, "Source has already been deactivated");

        rec.deactivated = block.timestamp;

        emit SourceEvent(pid, "DEACTIVATED");
    }

    /// @notice Returns all registerd TrustedSources.
    function getSources() external view returns (SourceRecord[] memory) {
        SourceRecord[] memory records = new SourceRecord[](pids.length);
        for (uint i = 0; i < pids.length; i++) {
            records[i] = sources[pids[i]];
        }
        return records;
    }

    /// @notice Returns the Trusted Source with the given PID.
    function getSource(string calldata pid) external view returns (SourceRecord memory) {
        return sources[pid];
    }
}
