// SPDX-License-Identifier: MIT
pragma solidity ^0.8.0;

contract Tracing {

    address public owner;

    modifier onlyOwner() {
        require(msg.sender == owner, "Only the contract owner can call this function");
        _;
    }

    constructor() {
        owner = msg.sender;
    }

    struct AssetRecord {
        string aid;
        string hash;
        address tid;
        uint registered;
        string predecessor;
        string successor;
    }

    mapping(string => AssetRecord) public assets;

    event AssetEvent(string aid, string action);

    /// @notice Registers a new Trusted Asset.
    function postAsset(string calldata aid, string calldata hash, string calldata predecessor) external {
        require(bytes(aid).length > 0, "Asset AID cannot be empty");
        require(bytes(aid).length <= 36, "Asset AID exceeds length limit"); // UUID or shorter
        require(bytes(hash).length == 66, "Asset hash value is not of the correct legth"); // 64 chars + 0x prefix

        AssetRecord storage rec = assets[aid];
        require(rec.registered == 0, "Asset is already registered");

        if (bytes(predecessor).length > 0) {
            // Predecessor provided - validate and link
            require(keccak256(abi.encodePacked(predecessor)) != keccak256(abi.encodePacked(aid)), "Asset cannot be its own predecessor");
            AssetRecord storage predecessorRec = assets[predecessor];
            require(predecessorRec.registered != 0, "Predecessor asset does not exist");
            require(bytes(predecessorRec.successor).length == 0, "Predecessor asset already has a successor");

            // Create new asset with given predecessor
            assets[aid] = AssetRecord(aid, hash, msg.sender, block.timestamp, predecessor, "");

            // Update predecessor asset to point to this new asset
            predecessorRec.successor = aid;
        } else {
            // No predecessor - create standalone asset
            assets[aid] = AssetRecord(aid, hash, msg.sender, block.timestamp, "", "");
        }

        emit AssetEvent(aid, "REGISTERED");
    }

    /// @notice Creates an asset entry for migration purposes (owner only).
    /// This method is idempotent - calling it multiple times with the same parameters has no effect after the first call.
    /// HOWEVER, keep in mind that if no state changes occur, no event will be emitted: this is due to how Besu internally
    /// handles transactions, and cannot be changed by the smart contract.
    function createEntry(
        string calldata aid,
        string calldata hash,
        address tid,
        uint registered,
        string calldata predecessor,
        string calldata successor
    ) external onlyOwner {
        require(bytes(aid).length > 0, "Asset AID cannot be empty");
        require(bytes(aid).length <= 36, "Asset AID exceeds length limit");
        require(bytes(hash).length == 66, "Asset hash value is not of the correct length");

        assets[aid] = AssetRecord(aid, hash, tid, registered, predecessor, successor);

        emit AssetEvent(aid, "MIGRATED");
    }


    /// @notice Retrieves a Trusted Asset by its AID.
    function getAsset(string calldata aid) external view returns (AssetRecord memory) {
        return assets[aid];
    }

    /// @notice Retrieves the lineage of an asset by tracing back its predecessors.
    function getLineage(string calldata aid) external view returns (string[] memory) {
        AssetRecord storage rec = assets[aid];

        // If asset doesn't exist, return empty array
        if (rec.registered == 0) {
            return new string[](0);
        }

        // Count the length of the lineage first
        uint256 count = 1;
        string memory currentAid = aid;
        while (bytes(assets[currentAid].predecessor).length > 0) {
            currentAid = assets[currentAid].predecessor;
            count++;
        }

        // Build the lineage array from root to current
        string[] memory lineage = new string[](count);

        // Start from the root (no predecessor)
        lineage[0] = currentAid;

        // Follow successors to build the chain
        for (uint256 i = 1; i < count; i++) {
            lineage[i] = assets[lineage[i-1]].successor;
        }

        return lineage;
    }
}
