#!/bin/bash
# Automated restart script for Besu RPC node container
# This script restarts the RPC node container and logs the operation

# Configuration - UPDATE THESE VALUES
RPC_CONTAINER_NAME="rpcnode"
LOG_FILE="/var/log/besu-rpc-restart.log"

# Function to log messages with timestamp
log_message() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S %Z')] $1" | tee -a "$LOG_FILE"
}

# Start the restart process
log_message "=========================================="
log_message "Starting RPC node restart process"

# Check if container exists
if ! docker ps -a --format '{{.Names}}' | grep -q "^${RPC_CONTAINER_NAME}$"; then
    log_message "ERROR: Container '${RPC_CONTAINER_NAME}' not found"
    log_message "Available containers:"
    docker ps -a --format '{{.Names}}' | tee -a "$LOG_FILE"
    exit 1
fi

# Get container status before restart
CONTAINER_STATUS=$(docker inspect -f '{{.State.Status}}' "$RPC_CONTAINER_NAME")
log_message "Container status before restart: $CONTAINER_STATUS"

# Restart the container
log_message "Restarting container: $RPC_CONTAINER_NAME"
if docker restart "$RPC_CONTAINER_NAME" >> "$LOG_FILE" 2>&1; then
    log_message "Container restart command executed successfully"

    # Wait a few seconds for the container to stabilize
    sleep 5

    # Verify container is running
    NEW_STATUS=$(docker inspect -f '{{.State.Status}}' "$RPC_CONTAINER_NAME")
    log_message "Container status after restart: $NEW_STATUS"

    if [ "$NEW_STATUS" = "running" ]; then
        log_message "SUCCESS: RPC node container is running"
    else
        log_message "WARNING: Container status is '$NEW_STATUS' (expected 'running')"
    fi
else
    log_message "ERROR: Failed to restart container"
    exit 1
fi

log_message "RPC node restart process completed"
log_message "=========================================="
