# Tracing Contract Migration Utility

This utility is designed to migrate data from the old Tracing contract to the new version by populating the ledger using the `createEntry()` function.

## Overview

The migration utility:
- Reads a JSON file containing asset records exported from the old system
- Filters records to only process those with a `pid` property (ignores others silently)
- For each valid record, fetches the canonical description from FDAC service
- Calculates hash values using the canonical description
- Executes `createEntry()` transactions sequentially on the new Tracing contract
- Waits for each transaction to complete (listening for "MIGRATED" events) before proceeding
- Provides progress tracking and error handling
- Fails fast on any transaction error (since `createEntry()` is idempotent, the process can be restarted)

## Prerequisites

1. **Environment Variables**: Ensure the following environment variables are configured:
   - `PRIVATE_KEY_PT`: Private key for the contract owner account
   - `MIGRATION_ACCOUNT`: Address to use as the `tid` for all migrated records
   - `CHAIN_URL`: WebSocket URL for the blockchain connection
   - `CHAIN_ID`: Chain ID for the blockchain
   - `CONTRACT_TRACING_ADDR`: Address of the new Tracing contract
   - `CONTRACT_TRACING_ABI`: Path to the Tracing contract ABI file
   - `FDAC_URL`: URL for the FDAC service
   - `FDAC_PUB_TOKEN`: Authentication token for FDAC service
   - `FDAC_CERT_TOKEN`: Certification token for FDAC service

2. **Dependencies**: Install required dependencies:
   ```bash
   npm install
   ```

## Data Format

The migration utility expects a JSON file containing an array of asset records exported from the old system. Each record should have the following structure:

```json
{
  "assetId": "string",     // Asset ID (required)
  "value": {               // Asset value object (required)
    "pid": "string",       // Persistent identifier (required for processing)
    "name": "string",      // Asset name
    "description": "string", // Asset description
    // ... other asset properties
  }
}
```

**Important**: Only records containing a `pid` property in the `value` object will be processed. Records without `pid` are silently skipped.

### Example Migration Data File

See `migration-data-sample.json` for an example:

```json
[
  {
    "assetId": "befa6d63138ee041f31f8783",
    "value": {
      "name": "SUMO tool",
      "description": "Eclipse SUMO simulation tool",
      "type": "Simulator",
      "version": "1.0"
    }
  },
  {
    "assetId": "BoQA5vPzG2BHKSZoX",
    "value": {
      "name": "Smart Deployment API-Server",
      "description": "Application to build everything needed to deploy models",
      "type": "Application",
      "pid": "fame_man_id:bKvyK3Hkfi2NVpqAxejsfb",
      "version": "1.0"
    }
  }
]
```

In this example, only the second record will be processed because it contains a `pid` property.

## Migration Process

For each record with a `pid` property, the utility:

1. **Extracts Asset ID**: Uses the `assetId` from the source record
2. **Fetches Canonical Description**: Calls `FdacService.getAssetCanonicalDescription(assetId)`
3. **Calculates Hash**: Uses `BlockchainService.getCanonicalHash(JSON.stringify(canonicalDescription))`
4. **Creates Migration Record**:
   - `aid` = assetId
   - `hash` = calculated hash
   - `tid` = MIGRATION_ACCOUNT environment variable
   - `registered` = current timestamp (seconds since epoch)
   - `predecessor` = empty string
   - `successor` = empty string
5. **Executes Transaction**: Calls the contract's `createEntry()` function
6. **Waits for Confirmation**: Listens for "MIGRATED" event before proceeding

## Usage

### Command Line

Run the migration utility using the npm script:

```bash
npm run migrate <migration-file.json>
```

Example:
```bash
npm run migrate ./migration-data-sample.json
npm run migrate /path/to/production-migration-data.json
```

### Direct TypeScript Execution

You can also run the utility directly:

```bash
npx ts-node src/migration/migration-utility.ts <migration-file.json>
```

## How It Works

1. **Initialization**:
   - Initializes blockchain service and connects to the network
   - Initializes FDAC service for fetching canonical descriptions
   - Validates all required environment variables
2. **Data Loading**:
   - Reads and validates the migration JSON file
   - Filters records to only include those with `pid` property
   - For each valid record, fetches canonical description from FDAC
   - Calculates hash and prepares migration data
3. **Event Subscription**: Subscribes to AssetEvent emissions from the Tracing contract
4. **Sequential Processing**:
   - Processes one record at a time
   - Executes `createEntry()` transaction
   - Waits for "MIGRATED" event before proceeding to next record
   - Includes 1-second delay between records to avoid overwhelming the system
5. **Progress Tracking**: Logs progress and completion status
6. **Error Handling**: Stops on any transaction failure (fatal error as specified)

## Key Features

- **Automatic Filtering**: Only processes records with `pid` property, silently skips others
- **FDAC Integration**: Fetches canonical descriptions and calculates correct hash values
- **Sequential Processing**: Only one transaction is processed at a time
- **Event-Driven**: Waits for blockchain confirmation before proceeding
- **Idempotent**: Safe to restart if interrupted (due to contract's idempotent design)
- **Duplicate Detection**: Checks if assets already exist on-chain before attempting migration
- **Progress Tracking**: Real-time progress updates and completion percentage
- **Validation**: Comprehensive input validation before processing
- **Error Handling**: Clear error messages and graceful failure handling

## Monitoring

The utility provides detailed logging including:
- Record filtering (`Loaded X total records, Y records have 'pid' property`)
- FDAC interactions (`Processing asset X/Y: assetId`)
- Hash calculations (`Prepared migration record for assetId with hash hashValue`)
- Progress updates (`Processing record X/Y - AID: asset-id`)
- Transaction confirmations (`Migration successful for record X/Y - AID: asset-id`)
- Error details if transactions fail
- Final completion message

## Security Considerations

- The utility uses the owner account configured via `PRIVATE_KEY_PT` environment variable
- Only the contract owner can call `createEntry()`
- The private key should be securely managed (e.g., Kubernetes secrets in production)
- The utility connects to the new contract address specified in environment variables
- FDAC authentication tokens should be securely managed

## Contract Addresses

- **Old Contract**: `0xe94fa3040d20a01a1b8f42De7c430A11c27e2524`
- **New Contract**: `0x55F96411Ff9e424e172aB2Aa79fa5f19B09632Ce` (configured via environment)

## Troubleshooting

### WebSocket Connection Issues

**Problem**: "Waiting for blockchain connection to stabilize..." or repeated WebSocket errors during execution.

**Causes**:
- Blockchain node is not accessible at the configured `CHAIN_URL`
- WebSocket endpoint is behind a firewall or proxy
- Blockchain node is overloaded or restarting
- Network connectivity issues

**Solutions**:
1. **Verify WebSocket URL**: Ensure `CHAIN_URL` is correct and accessible
   ```bash
   # Test WebSocket connection (PowerShell)
   Test-NetConnection -ComputerName rpc-node -Port 8546
   ```

2. **Check if endpoint is for internal network only**: The URL `ws://rpc-node:8546/websocket` suggests this is an internal hostname. Ensure you're running the migration from within the correct network/cluster.

3. **Use alternative connection method**: If WebSocket is unstable, consider using HTTP instead (requires code changes to BlockchainService).

4. **Increase connection timeout**: The utility waits up to 30 seconds for connection to stabilize. If the blockchain node is slow to respond, this may not be enough.

5. **Check blockchain node status**: Verify the blockchain node is running and responsive.

### Besu Event Emission Behavior

**Important Note**: The blockchain uses Hyperledger Besu, which has an optimization where events are not emitted when a transaction doesn't actually change the state. This means:

- If `createEntry()` is called with data that matches an existing asset, the transaction will succeed (status=1) but emit **no events**
- The transaction receipt will have `logs.length = 0` even though the transaction was successful
- This is **by design** for efficiency - Besu doesn't emit events when state remains unchanged

**Impact on Migration**:
- The utility checks if assets already exist before attempting migration
- Assets with `registered != 0` are skipped to avoid unnecessary transactions
- This prevents the utility from hanging while waiting for events that will never arrive

### Other Common Issues

1. **Permission Errors**: Ensure `PRIVATE_KEY_PT` corresponds to contract owner
2. **FDAC Errors**: Check `FDAC_URL`, `FDAC_PUB_TOKEN`, and `FDAC_CERT_TOKEN`
3. **Missing Assets**: Verify assets exist in FDAC system
4. **Transaction Failures**: Check contract address, ABI, and data format
5. **Missing MIGRATION_ACCOUNT**: Ensure `MIGRATION_ACCOUNT` environment variable is set

## Recovery

If the migration is interrupted:
1. Note the last successfully migrated AID from logs
2. Remove all records up to and including that AID from the JSON file
3. Restart the migration with the remaining records

The process is safe to restart due to the idempotent nature of `createEntry()`.