# FAME Provenance & Tracing API

The **Provenance & Tracing API** is a core backend module of the **FAME Marketplace platform**, responsible for managing data provenance, asset tracing, and blockchain-based immutable records. It provides comprehensive tracking capabilities for data assets and offerings throughout their lifecycle on the FAME platform.

## Overview

This service acts as a bridge between the FAME Marketplace and blockchain infrastructure, enabling:

- **Data Provenance**: Tracking the origin, ownership, and lineage of data assets
- **Asset Tracing**: Recording and retrieving complete audit trails for data assets
- **Blockchain Integration**: Interfacing with smart contracts for immutable record-keeping
- **Permission Management**: Managing access control and usage rights for assets
- **Statistics & Analytics**: Providing insights into catalogue and blockchain operations

The API is built with **NestJS** and TypeScript, offering both public and internal endpoints for integration with other FAME platform components.

## Quick Start

There are three primary ways to deploy this service:

### 1. Local Development (Node.js)
For development and testing purposes:
```bash
npm install
npm run build
# Set required environment variables (see Environment Configuration)
export PRIVATE_KEY_PT="..."
export DATABASE_URI="..."
# ... (other sensitive vars)
node dist/src/main.js
```
→ Access at `http://localhost:7005/swagger` (public) or `http://localhost:7010/swagger` (internal)

### 2. Docker Container
For containerized deployment:
```bash
npm run build
npm run docker:build
npm run docker:push
docker run -d -p 7005:7005 --env-file .env -e PRIVATE_KEY_PT='...' harbor.gftinnovation.eu/fame/framework/pt:3.3
```
→ Suitable for standalone deployment with manual configuration

### 3. Kubernetes (Production)
For production deployment on Kubernetes clusters:
```bash
# Create ConfigMap and Secrets
kubectl apply -f pt-service.yaml
```
→ Recommended for production - separates configuration from secrets, supports dual deployment (public + internal)

**Important Security Note**: The `.env` file in the repository does **not** contain sensitive values. You must set the following environment variables separately:
- `PRIVATE_KEY_PT`, `PRIVATE_KEY_PERMISSION` - Blockchain private keys
- `FDAC_PUB_TOKEN`, `FDAC_CERT_TOKEN` - FDAC tokens
- `DATABASE_URI` - MongoDB connection string with credentials

## Table of Contents

- [Overview](#overview)
- [Architecture](#architecture)
- [Prerequisites](#prerequisites)
- [Environment Configuration](#environment-configuration)
- [Building the Application](#building-the-application)
- [Running the Application Locally](#running-the-application-locally)
- [Docker Deployment](#docker-deployment)
- [Kubernetes Deployment](#kubernetes-deployment)
- [Deployment Checklist](#deployment-checklist)
- [API Documentation](#api-documentation)
- [Data Migration](#data-migration)
- [Troubleshooting](#troubleshooting)
- [License](#license)
- [Project Status](#project-status)

## Architecture

The service is organized into several functional modules:

- **Assets Module**: Asset lifecycle management and metadata handling
- **Offerings Module**: Offering catalogue and descriptor management
- **Sources Module**: Data source tracking and registration
- **Permissions Module**: Access control and authorization
- **Stats Module**: Platform statistics and analytics
- **Integration Module**: Communication with other FAME services (GOV, APM, TM, PAT, FDAC)

## Prerequisites

Before building and deploying the service, ensure you have the following installed:

- **Node.js** 20.x (LTS Alpine recommended for Docker builds)
- **npm** (comes with Node.js)
- **Docker** (for containerized deployment)
- For deployment to the GFT hosting infrastructure: access to **Harbor registry** at `harbor.gftinnovation.eu`

## Environment Configuration

The service requires various environment variables to be configured. The configuration approach differs depending on the deployment scenario:

### Local Development/Testing (using `.env` file)

For running the service locally during development and testing, create a `.env` file at the project root. **Important**: The `.env` file in the repository does not contain sensitive values (secrets) for security reasons. You must manually set the following **sensitive variables** as environment variables or add them to your local `.env` file (do not commit them):

- `PRIVATE_KEY_PT` - Private key for blockchain provenance transactions
- `PRIVATE_KEY_PERMISSION` - Private key for blockchain permission transactions
- `FDAC_PUB_TOKEN` - FDAC public API token
- `FDAC_CERT_TOKEN` - FDAC certificate token
- `DATABASE_URI` - Complete MongoDB connection string including credentials

### Production Deployment (Kubernetes)

For production deployment on Kubernetes (as used in the FAME production environment):

- **Non-sensitive configuration** is defined in a **ConfigMap** (see example in `fame-integration/pt/pt-service.yaml`)
- **Sensitive values** are stored in **Kubernetes Secrets** and injected as environment variables using `secretKeyRef`

The Kubernetes deployment uses:
- ConfigMap named `pt-config` for general configuration
- Secret named `fame-secrets` containing:
  - `PT_PKEY` → `PRIVATE_KEY_PT`
  - `GOV_PKEY` → `PRIVATE_KEY_PERMISSION`
  - `FDAC_TOK_PUB` → `FDAC_PUB_TOKEN`
  - `FDAC_TOK_CERT` → `FDAC_CERT_TOKEN`
  - `UNIDB_PASS` → Used to construct `DATABASE_URI`

### Configuration Parameters

You will need to know the network addresses of all the services that this service depends on. These addresses depend on your specific hosting environment. Key configuration parameters include:

### Blockchain Configuration
```bash
CHAIN_ID=6666
CHAIN_URL=ws://<address:port of Blockchain RPC server>/websocket
CHAIN_KEEPALIVE=30000
CONTRACT_PROVENANCE_ABI=integration/Provenance.abi
CONTRACT_PROVENANCE_ADDR=0x6102a67CA8Bb97E0f5fA9991EB13f952e8cf9E4B
CONTRACT_TRACING_ABI=integration/Tracing.abi
CONTRACT_TRACING_ADDR=0x55F96411Ff9e424e172aB2Aa79fa5f19B09632Ce
CONTRACT_PERMISSION_ABI=integration/Permission.abi
CONTRACT_PERMISSION_ADDR=0x7cA5543f9B2C35F0E972f1B45b61A2FE53fF1ed9
```

### Database Configuration
```bash
DATABASE_URI=mongodb://<username:password for MongoDB access>@<address:port of MongoDB server>
DATABASE_NAME=FAME
DATABASE_COLLECTION=offerings
DATABASE_AUTH_SOURCE=admin
```

### Authentication & Authorization
```bash
AUTH_URL=https://<address of AAI API public endpoint>/api
DISABLE_AUTH=false
DISABLE_ROLES=false
```

### Service Integration URLs
```bash
INTERNAL_URL=http://<address of PT API internal endpoint>/pt/v1.0
GOV_URL=https://<address of GOV API internal endpoint>/gov/v1.0
APM_URL=https://<address of APM API internal endpoint>/private/api/v1.0
TM_URL=https://<address of TM API internal endpoint>/tm/v1.0
PAT_URL=https://<address of PAT API internal endpoint>/pat-internal/v1.0
FDAC_URL=https://<address of FDAC API public endpoint>/api
```


## Building the Application

### 1. Install Dependencies

```bash
npm install
```

This will install all required dependencies from [package.json](package.json), including:
- NestJS framework and modules
- MongoDB driver (Mongoose)
- Web3.js for blockchain interaction
- JWT authentication libraries
- Validation and transformation utilities

### 2. Compile TypeScript Code

```bash
npm run build
```

This command executes the NestJS build process, which:
- Compiles TypeScript files from `src/` to JavaScript in `dist/`
- Copies asset files (ABI definitions) to the output directory
- Generates source maps for debugging
- Validates TypeScript configuration

The compiled output will be placed in the `dist/` directory with the following structure:
```
dist/
└── src/
    ├── main.js
    ├── app.module.js
    ├── assets/
    ├── offerings/
    ├── sources/
    ├── permissions/
    ├── stats/
    ├── integration/
    │   ├── Provenance.abi
    │   ├── Tracing.abi
    │   └── Permission.abi
    └── ...
```

### 3. Code Quality (Optional)

Run the linter to check and auto-fix code style issues:

```bash
npm run lint
```

## Running the Application Locally

**Prerequisites**: Before running locally, ensure you have:
1. Completed the build process (`npm install` && `npm run build`)
2. Created a `.env` file with all required configuration (see [Environment Configuration](#environment-configuration))
3. Manually set all sensitive environment variables (these are **not** in the repository `.env` file)

### Standard Mode (Public API)

The public API runs on port **7005** and exposes endpoints for external consumption.

```bash
# Ensure all sensitive environment variables are set
export PRIVATE_KEY_PT="your-blockchain-private-key-provenance"
export PRIVATE_KEY_PERMISSION="your-blockchain-private-key-permission"
export FDAC_PUB_TOKEN="your-fdac-public-token"
export FDAC_CERT_TOKEN="your-fdac-certificate-token"
export DATABASE_URI="mongodb://username:password@host:port/FAME"

# Set INTERNAL_API to false or omit it
export INTERNAL_API=false

# Start the application
node dist/src/main.js
```

Access points:
- **API**: `http://localhost:7005`
- **Swagger Documentation**: `http://localhost:7005/swagger`

### Internal Mode (Private API)

The internal API runs on port **7010** and provides privileged endpoints for inter-service communication.

```bash
# Ensure all sensitive environment variables are set (same as above)
export PRIVATE_KEY_PT="your-blockchain-private-key-provenance"
export PRIVATE_KEY_PERMISSION="your-blockchain-private-key-permission"
export FDAC_PUB_TOKEN="your-fdac-public-token"
export FDAC_CERT_TOKEN="your-fdac-certificate-token"
export DATABASE_URI="mongodb://username:password@host:port/FAME"

# Set INTERNAL_API to true
export INTERNAL_API=true

# Start the application
node dist/src/main.js
```

Access points:
- **API**: `http://localhost:7010`
- **Swagger Documentation**: `http://localhost:7010/swagger`

The application will automatically generate the appropriate Swagger specification file (`swagger-specification.public.json` or `swagger-specification.private.json`) on startup based on the `INTERNAL_API` setting.

## Docker Deployment

### Building the Docker Image

The service uses a multi-stage Docker build optimized for production deployment.

```bash
npm run docker:build
```

This executes:
```bash
docker build -t harbor.gftinnovation.eu/fame/framework/pt:3.3 .
```

The [Dockerfile](Dockerfile) performs the following:
1. Uses **Node.js 20 Alpine** base image for minimal footprint
2. Sets working directory to `/app`
3. Installs production dependencies using npm cache mounts for faster builds
4. Copies the entire source code (including pre-built `dist/` directory)
5. Exposes port **7005** (public API) or **7010** (internal API)
6. Sets entrypoint to `node dist/src/main.js`

**Important**: You must run `npm run build` **before** building the Docker image, as the Dockerfile expects the compiled code to be present in the `dist/` directory.

### Pushing to Harbor Registry

After building the image, push it to the Harbor container registry:

```bash
npm run docker:push
```

This executes:
```bash
docker login harbor.gftinnovation.eu
docker push harbor.gftinnovation.eu/fame/framework/pt:3.3
```

You will be prompted for Harbor credentials. Ensure you have the appropriate permissions to push to the `fame/framework` project.

### Running the Docker Container

**Important**: When running Docker containers, you must provide sensitive environment variables separately (do not include them in the `.env` file). You can pass them as individual `-e` flags or through a secure secrets management system.

#### Public API Container

```bash
docker run -d \
  --name pt-public \
  -p 7005:7005 \
  --env-file .env \
  -e INTERNAL_API=false \
  -e PRIVATE_KEY_PT='<blockchain-private-key-provenance>' \
  -e PRIVATE_KEY_PERMISSION='<blockchain-private-key-permission>' \
  -e FDAC_PUB_TOKEN='<fdac-public-token>' \
  -e FDAC_CERT_TOKEN='<fdac-certificate-token>' \
  -e DATABASE_URI='mongodb://username:password@host:port/FAME' \
  harbor.gftinnovation.eu/fame/framework/pt:3.3
```

#### Internal API Container

```bash
docker run -d \
  --name pt-internal \
  -p 7010:7010 \
  --env-file .env \
  -e INTERNAL_API=true \
  -e PRIVATE_KEY_PT='<blockchain-private-key-provenance>' \
  -e PRIVATE_KEY_PERMISSION='<blockchain-private-key-permission>' \
  -e FDAC_PUB_TOKEN='<fdac-public-token>' \
  -e FDAC_CERT_TOKEN='<fdac-certificate-token>' \
  -e DATABASE_URI='mongodb://username:password@host:port/FAME' \
  harbor.gftinnovation.eu/fame/framework/pt:3.3
```

#### With MongoDB and Blockchain Node (Docker Compose Example)

For local development with all dependencies:

```yaml
version: '3.8'
services:
  mongodb:
    image: mongo:7
    ports:
      - "27017:27017"
    environment:
      MONGO_INITDB_ROOT_USERNAME: admin
      MONGO_INITDB_ROOT_PASSWORD: password
    volumes:
      - mongo-data:/data/db

  pt-api:
    image: harbor.gftinnovation.eu/fame/framework/pt:3.3
    ports:
      - "7005:7005"
    environment:
      DATABASE_URI: mongodb://admin:password@mongodb:27017
      DATABASE_NAME: FAME
      DATABASE_AUTH_SOURCE: admin
      CHAIN_URL: ws://rpc-node:8546/websocket
      INTERNAL_API: "false"
    depends_on:
      - mongodb

volumes:
  mongo-data:
```

## Kubernetes Deployment

For production deployment on Kubernetes clusters (as used in the FAME production environment), the service is deployed using Kubernetes manifests that separate configuration from secrets.

### Deployment Architecture

The PT API is deployed as **two separate deployments** in Kubernetes:

1. **Public API** (`pt`) - Exposed externally on port 7005
2. **Internal API** (`pt-internal`) - Used for inter-service communication on port 7010

Both deployments use the same Docker image but differ in the `INTERNAL_API` environment variable.

### Kubernetes Resources

The deployment consists of:

- **ConfigMap** (`pt-config`) - Non-sensitive configuration variables
- **Secret** (`fame-secrets`) - Sensitive credentials and keys
- **Deployments** (`pt` and `pt-internal`) - Application pods
- **Services** (`pt` and `pt-internal`) - Network services
- **Ingress** - External routing rules (defined in `pt-ingress.yaml`)

### Creating the ConfigMap

Create a ConfigMap with all non-sensitive configuration:

```yaml
apiVersion: v1
kind: ConfigMap
metadata:
  name: pt-config
  namespace: fame
data:
  CHAIN_ID: "6666"
  CHAIN_URL: ws://rpc-node:8546/websocket
  CHAIN_KEEPALIVE: "30000"
  CONTRACT_PROVENANCE_ABI: integration/Provenance.abi
  CONTRACT_PROVENANCE_ADDR: "0x6102a67CA8Bb97E0f5fA9991EB13f952e8cf9E4B"
  CONTRACT_TRACING_ABI: integration/Tracing.abi
  CONTRACT_TRACING_ADDR: "0x55F96411Ff9e424e172aB2Aa79fa5f19B09632Ce"
  CONTRACT_PERMISSION_ABI: integration/Permission.abi
  CONTRACT_PERMISSION_ADDR: "0x7cA5543f9B2C35F0E972f1B45b61A2FE53fF1ed9"
  FDAC_URL: https://fame-fdac.iot-catalogue.com/api
  AUTH_URL: https://auth.fame-horizon.eu/api
  INTERNAL_URL: http://pt-internal:7010/pt/v1.0
  GOV_URL: http://gov-internal:7008/gov/v1.0
  APM_URL: http://apm:8080/private/api/v1.0
  TM_URL: http://tm-internal:3000/tm/v1.0
  PAT_URL: http://pat-internal:7008/pat-internal/v1.0
  DATABASE_NAME: FAME
  DATABASE_COLLECTION: offerings
  DATABASE_AUTH_SOURCE: admin
  NODE_TLS_REJECT_UNAUTHORIZED: "0"
  DISABLE_AUTH: "false"
  DISABLE_ROLES: "false"
  DISABLE_RQUEUE: "false"
  DISABLE_FDAC: "false"
  DISABLE_APM: "false"
  DISABLE_TM: "false"
  DISABLE_OFFERING_XCHECK: "false"
```

### Creating the Secrets

Create Kubernetes secrets for sensitive values:

```bash
kubectl create secret generic fame-secrets \
  --from-literal=PT_PKEY='<blockchain-private-key-provenance>' \
  --from-literal=GOV_PKEY='<blockchain-private-key-permission>' \
  --from-literal=FDAC_TOK_PUB='<fdac-public-token>' \
  --from-literal=FDAC_TOK_CERT='<fdac-certificate-token>' \
  --from-literal=UNIDB_PASS='<mongodb-password>' \
  --namespace=fame
```

### Deploying to Kubernetes

Once the ConfigMap and Secrets are created, apply the deployment manifests:

```bash
# Apply the deployment and service definitions
kubectl apply -f pt-service.yaml

# Verify deployments
kubectl get deployments -n fame | grep pt

# Check pod status
kubectl get pods -n fame | grep pt

# View logs
kubectl logs -n fame deployment/pt
kubectl logs -n fame deployment/pt-internal
```

### Example Deployment Manifest Structure

The deployment manifest includes both public and internal API deployments:

```yaml
apiVersion: apps/v1
kind: Deployment
metadata:
  name: pt
  namespace: fame
spec:
  replicas: 1
  selector:
    matchLabels:
      app: pt
  template:
    metadata:
      labels:
        app: pt
    spec:
      containers:
      - name: pt
        image: harbor.gftinnovation.eu/fame/framework/pt:3.3
        imagePullPolicy: Always
        command: ["node", "dist/src/main.js"]
        resources:
          requests:
            memory: "512Mi"
            cpu: "100m"
          limits:
            memory: "512Mi"
            cpu: "100m"
        # Load all non-sensitive config from ConfigMap
        envFrom:
        - configMapRef:
            name: pt-config
        # Inject sensitive values from Secrets
        env:
        - name: PRIVATE_KEY_PT
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: PT_PKEY
        - name: PRIVATE_KEY_PERMISSION
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: GOV_PKEY
        - name: FDAC_PUB_TOKEN
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: FDAC_TOK_PUB
        - name: FDAC_CERT_TOKEN
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: FDAC_TOK_CERT
        - name: DATABASE_PASS
          valueFrom:
            secretKeyRef:
              name: fame-secrets
              key: UNIDB_PASS
        - name: DATABASE_URI
          value: mongodb://root:$(DATABASE_PASS)@catalogue-db:27018/FAME
        - name: INTERNAL_API
          value: "false"
        ports:
        - containerPort: 7005
      imagePullSecrets:
        - name: registrysecret
```

### Updating the Deployment

To update the service with a new version:

```bash
# Build and push new Docker image (update version tag)
npm run build
docker build -t harbor.gftinnovation.eu/fame/framework/pt:3.4 .
docker push harbor.gftinnovation.eu/fame/framework/pt:3.4

# Update the deployment manifest with new image tag
# Then apply the changes
kubectl apply -f pt-service.yaml

# Or use kubectl set image for quick updates
kubectl set image deployment/pt pt=harbor.gftinnovation.eu/fame/framework/pt:3.4 -n fame
kubectl set image deployment/pt-internal pt-internal=harbor.gftinnovation.eu/fame/framework/pt:3.4 -n fame

# Monitor rollout status
kubectl rollout status deployment/pt -n fame
kubectl rollout status deployment/pt-internal -n fame
```

### Accessing Kubernetes Services

Once deployed, the services are accessible:

- **Public API** (external): Via Ingress rules defined in `pt-ingress.yaml`
- **Internal API** (cluster-internal): `http://pt-internal:7010` from other pods
- **Swagger Documentation**:
  - Public: `https://<ingress-host>/pt/swagger`
  - Internal: `http://pt-internal:7010/swagger`

### Complete Kubernetes Manifests

For complete reference implementations, see the manifest files in the `fame-integration` project:
- [pt-service.yaml](../fame-integration/pt/pt-service.yaml) - Deployments, Services, and ConfigMap
- [pt-ingress.yaml](../fame-integration/pt/pt-ingress.yaml) - Ingress routing rules
- [catalogue-db.yaml](../fame-integration/pt/catalogue-db.yaml) - MongoDB deployment

## Deployment Checklist

### For Local Development/Testing

When running the service locally for development and testing:

1. **Build Process**
   - [ ] Run `npm install` to install dependencies
   - [ ] Run `npm run build` to compile TypeScript
   - [ ] Verify `dist/` directory contains compiled code
   - [ ] Check that ABI files are copied to `dist/src/integration/`

2. **Environment Configuration**
   - [ ] Copy `.env` file from repository as starting point
   - [ ] Manually set sensitive environment variables:
     - [ ] `PRIVATE_KEY_PT` - Blockchain provenance private key
     - [ ] `PRIVATE_KEY_PERMISSION` - Blockchain permission private key
     - [ ] `FDAC_PUB_TOKEN` - FDAC public token
     - [ ] `FDAC_CERT_TOKEN` - FDAC certificate token
     - [ ] `DATABASE_URI` - MongoDB connection string with credentials
   - [ ] Configure service URLs to point to your local or test environment
   - [ ] Set `INTERNAL_API=false` for public API or `INTERNAL_API=true` for internal API

3. **Infrastructure Access**
   - [ ] Ensure MongoDB is accessible and credentials are correct
   - [ ] Verify blockchain RPC node connectivity
   - [ ] Confirm network access to dependent services (GOV, APM, TM, PAT, FDAC)

4. **Verification**
   - [ ] Application starts without errors: `node dist/src/main.js`
   - [ ] Swagger UI accessible at `http://localhost:7005/swagger` or `http://localhost:7010/swagger`
   - [ ] Test basic API endpoints
   - [ ] Check logs for blockchain connection status
   - [ ] Validate database connection (check logs for MongoDB connection)

### For Docker Deployment

When deploying using Docker containers:

1. **Build Process**
   - [ ] Run `npm install` to install dependencies
   - [ ] Run `npm run build` to compile TypeScript
   - [ ] Verify `dist/` directory contains compiled code
   - [ ] Check that ABI files are copied to `dist/src/integration/`

2. **Docker Image**
   - [ ] Build Docker image: `npm run docker:build`
   - [ ] Tag image appropriately (update version in [package.json](package.json))
   - [ ] Push to Harbor registry: `npm run docker:push`
   - [ ] Verify image is accessible in Harbor

3. **Environment Configuration**
   - [ ] Create `.env` file with non-sensitive values
   - [ ] Set sensitive environment variables separately (do not include in `.env` file)
   - [ ] Configure `INTERNAL_API` flag based on deployment mode

4. **Deployment**
   - [ ] Run container with appropriate environment variables
   - [ ] Map correct port (7005 for public, 7010 for internal)
   - [ ] Configure network connectivity to MongoDB and blockchain RPC
   - [ ] Set up networking for inter-service communication

5. **Verification**
   - [ ] Container starts without errors: `docker logs <container-name>`
   - [ ] API responds at `http://<host>:<port>/swagger`
   - [ ] Test authentication endpoints
   - [ ] Confirm blockchain connectivity (check container logs)
   - [ ] Validate database connection (check container logs)

### For Kubernetes Production Deployment

When deploying to production Kubernetes clusters:

1. **Build Process**
   - [ ] Run `npm install` to install dependencies
   - [ ] Run `npm run build` to compile TypeScript
   - [ ] Verify `dist/` directory contains compiled code
   - [ ] Check that ABI files are copied to `dist/src/integration/`

2. **Docker Image**
   - [ ] Build Docker image with appropriate version tag
   - [ ] Push to Harbor registry: `npm run docker:push`
   - [ ] Verify image is accessible in Harbor
   - [ ] Update image tag in Kubernetes deployment manifests

3. **Kubernetes Configuration**
   - [ ] Create or update ConfigMap (`pt-config`) with non-sensitive configuration:
     - [ ] Blockchain configuration (chain ID, RPC URL, contract addresses)
     - [ ] Service integration URLs (GOV, APM, TM, PAT, FDAC, AUTH)
     - [ ] Database configuration (name, collection, auth source)
     - [ ] Feature flags (DISABLE_* variables)
   - [ ] Create or update Kubernetes Secrets (`fame-secrets`) with sensitive values:
     - [ ] `PT_PKEY` → `PRIVATE_KEY_PT`
     - [ ] `GOV_PKEY` → `PRIVATE_KEY_PERMISSION`
     - [ ] `FDAC_TOK_PUB` → `FDAC_PUB_TOKEN`
     - [ ] `FDAC_TOK_CERT` → `FDAC_CERT_TOKEN`
     - [ ] `UNIDB_PASS` → MongoDB password
   - [ ] Review deployment manifests for both `pt` and `pt-internal`
   - [ ] Ensure `INTERNAL_API` is set correctly (false for `pt`, true for `pt-internal`)

4. **Infrastructure**
   - [ ] Ensure MongoDB service is accessible within the cluster
   - [ ] Verify blockchain RPC node connectivity (check hostAliases if needed)
   - [ ] Confirm network policies allow inter-service communication
   - [ ] Review Ingress rules for external access to public API
   - [ ] Configure resource limits (CPU, memory) appropriately

5. **Deployment Execution**
   - [ ] Apply ConfigMap: `kubectl apply -f <configmap-file>`
   - [ ] Apply or verify Secrets exist in namespace
   - [ ] Apply deployments: `kubectl apply -f pt-service.yaml`
   - [ ] Apply ingress rules: `kubectl apply -f pt-ingress.yaml`
   - [ ] Verify deployments are running: `kubectl get deployments -n fame`
   - [ ] Check pod status: `kubectl get pods -n fame | grep pt`

6. **Verification**
   - [ ] Pods start without errors: `kubectl logs -n fame deployment/pt`
   - [ ] Public API accessible via Ingress: `https://<ingress-host>/pt/swagger`
   - [ ] Internal API accessible from within cluster: `http://pt-internal:7010`
   - [ ] Test authentication endpoints
   - [ ] Confirm blockchain connectivity (check pod logs)
   - [ ] Validate database connection (check pod logs)
   - [ ] Monitor resource usage: `kubectl top pods -n fame | grep pt`
   - [ ] Verify no crashloop or restart issues: `kubectl get pods -n fame -w`

## API Documentation

Once the service is running, comprehensive API documentation is available via Swagger UI:

- **Public API**: `http://<hostname>:7005/swagger`
- **Internal API**: `http://<hostname>:7010/swagger`

The Swagger interface provides:
- Complete endpoint documentation with request/response schemas
- Interactive API testing capabilities
- Authentication flow examples
- DTO validation rules

Static Swagger specifications are also generated as JSON files:
- [swagger-specification.public.json](swagger-specification.public.json)
- [swagger-specification.private.json](swagger-specification.private.json)

## Data Migration

The service includes a data migration utility for blockchain ledger synchronization and bulk extraction of offerings. This utility is primarily used for:
- Extracting offering data from blockchain
- Migrating data between environments
- Bulk operations on the internal network

For detailed instructions on using the migration utility, see [MIGRATION-README.md](MIGRATION-README.md).

To run migrations:
```bash
npm run migrate
```

## Troubleshooting

### Common Issues

**Database Connection Failed**
- Verify `DATABASE_URI` is correctly formatted
- Ensure MongoDB is running and accessible
- Check authentication credentials and `DATABASE_AUTH_SOURCE`

**Blockchain Connection Failed**
- Confirm `CHAIN_URL` points to a running RPC node
- Check WebSocket connectivity (use `ws://` or `wss://` scheme)
- Verify firewall rules allow outbound connections

**Smart Contract Interaction Errors**
- Validate contract addresses in `.env` match deployed contracts
- Ensure ABI files exist in `src/integration/` and are copied to `dist/`
- Confirm `PRIVATE_KEY_PT` and `PRIVATE_KEY_PERMISSION` are properly set
- Check that the blockchain account has sufficient balance for transactions

**Authentication Failures**
- Verify `AUTH_URL` is correct and the auth service is accessible
- If testing locally, consider setting `DISABLE_AUTH=true`

**Port Already in Use**
- Check if another service is running on port 7005 or 7010
- Stop conflicting services or change the port in [src/main.ts](src/main.ts:48)

## License

This project is licensed under the terms specified in [LICENSE](LICENSE).

## Project Status

**Active Development** - This service is actively maintained as part of the FAME Marketplace platform.

For issues, feature requests, or contributions, please contact the FAME development team.
