#!/usr/bin/env node

/**
 * Simple script to check if an asset exists in the Tracing contract
 * Usage: node check-asset.js <asset-id>
 *
 * This helps debug migration issues by checking if a transaction
 * actually succeeded on-chain even if the event wasn't received.
 */

const { Web3 } = require('web3');
const fs = require('fs');
const path = require('path');

// Get asset ID from command line
const assetId = process.argv[2];
if (!assetId) {
    console.error('Usage: node check-asset.js <asset-id>');
    console.error('Example: node check-asset.js 3Hcw8prSEE6D5rRvH');
    process.exit(1);
}

// Configuration from environment or defaults
const CHAIN_URL = process.env.CHAIN_URL || 'ws://162.55.94.15:8546/websocket';
const CONTRACT_ADDRESS = process.env.CONTRACT_TRACING_ADDR || '0x55F96411Ff9e424e172aB2Aa79fa5f19B09632Ce';
const ABI_PATH = process.env.CONTRACT_TRACING_ABI || 'integration/Tracing.abi';

async function checkAsset() {
    console.log(`Checking asset: ${assetId}`);
    console.log(`Chain URL: ${CHAIN_URL}`);
    console.log(`Contract: ${CONTRACT_ADDRESS}`);
    console.log('');

    let web3;
    try {
        // Connect to blockchain
        console.log('Connecting to blockchain...');
        web3 = new Web3(CHAIN_URL);

        // Load contract ABI
        const abiAbsPath = path.join(__dirname, 'src', ABI_PATH);
        const abi = JSON.parse(fs.readFileSync(abiAbsPath, 'utf8'));

        // Create contract instance
        const contract = new web3.eth.Contract(abi, CONTRACT_ADDRESS);

        // Call getAsset
        console.log('Querying blockchain...');
        const asset = await contract.methods.getAsset(assetId).call();

        console.log('Result:');
        console.log('-------');
        console.log(`  AID: ${asset.aid}`);
        console.log(`  Hash: ${asset.hash}`);
        console.log(`  TID: ${asset.tid}`);
        console.log(`  Registered: ${asset.registered} (${new Date(Number(asset.registered) * 1000).toISOString()})`);
        console.log(`  Predecessor: ${asset.predecessor || '(none)'}`);
        console.log(`  Successor: ${asset.successor || '(none)'}`);
        console.log('');

        if (asset.registered === '0') {
            console.log('❌ Asset NOT FOUND in contract (registered = 0)');
            console.log('This means the transaction did not execute successfully.');
        } else {
            console.log('✅ Asset EXISTS in contract');
            console.log('The transaction was successful on-chain.');
        }

        process.exit(0);
    } catch (error) {
        console.error('Error:', error.message);
        if (error.stack) {
            console.error('Stack:', error.stack);
        }
        process.exit(1);
    } finally {
        // Cleanup connection
        if (web3 && web3.provider) {
            try {
                await web3.provider.disconnect();
            } catch (e) {
                // Ignore cleanup errors
            }
        }
    }
}

checkAsset();
