import { Body, Controller, Delete, Get, HttpCode, Logger, Param, Post, Put, Query, UseGuards } from '@nestjs/common';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiTags } from '@nestjs/swagger';
import { AssetService } from './asset-service';
import { AssetDescriptor } from '../dtos/asset-descriptor';
import { AssetHashingTrxInput } from '../dtos/asset-hashing-trxin';
import { AuthGuard } from "../auth/auth-guard";
import { AssetMetadata } from 'src/dtos/asset-metadata';
import { AssetReference } from 'src/dtos/asset-reference';
import { RolesGuard } from 'src/auth/roles-guard';
import { Roles } from 'src/auth/roles-decorator';
import { User } from 'src/auth/user-decorator';
import { UserInfo } from 'src/auth/user-info';
import { ROLE_ADMIN, ROLE_OPERATOR, ROLE_SUPERUSER, ROLE_USER } from 'src/auth/auth-constants';
import { translateToHttpException } from 'src/utils/generic-utils';
import { HTTP_ERR_400, HTTP_ERR_401, HTTP_ERR_403, HTTP_ERR_404, HTTP_ERR_500, HTTP_ERR_502 } from 'src/constants';
import { ApmService } from 'src/integration/apm-service';

@ApiTags('Open API: Trusted Assets')
@Controller('/api/v1.0/assets')
@ApiBearerAuth()
@UseGuards(AuthGuard, RolesGuard)
export class AssetController {

    constructor(
        private readonly assetsService: AssetService,
        private readonly apmService: ApmService
    ) { }

    @Post()
    @HttpCode(202)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Publish asset',
        description: `Initiates the process for publishing a new trusted asset on the marketplace, by creating a matching
 catalogue entry in the FDAC module and a default policy in the APM module (first phase). The caller must be the owner of
 an enrolled trading account. This operation returns the input data for a blockchain transaction that targets the Tracing
 Ledger contract: to complete the publishing process, adding the asset’s record to the Tracing Ledger and flagging the
 catalogue entry as “trusted” (second phase), the caller shall sign and submit the received transaction using an enrolled
 trading account, which will be permanently linked to the asset as its “owner”.`
    })
    @ApiBody({
        type: AssetDescriptor,
        description: `Asset properties according to DCAT semantics (https://www.w3.org/TR/vocab-dcat/), with the addition
 of FAME extensions. The "dcterms_title", "dcterms_description_short", "dcterms_type", "dcat_endpointURL" fields are required
 and must contain a valid value. The "dcterms_identifier" and "dcterms_creator" fields, if present, are ignored, as their
 value is assigned by the system.`,
        required: true,
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The response body contains the required information for a blockchain transaction that shall be
 be submitted by the user to complete the publishing of the asset. The "aid" field is the identifier assigned to the asset
 by the system, which is also used as a reference for tracking the progress of the publishing process in the user request
 queue.`,
        type: AssetHashingTrxInput
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async publishAsset(@User() user: UserInfo, @Body() asset: AssetDescriptor): Promise<AssetHashingTrxInput> {
        Logger.log('Publishing requested for asset: ' + JSON.stringify(asset));
        Logger.log('Calling user: ' + JSON.stringify(user));
        try {
            return await this.assetsService.publishAsset(user, asset);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put('/:aid')
    @HttpCode(202)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Republish asset',
        description: `Initiates the process for republishing the existing marketplace entry of an asset, by
 updating the entry with new metadata (first phase). The caller must be affiliated with the same organization that
 made the original publishing. This operation returns the input data for a blockchain transaction that targets the
 Tracing Ledger contract: to complete the re-publishing process, creating a new asset’s record on the Tracing Ledger and flagging
 the re-published catalogue entry as “trusted” (second phase), the caller must sign and submit the received transaction.`
    })
    @ApiBody({
        type: AssetDescriptor,
        description: `Asset properties according to DCAT semantics (https://www.w3.org/TR/vocab-dcat/), with the addition
 of FAME extensions. The "dcterms_title", "dcterms_description_short", "dcterms_type", "dcat_endpointURL" fields are required
 and must contain a valid value. However, the values in "dcterms_title" and "dcterms_type" are expected to be identical to the
 ones in the existing entry, as these cannot be changed after publishing. The "dcterms_identifier" and "dcterms_creator" fields,
 if present, are ignored, as their value is assigned by the system.`,
        required: true,
    })
    @ApiParam({
        name: 'aid',
        required: true,
        type: String,
        description: 'AID of the asset to be re-published'
    })
    @ApiResponse({
        status: 202, // Accepted
        description: `The response body contains the required information for a blockchain transaction that shall be
 be submitted by the user to complete the revision of the asset. The "aid" field is the asset identifier, which is also
 used as a reference for tracking the progress of the revision process in the user request queue.`,
        type: AssetHashingTrxInput,
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async republishAsset(@User() user: UserInfo, @Param('aid') aid: string, @Body() asset: AssetDescriptor): Promise<AssetHashingTrxInput> {
        Logger.log('Republishing requested for asset with AID ' + aid + ', new data: ' + JSON.stringify(asset));
        Logger.log('Calling user: ' + JSON.stringify(user));
        try {
            return await this.assetsService.republishAsset(user, aid, asset);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Delete('/:aid')
    @HttpCode(204)
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Unpublish asset',
        description: `Removes a trusted asset from the marketplace, by marking both the asset’s catalogue entry and any associated
  offerings as “archived”. The caller must be the owner of the trading account used for the publishing of the target asset.
  Note that this operation, although it does not physically delete any records, cannot be reverted.`,
    })
    @ApiParam({
        name: 'aid',
        required: true,
        type: String,
        description: 'AID of the target trusted asset'
    })
    @ApiResponse({
        status: 204, // No Content
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 403, description: HTTP_ERR_403 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async unpublishAsset(@User() user: UserInfo, @Param('aid') aid: string): Promise<void> {
        Logger.log('Unpublishing requested for asset with AID ' + aid);
        Logger.log('Calling user: ' + JSON.stringify(user));
        try {
            await this.assetsService.unpublishAsset(user, aid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get()
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'List assets for publisher',
        description: `Returns a list of references pointing to all the trusted assets published by a given trading account.
 Note that unpublished assets are not included in the result. Note also that, unless the caller has the administrator role,
 this operation will always return an empty list for trading accounts that are not owned by the caller.`,
    })
    @ApiQuery({
        name: 'tid',
        required: true,
        type: String,
        description: `TID of publisher's trading account`
    })
    @ApiQuery({
        name: 'l',
        required: false,
        type: Number,
        description: 'Maximum number of results to return'
    })
    @ApiQuery({
        name: 'o',
        required: false,
        type: Number,
        description: 'Offset of the first result to return'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'Array of references to the trused assets published by the given trading account',
        type: AssetReference,
        isArray: true
    })
    @ApiResponse({ status: 400, description: HTTP_ERR_400 })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async listAssetsForPublisher(@User() user: UserInfo, @Query('tid') tid: string, @Query('l') length?: number, @Query('o') offset?: number): Promise<AssetReference[]> {
        return []; // TODO implement
    }

    @Get('/:aid')
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Retrieve asset',
        description: `Retrieves the marketplace catalogue entry of a given asset, in DCAT format.`,
    })
    @ApiParam({
        name: 'aid',
        required: true,
        type: String,
        description: 'AID of the target asset'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The catalogue entry of the requested asset, in DCAT format',
        type: AssetDescriptor
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async retrieveAssetData(@User() user: UserInfo, @Param('aid') aid: string): Promise<AssetDescriptor> {
        try {
            return await this.assetsService.retrieveAssetData(user, aid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Get('/:aid/tracing')
    @Roles(ROLE_USER, ROLE_SUPERUSER, ROLE_OPERATOR, ROLE_ADMIN)
    @ApiOperation({
        summary: 'Retrieve asset with tracing info',
        description: `Retrieves the marketplace catalogue entry of a given trusted asset in CANONICAL format,
  together with metadata from the Tracing Ledger and a “verified” flag that is only set if the catalogue entry has
  passed the standard integrity check. Note that the asset publisher’s trading account is included in the result
  only if the caller has the administrator role.`,
    })
    @ApiParam({
        name: 'aid',
        required: true,
        type: String,
        description: 'AID of the target trused asset'
    })
    @ApiResponse({
        status: 200, // OK
        description: 'The catalogue entry and ledger metadata of the requested trused asset',
        type: AssetMetadata
    })
    @ApiResponse({ status: 401, description: HTTP_ERR_401 })
    @ApiResponse({ status: 404, description: HTTP_ERR_404 })
    @ApiResponse({ status: 500, description: HTTP_ERR_500 })
    @ApiResponse({ status: 502, description: HTTP_ERR_502 })
    async retrieveTracingData(@User() user: UserInfo, @Param('aid') aid: string): Promise<AssetMetadata> {
        try {
            return await this.assetsService.retrieveTracingData(user, aid);
        } catch (err) {
            translateToHttpException(err);
        }
    }

    @Put('/:aid/check-visible')
    @Roles(ROLE_ADMIN)
    async checkVisible(@Param('aid') aid: string, @Body() user: UserInfo): Promise<boolean> {
        Logger.log('Checking visibility on asset ' + aid + ' for user context: ' + JSON.stringify(user));
        return await this.apmService.checkVisibility(aid, user);
    }

    @Put('/:aid/archive')
    @Roles(ROLE_ADMIN)
    async setArchived(@Param('aid') aid: string, @Body() user: UserInfo): Promise<void> {
        Logger.log('Setting archived status on asset ' + aid + ' for user context: ' + JSON.stringify(user));
        await this.apmService.setArchived(aid, user);
    }

    @Put('/:aid/check-status')
    @Roles(ROLE_ADMIN)
    async checkEditable(@Param('aid') aid: string, @Body() user: UserInfo): Promise<boolean> {
        Logger.log('Checking editable status on asset ' + aid + ' for user context: ' + JSON.stringify(user));
        return await this.apmService.checkEditable(aid, user);
    }
  
    @Put('/:aid/retrieve-status')
    @Roles(ROLE_ADMIN)
    async getStatus(@Param('aid') aid: string, @Body() user: UserInfo): Promise<string> {
        Logger.log('Retrieveng status info on asset ' + aid + ' for user context: ' + JSON.stringify(user));
        return await this.apmService.getStatus(aid, user);
    }
}

