import { Injectable, OnModuleInit, OnModuleDestroy, Logger } from '@nestjs/common';

export interface OfferingCloneItem {
    oid: string;
    startedAt: number;
}

export interface AssetRepublishJob {
    jobId: string; // This will be the new asset AID
    predecessorAid: string;
    newAssetAid: string;
    pendingOfferings: string[]; // OIDs of offerings to clone
    currentItem: OfferingCloneItem | null;
    completedOfferings: string[];
    failedOfferings: { oid: string; error: string; timestamp: number }[];
    createdAt: number;
    status: 'running' | 'completed' | 'failed';
}

@Injectable()
export class AssetRepublishJobState implements OnModuleInit, OnModuleDestroy {

    private readonly MAX_AGE: number = 300000; // 5 minutes max age for completed jobs
    private readonly TIMEOUT_MS: number = 15000; // 15 seconds timeout per offering
    private readonly MAX_RETRIES: number = 0; // No retries

    private cleanupIntervalId: NodeJS.Timeout;
    private timeoutCheckIntervalId: NodeJS.Timeout;

    private jobs = new Map<string, AssetRepublishJob>();

    // Callback for when a job completes (with or without failures)
    private onJobCompleted?: (job: AssetRepublishJob) => void;

    onModuleInit(): void {
        this.cleanupIntervalId = setInterval(() => this.cleanup(), this.MAX_AGE);
        this.timeoutCheckIntervalId = setInterval(() => this.checkTimeouts(), 1000); // Check every second
        Logger.debug('AssetRepublishJobState intervals started');
    }

    onModuleDestroy(): void {
        if (this.cleanupIntervalId) {
            clearInterval(this.cleanupIntervalId);
        }
        if (this.timeoutCheckIntervalId) {
            clearInterval(this.timeoutCheckIntervalId);
        }
        Logger.debug('AssetRepublishJobState intervals cleared');
    }

    createJob(newAssetAid: string, predecessorAid: string, offeringOids: string[]): void {
        if (this.jobs.has(newAssetAid)) {
            throw new Error(`Asset republish job with ID "${newAssetAid}" already exists.`);
        }

        const job: AssetRepublishJob = {
            jobId: newAssetAid,
            predecessorAid,
            newAssetAid,
            pendingOfferings: [...offeringOids],
            currentItem: null,
            completedOfferings: [],
            failedOfferings: [],
            createdAt: Date.now(),
            status: 'running'
        };

        this.jobs.set(newAssetAid, job);
        Logger.debug(`Created asset republish job ${newAssetAid} for predecessor ${predecessorAid} with ${offeringOids.length} offerings to clone`);
    }

    getNextOffering(jobId: string): string | null {
        const job = this.jobs.get(jobId);
        if (!job || job.status !== 'running' || job.pendingOfferings.length === 0) {
            return null;
        }

        const nextOid = job.pendingOfferings.shift();
        if (nextOid) {
            job.currentItem = {
                oid: nextOid,
                startedAt: Date.now()
            };
            Logger.debug(`Job ${jobId}: Processing offering ${nextOid}`);
        }

        return nextOid;
    }

    markOfferingCompleted(jobId: string, oid: string): void {
        const job = this.jobs.get(jobId);
        if (!job) {
            Logger.warn(`Job ${jobId} not found when marking offering ${oid} as completed`);
            return;
        }

        if (job.currentItem?.oid === oid) {
            job.completedOfferings.push(oid);
            job.currentItem = null;
            Logger.debug(`Job ${jobId}: Offering ${oid} cloned successfully`);

            // Check if job is finished
            if (job.pendingOfferings.length === 0) {
                job.status = 'completed';
                Logger.log(`Asset republish job ${jobId} completed. Cloned ${job.completedOfferings.length} offerings, ${job.failedOfferings.length} failed`);

                // Trigger completion callback
                if (this.onJobCompleted) {
                    this.onJobCompleted(job);
                }
            }
        }
    }

    markOfferingFailed(jobId: string, oid: string, error: string): void {
        const job = this.jobs.get(jobId);
        if (!job) {
            Logger.warn(`Job ${jobId} not found when marking offering ${oid} as failed`);
            return;
        }

        if (job.currentItem?.oid === oid) {
            job.failedOfferings.push({
                oid,
                error,
                timestamp: Date.now()
            });
            job.currentItem = null;
            Logger.error(`Job ${jobId}: Offering ${oid} cloning failed - ${error}`);

            // Check if job is finished
            if (job.pendingOfferings.length === 0) {
                job.status = 'completed';
                Logger.log(`Asset republish job ${jobId} completed. Cloned ${job.completedOfferings.length} offerings, ${job.failedOfferings.length} failed`);

                // Trigger completion callback
                if (this.onJobCompleted) {
                    this.onJobCompleted(job);
                }
            }
        }
    }

    getCurrentItem(jobId: string): OfferingCloneItem | null {
        const job = this.jobs.get(jobId);
        return job?.currentItem || null;
    }

    getJob(jobId: string): AssetRepublishJob | null {
        return this.jobs.get(jobId) || null;
    }

    getAllJobs(): Map<string, AssetRepublishJob> {
        return this.jobs;
    }

    setJobCompletionCallback(callback: (job: AssetRepublishJob) => void): void {
        this.onJobCompleted = callback;
    }

    private checkTimeouts(): void {
        const now = Date.now();

        for (const [jobId, job] of this.jobs) {
            if (job.currentItem && (now - job.currentItem.startedAt) > this.TIMEOUT_MS) {
                Logger.warn(`Job ${jobId}: Offering ${job.currentItem.oid} timed out after ${this.TIMEOUT_MS}ms`);
                this.markOfferingFailed(jobId, job.currentItem.oid, 'Timeout during offering cloning process');
            }
        }
    }

    private cleanup(): void {
        const now = Date.now();

        for (const [jobId, job] of this.jobs) {
            if (job.status === 'completed' && (now - job.createdAt) > this.MAX_AGE) {
                this.jobs.delete(jobId);
                Logger.debug(`Expired asset republish job ${jobId} removed`);
            }
        }
    }
}
